package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.apimgmt.dao.ApiMgmtSyncMdcFailDataDao;
import com.digiwin.athena.apimgmt.infra.context.IApiMgmtMdcSyncContext;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtProp;
import com.digiwin.athena.apimgmt.model.SyncMdcFailData;
import com.digiwin.athena.apimgmt.services.ApiMgmtSyncMdcServiceProcessor;
import jakarta.annotation.PostConstruct;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 补偿同步mdc排程
 * 定时任务
 */
@Service
@Slf4j
public class ApiMgmtSyncMdcFailDataService {
	@Autowired
	private ApiMgmtSyncMdcFailDataDao syncMdcFailDataDao;

	@Autowired
	private ApiMgmtSyncMdcServiceProcessor syncMdcServiceProcessor;

    @Autowired
    private ApiMgmtProp apiMgmtProp;

    @Autowired
    private IApiMgmtMdcSyncContext mdcSyncContext;


    public ApiMgmtSyncMdcFailDataService() {
		super();
	}

	/**
	 * 采用ScheduledExecutorService做一个建议的定时任务组件
	 * 每隔1分钟发起一次。
	 */
	@PostConstruct
	public void initScheduledTask(){
        if (!apiMgmtProp.getSync().getMdc().isEnable()) {
            log.info("MDC同步定时任务已关闭");
            return;
        }

        long interval = apiMgmtProp.getSync().getMdc().getInterval().getSeconds();
        log.info("MDC同步定时任务已开启，间隔时间: {} 秒", interval);

        ScheduledExecutorService executorService = new ScheduledThreadPoolExecutor(5);
		executorService.scheduleWithFixedDelay(() -> {
            long startTime = System.currentTimeMillis();
            long nextSyncTime = mdcSyncContext.getNextSyncTime();
            if (startTime < nextSyncTime) {
                return;
            }

            try {
				log.info("定时任务开启: " + Calendar.getInstance().getTime());
				List<SyncMdcFailData> list = syncMdcFailDataDao.getData();
				log.info("查询出的数据量: " + list.size());
                if (CollUtil.isNotEmpty(list)) {
					Map<String, List<SyncMdcFailData>> map = list.stream().collect(Collectors.groupingBy(SyncMdcFailData::getSyncFactory));
					for (String syncFactory : map.keySet()) {
						syncMdcServiceProcessor.reqSyncMdcMetadataAddCompensate(map.get(syncFactory), syncFactory);
					}
				}
				log.info("补偿任务执行完毕耗时：" + (System.currentTimeMillis() - startTime));
			} catch (Exception e) {
				log.error(e.getMessage());
			}
        }, 1, interval, TimeUnit.SECONDS);
	}
}
