package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.date.DateUtil;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtVerifyLogDao;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.model.VerifyLog;
import com.digiwin.athena.apimgmt.service.util.ApiVersionServiceUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.ParseException;
import java.util.*;

/**
 * 查詢API審核歷程
 * /restful/standard/apimgmt/ApiApprovedProcess/Get
 */
@Slf4j
@Service
public class ApiApprovedProcessGetService extends AbstractApiService {

	@Autowired
    ApiMgmtVerifyLogDao verifyLogDao;

	@Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

	public ApiApprovedProcessGetService() {
		super();
		jsonSchemaFileName = ApimgmtSchemaConstant.API_APPROVED_PROCESS_GET_SCHEMA;
	}

	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, ParseException {
		// 取得語系
		String tLocale = ApiMgmtServiceContextHolder.getLocale();
		// header沒傳語系的話，默認回傳英文
		tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
		// 取得request node
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		// 建立response node
		ObjectNode tResponseNode = mapper.createObjectNode();
		ArrayNode tApiProcessAryNode = mapper.createArrayNode();
		tResponseNode.set(ApiAttributeEnum.apiProcess.toString(), tApiProcessAryNode);
		// 取得入參api名稱
		String tApiName = tRequestJsonNode.get(ApiAttributeEnum.apiName.toString()).asText();
		// 取得入參api版號
		String tVersion = tRequestJsonNode.get(ApiAttributeEnum.version.toString()).asText();
		// branch
		String branch = ApiVersionServiceUtil.getBranchFromVersion(tVersion);
		String tTenant = "";
		// 取得入參租戶id
		if (tRequestJsonNode.get(ApiAttributeEnum.tenantId.toString()) != null) {
			tTenant = tRequestJsonNode.get(ApiAttributeEnum.tenantId.toString()).asText();
		}
		// 找出api的所有版本號
		List<StandardApiVersion> tStandardApiVersionList = standardApiVersionDao.fetchApiVersionsByName(tApiName, branch, tTenant);
		// 找出入參對應版本號的相關資訊
		StandardApiVersion tStandardApiVersion = null;
		for (StandardApiVersion standardApiVersion : tStandardApiVersionList) {
			// 有找到對應版本號數據
			if (standardApiVersion.getVersion().equals(tVersion)) {
				tStandardApiVersion = standardApiVersion;
			}
		}
		// 有查到對應版本的API資料
		if (tStandardApiVersion != null) {
			List<VerifyLog> verifyLogs = verifyLogDao.getVerifyLogsByApiVerId(tStandardApiVersion.getId());
			if (verifyLogs.size() != 0) {
				for (VerifyLog verifyLog : verifyLogs) {
					String tOperation;
					ObjectNode tLogNode = mapper.createObjectNode();
					tLogNode.put(ApiAttributeEnum.apiName.toString(), tApiName);
					tLogNode.put(ApiAttributeEnum.auditor.toString(), verifyLog.getVerifyAcct());
					switch (tLocale) {
					case ApimgmtConstant.ZH_CN:
						tOperation = verifyLog.getVerityOperate().getNameZhCn();
						break;
					case ApimgmtConstant.ZH_TW:
						tOperation = verifyLog.getVerityOperate().getNameZhTw();
						break;
					case ApimgmtConstant.EN_US:
					default:
						tOperation = verifyLog.getVerityOperate().getNameEnUs();
						break;
					}
					tLogNode.put(ApiAttributeEnum.operation.toString(), tOperation);
					tLogNode.put(ApiAttributeEnum.operationDesc.toString(), verifyLog.getReason());
					tLogNode.put(ApiAttributeEnum.auditor.toString(), verifyLog.getVerifyAcct());
					tLogNode.put(ApiAttributeEnum.version.toString(), tVersion);
					tLogNode.put(ApiAttributeEnum.approvedTime.toString(), DateUtil.formatLocalDateTime(verifyLog.getVerifyTime()));
					tApiProcessAryNode.add(tLogNode);
				}
				tApiProcessAryNode = sortJsonArrayByAttribute(tApiProcessAryNode, ApiAttributeEnum.approvedTime.toString());
				tResponseNode.set(ApiAttributeEnum.apiProcess.toString(), tApiProcessAryNode);
			}
		}
		// 根據api名稱以及版號取得其審核歷程
		StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
		String tDescription = tStateCode.getDescription();
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	/**
	 * 排序
	 * @param pArrayNode pArrayNode
	 * @param pAttribute pAttribute
	 * @return ArrayNode
	 */
	public ArrayNode sortJsonArrayByAttribute(ArrayNode pArrayNode, String pAttribute) {
		// 排序後的json array
		ArrayNode sortedJsonArray = mapper.createArrayNode();
		List<ObjectNode> list = new ArrayList<>();
		// 把ObjectNode放進list
		for (int i = 0; i < pArrayNode.size(); i++) {
			list.add((ObjectNode) pArrayNode.get(i));
		}
		list.sort(new Comparator() {
			@Override
			public int compare(Object o1, Object o2) {
				String approvedTime1;
				String approvedTime2;
				Date date1 = null;
				Date date2 = null;
				try {
					if (StringUtil.isEmptyOrSpace(((ObjectNode) o1).get(pAttribute).asText())) {
						approvedTime1 = ApimgmtConstant.EARLIEST_TIME;
					} else {
						approvedTime1 = ((ObjectNode) o1).get(pAttribute).asText();
					}
					if (StringUtil.isEmptyOrSpace(((ObjectNode) o2).get(pAttribute).asText())) {
						approvedTime2 = ApimgmtConstant.EARLIEST_TIME;
					} else {
						approvedTime2 = ((ObjectNode) o2).get(pAttribute).asText();
					}
					date1 = DateUtil.parse(approvedTime1);
					date2 = DateUtil.parse(approvedTime2);
				} catch (Exception e) {
					log.error(e.getMessage(), e);
				}
				return Objects.requireNonNull(date1).compareTo(date2);
			}
		});
		for (int i = 0; i < pArrayNode.size(); i++) {
			sortedJsonArray.add(list.get(i));
		}
		return sortedJsonArray;
	}
}
