package com.digiwin.athena.apimgmt.model;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.annotation.IdType;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.annotation.TableName;
import com.digiwin.athena.apimgmt.constants.ApiTypeConstants;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import jakarta.annotation.Nonnull;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;

import java.beans.Transient;
import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * The persistent class for the standard_api database table.
 */
@Data
@ToString(onlyExplicitlyIncluded = true)
@EqualsAndHashCode(exclude = {"standardApiVersions", "standardApiGroup",
        "standardApiSyncType", "standardApiPaging", "standardApiCategory",
        "projectVersionRelations", "projectName", "projectVersionNameList"})
@TableName(value = "standard_api", resultMap = "StandardApiResultMap")
public class StandardApi implements Serializable {
    private static final long serialVersionUID = 1L;

    @TableId(type = IdType.AUTO)
    private Long id;

    @TableField("build_acct")
    private String buildAcct;

    @TableField("build_time")
    private LocalDateTime buildTime;

    @TableField("description_en_us")
    private String descriptionEnUs;

    @TableField("description_zh_cn")
    private String descriptionZhCn;

    @TableField("description_zh_tw")
    private String descriptionZhTw;

    private String name;

    private String provider;

    @TableField("remark_en_us")
    private String remarkEnUs;

    @TableField("remark_zh_cn")
    private String remarkZhCn;

    @TableField("remark_zh_tw")
    private String remarkZhTw;

    private String requester;

    @TableField("tag_en_us")
    private String tagEnUs;

    @TableField("tag_zh_cn")
    private String tagZhCn;

    @TableField("tag_zh_tw")
    private String tagZhTw;

    @TableField("is_batch")
    private Boolean isBatch;

    // [S00-20200422001]增加欄位用以標記某些舊申請的API，XML範例中M節點的parameter key不要加_data
    @TableField("msg_m_data")
    private Boolean msgMdata;

    @TableField("msg_format")
    private String msgFormat;

    @TableField("tenant_id")
    private String tenantId;

    @TableField("package_name")
    private String packageName;

    @TableField("`group`")
    private Integer group;

    // bi-directional many-to-one association to ApprovedStatus
    @TableField(exist = false)
    private StandardApiGroup standardApiGroup;

    private Integer syncType;

    // bi-directional many-to-one association to ApprovedStatus
    @TableField(exist = false)
    private StandardApiSyncType standardApiSyncType;

    private Integer paging;

    // bi-directional many-to-one association to ApprovedStatus
    @TableField(exist = false)
    private StandardApiPaging standardApiPaging;

    private Long category;

    // bi-directional many-to-one association to StandardApiCategory
    @TableField(exist = false)
    private StandardApiCategory standardApiCategory;

    @TableField(exist = false)
    private List<StandardApiVersion> standardApiVersions = new ArrayList<>();

    @TableField(exist = false)
    private Long projectId;

    @TableField(exist = false)
    private List<Long> projectVersion;

    /**
     * 设计租户ID
     * by songwq 20230625 新增设计租户ID用于对进版功能进行限制
     */
    @TableField("design_tenant_id")
    private String designTenantId;

    /**
     * by songwq 20230705 新增teamId 1：鼎捷租户类型 2 外部租户类型 3 其他
     */
    @TableField("team_id")
    private String teamId;

    /**
     * by songwq 20230718 新增teamType 1：鼎捷租户类型 2 外部租户类型
     */
    @TableField("team_type")
    private String teamType;

    /**
     * 父类API名称
     */
    @TableField("parent_api_name")
    private String parentApiName;

    /**
     * 父类API名称
     */
    @TableField("parent_branch")
    private String parentBranch;

    /**
     * 父类API租户Id
     */
    @TableField("parent_api_tenant_id")
    private String parentApiTenantId;

    /**
     * 扩展字段
     */
    @TableField("api_extend")
    private String apiExtend;

    /**
     * 扩展字段
     */
    @TableField("first_applicant")
    private String firstApplicant;

    /**
     * API类型
     */
    @TableField("api_type")
    private String apiType;

    /**
     * 协议选择
     */
    @TableField("protocol")
    private String protocol;

    /**
     * 请求方法
     */
    @TableField("request_method")
    private String requestMethod;

    /**
     * 分支
     */
    @TableField("branch")
    private String branch;

    /**
     * 分支
     */
    @TableField("api_path")
    private String apiPath;

    @TableField(exist = false)
    private List<ProjectVersionRelation> projectVersionRelations = new ArrayList<>();

    @TableField(exist = false)
    private String projectName;

    @TableField(exist = false)
    private List<String> projectVersionNameList;
    @TableField(exist = false)
    private Boolean hasVersions = false;

    public StandardApi() {
        msgMdata = true;
        tenantId = "";
    }

    public static boolean validateName(@Nonnull String apiType, String apiName) {
        if (StrUtil.isBlank(apiName)) {
            return false;
        }

        if (ApiTypeConstants.ESP_CODE.equals(apiType)) {
            return ApimgmtConstant.PATTERN_API_NAME.matcher(apiName).matches();
        } else if (ApiTypeConstants.OPENAPI_CODE.equals(apiType)) {
            return ApimgmtConstant.PATTERN_OPEN_API_NAME.matcher(apiName).matches();
        } else {
            return false;
        }
    }

    public Boolean getBatch() {
        return isBatch;
    }

    public void setBatch(Boolean batch) {
        isBatch = batch;
    }

    public StandardApiVersion addStandardApiVersion(StandardApiVersion standardApiVersion) {
        getStandardApiVersions().add(standardApiVersion);
        standardApiVersion.setStandardApi(this);

        return standardApiVersion;
    }

    @Transient
    public String getApiTypeOrDefault() {
        String apiType = getApiType();
        if (StrUtil.isBlank(apiType)) {
            return ApiTypeConstants.DEFAULT_CODE;
        }

        return apiType;
    }

    public boolean validateName() {
        return validateName(getApiTypeOrDefault(), getName());
    }

    public boolean validateBranch() {
        return ApimgmtConstant.PATTERN_BRANCH.matcher(getBranch()).matches();
    }

    public boolean isOpenApi() {
        return ApiTypeConstants.OPENAPI_CODE.equals(getApiTypeOrDefault());
    }


    public void setId(Long id) {
        this.id = id;
        if (standardApiVersions != null) {
            standardApiVersions.forEach(v -> v.setStandardApi(this));
        }

        if (projectVersionRelations != null) {
            projectVersionRelations.forEach(v -> v.setStandardApi(this));
        }
    }

    public void setStandardApiCategory(StandardApiCategory standardApiCategory) {
        this.standardApiCategory = standardApiCategory;
        if (standardApiCategory != null) {
            this.category = standardApiCategory.getId();
        }
    }

    public void setStandardApiGroup(StandardApiGroup standardApiGroup) {
        this.standardApiGroup = standardApiGroup;
        if (standardApiGroup != null) {
            this.group = Optional.ofNullable(standardApiGroup.getId())
                    .map(Long::intValue).orElse(null);
        }
    }

    public void setStandardApiSyncType(StandardApiSyncType standardApiSyncType) {
        this.standardApiSyncType = standardApiSyncType;
        if (standardApiSyncType != null) {
            this.syncType = Optional.ofNullable(standardApiSyncType.getId())
                    .map(Long::intValue).orElse(null);
        }
    }

    public void setStandardApiPaging(StandardApiPaging standardApiPaging) {
        this.standardApiPaging = standardApiPaging;
        if (standardApiPaging != null) {
            this.paging = Optional.ofNullable(standardApiPaging.getId())
                    .map(Long::intValue).orElse(null);
        }
    }
}