package com.digiwin.athena.apimgmt.services;

import cn.hutool.http.HttpStatus;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtInterfaceConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtSyncMdcFailDataDao;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.http.HttpFailedException;
import com.digiwin.athena.apimgmt.infra.http.HttpRequest;
import com.digiwin.athena.apimgmt.infra.http.HttpResponse;
import com.digiwin.athena.apimgmt.infra.http.RequestOption;
import com.digiwin.athena.apimgmt.infra.util.HttpUtil;
import com.digiwin.athena.apimgmt.model.SyncMdcFailData;
import com.digiwin.athena.apimgmt.service.model.AddApiData;
import com.digiwin.athena.apimgmt.service.model.AddApiVer;
import com.digiwin.athena.apimgmt.service.model.MdcResponse;
import com.digiwin.athena.apimgmt.service.util.MailServiceUtil;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.Calendar;
import java.util.List;

@Slf4j
@Service
public class ApiMgmtSyncMdcService {


    @Autowired
    private ApiMgmtSyncMdcFailDataDao syncMdcFailDataDao;

    /**
     * 同步MDC处理
     *
     * @param data        版本Id
     * @param tAddApiData tAddApiData
     * @param pSyncUrl    pSyncUrl
     * @param tUserId     用户id
     */
    @Async
    public void reqSyncMdcMetadataAdd(List<Long> data, AddApiData tAddApiData, String pSyncUrl, String tUserId, String tToken, String routerKey) {
        // 判斷是哪一區，並取得對應的參數
        String tUrl = pSyncUrl + ApimgmtInterfaceConstant.MDC_METADATA_UPDATE;
        String tStrJsonRequestBody = "";
        int httpStatusCode = -1;
        String tResponseMessage;
        try {
            tStrJsonRequestBody = (new GsonBuilder().disableHtmlEscaping()).create().toJson(tAddApiData);
            HttpRequest tHttpPost = HttpRequest.ofPost(tUrl);
            tHttpPost.setEntity(tStrJsonRequestBody);
            tHttpPost.setHeader("userId", tUserId);
            tHttpPost.setHeader(ApimgmtConstant.TOKEN, tToken);
            if (null != routerKey) {
                tHttpPost.setHeader(ApimgmtConstant.ROUTER_KEY, routerKey);
            }
            MdcResponse tResponse = HttpUtil.execute(tHttpPost, MdcResponse.class);
            String tMdcStatCode = tResponse.getExecution().getCode();
            tResponseMessage = tResponse.getExecution().getDescription();
            if (!"000".equals(tMdcStatCode)) {
                // 送邮件通知
                String tMailContent = MessageFormat.format("<HTML><BODY>{0} {1}<BR><BR>{2}<BR><BR><BR>請求MDC url: {3}<BR>request message body: {4}<BR>HTTP狀態碼: {5}<BR>MDC回應的訊息: {6}</BODY></HTML>", ApimgmtConstant.SUBJECT_SYNC_ADD_FAIL,
                        (new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")).format(Calendar.getInstance().getTime()), genAddApiDataInfoStr(tAddApiData), tUrl, tStrJsonRequestBody, tMdcStatCode + "/" + httpStatusCode, tResponseMessage);
                MailServiceUtil.sendMailToAdmin(tMailContent, ApimgmtConstant.SUBJECT_SYNC_ADD_FAIL);
            }
        } catch (Exception e) {
            if (e instanceof HttpFailedException) {
                httpStatusCode = ((HttpFailedException) e).getStatusCode();
                tResponseMessage = ((HttpFailedException) e).getRespBody();
            } else {
                httpStatusCode = 500;
                tResponseMessage = e.getMessage();
            }

            // 入库
            for (Long apiVersionId : data) {
                SyncMdcFailData syncMdcFailData = new SyncMdcFailData();
                syncMdcFailData.setApiVerId(apiVersionId);
                syncMdcFailData.setSyncFactory(pSyncUrl);
                syncMdcFailData.setSuccessFlag(false);
                syncMdcFailData.setBuildTime(LocalDateTime.now());
                syncMdcFailData.setUpdateTime(LocalDateTime.now());
                syncMdcFailDataDao.save(syncMdcFailData);
            }
            String tMailContent = MessageFormat.format("<HTML><BODY>{0} {1}<BR><BR>{2}<BR><BR><BR>請求MDC url: {3}<BR>request message body: {4}<BR>HTTP狀態碼: {5}<BR>MDC回應的訊息: {6}</BODY></HTML>", ApimgmtConstant.SUBJECT_SYNC_ADD_FAIL,
                    (new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")).format(Calendar.getInstance().getTime()), genAddApiDataInfoStr(tAddApiData), tUrl, tStrJsonRequestBody, httpStatusCode, tResponseMessage);
            MailServiceUtil.sendMailToAdmin(tMailContent, ApimgmtConstant.SUBJECT_SYNC_ADD_FAIL);
        }
    }

    /**
     * 补偿同步MDC处理
     *
     * @param syncMdcFailDataList syncMdcFailDataList
     * @param tAddApiData         tAddApiData
     * @param pSyncUrl            pSyncUrl
     */
    public void reqSyncMdcMetadataAddCompensate(List<SyncMdcFailData> syncMdcFailDataList, AddApiData tAddApiData, String pSyncUrl) {
        String tUrl = pSyncUrl + ApimgmtInterfaceConstant.MDC_METADATA_UPDATE;
        try {
            String tStrJsonRequestBody = (new GsonBuilder().disableHtmlEscaping()).create().toJson(tAddApiData);
            HttpRequest tHttpPost = HttpRequest.ofPost(tUrl);
            tHttpPost.setEntity(tStrJsonRequestBody);
            tHttpPost.setHeader("userId", "compensation_task");
            // 定时任务，并没有routerKey，先留着吧
            Object routerKeyValue = ApiMgmtServiceContextHolder.getRouterKey();
            if (null != routerKeyValue) {
                tHttpPost.setHeader(ApimgmtConstant.ROUTER_KEY, routerKeyValue.toString());
            }
            HttpResponse tResponse = HttpUtil.execute(tHttpPost, HttpResponse.class, RequestOption.of().setRetryEnabled(false));
            int httpStatusCode = tResponse.getStatusCode();
            String tResponseMessage = tResponse.getBody();
            if (httpStatusCode == HttpStatus.HTTP_OK) {
                MdcResponse tMdcResponse = (new Gson()).fromJson(tResponseMessage, MdcResponse.class);
                String tMdcStatCode = tMdcResponse.getExecution().getCode();
                if ("000".equals(tMdcStatCode)) {
                    for (SyncMdcFailData syncMdcFailData : syncMdcFailDataList) {
                        syncMdcFailData.setSuccessFlag(true);
                        syncMdcFailData.setUpdateTime(LocalDateTime.now());
                        syncMdcFailDataDao.saveOrupdate(syncMdcFailData);
                    }
                }
            } else {
                log.error("补偿同步MDC处理响应码异常！status:{},msg:{}", httpStatusCode, tResponseMessage);
            }
        } catch (Exception e) {
            log.error("补偿同步MDC处理异常！", e);
        }
    }

    private String genAddApiDataInfoStr(AddApiData tAddApiData) {
        String tStr = "";
        try {
            for (AddApiVer tAddApiVer : tAddApiData.getApi_data()) {
                String tApiName = tAddApiVer.getApi_name();
                String tApiVer = tAddApiVer.getApi_version_info().get(0).getApi_version();
                String tTenantId = tAddApiVer.getTenant_id();
                tStr = MessageFormat.format("{0}{1}API名稱: {2}, API版本: {3}, 租戶ID: {4}", tStr, "<BR>", tApiName, tApiVer, tTenantId);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return tStr;
    }
}
