package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.annotate.OperateAuthorityVerify;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.*;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.ApiVersionEnum;
import com.digiwin.athena.apimgmt.enums.MessageFormatEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtMdcProp;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.service.util.*;
import com.digiwin.athena.apimgmt.services.ApiMgmtApiSaveAndReviewVerifyProcessor;
import com.digiwin.athena.apimgmt.services.ApiMgmtSyncMdcServiceProcessor;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.FileOutputStream;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * API送審
 * /restful/standard/apimgmt/ApiReview/Update
 */
@Slf4j
@Service
@OperateAuthorityVerify
public class ApiReviewUpdateService extends AbstractApiService {

	private static final Lock LOCK = new ReentrantLock(true);

	@Autowired
    ApiMgmtStandardApiDao standardApiDao;

	@Autowired
    ApiMgmtStandardApiVersionDao standardApiVersionDao;

	@Autowired
    ApiMgmtStandardApiDataNameDao standardApiDataNameDao;

	@Autowired
    ApiMgmtApprovedStatusDao approvedStatusDao;

	@Autowired
    ApiMgmtTenantConfigDao tenantConfigDao;

	@Autowired
    ApiMgmtStandardDataNameDao standardDataNameDao;

	@Autowired
    private ApiMgmtSyncMdcServiceProcessor syncMdcServiceProcessor;

	@Autowired
    private ApiMgmtApiSaveAndReviewVerifyProcessor apiSaveAndReviewVerifyProcessor;

    @Autowired
    private ApiMgmtMdcProp mdcProp;

	public ApiReviewUpdateService() {
		super();
		jsonSchemaFileName = ApimgmtSchemaConstant.API_REVIEW_SCHEMA;
	}

	/**
	 * API送审
	 * @param validatorResult validatorResult
	 * @return map
	 * @throws BaseException BaseException
	 * @throws Exception Exception
	 */
	@Override
	protected Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception {
		Map<String, Object> tResponse = new HashMap<>();
		StateCode tStateCode = getStateCode(ValidateStateEnum.SUCCESS.getCode());
		String tDescription = tStateCode.getDescription();
		ArrayList<Long> tListApiVerIdNeedToSyncMetadata = new ArrayList<>();
        String routerKey = ApiMgmtServiceContextHolder.getRouterKey();
        String tTenantId = ApiMgmtServiceContextHolder.getTenantId();
        String tUserId = ApiMgmtServiceContextHolder.getUserId();
		// 如果Tenant被設定為略過不檢查字段，則不進記憶體or資料庫查Data Name清單
		boolean tSkipReview = tenantConfigDao.getSkipReviewDataNameTenant(tTenantId);
		// 是否要略過審核
		boolean tTanantIsSkipReview = tenantConfigDao.getSkipReviewTenant(tTenantId);
		boolean tApiSpecErrorExist = false;
		// 全部API的錯誤總Map
		Map<String, List<Map<String, String>>> tTotalApiErrorMap = new HashMap<>();
		JsonNode tRequestJsonNode = validatorResult.getJsonContent();
		ArrayNode tApiArrayNode = (ArrayNode) tRequestJsonNode.get(ApimgmtConstant.API_LIST);
		for (JsonNode tApiNode : tApiArrayNode) {
			String tApiName = tApiNode.get(ApiAttributeEnum.apiName.toString()).asText();
			String tVersion = tApiNode.get(ApiAttributeEnum.version.toString()).asText();
			Map<String, String> tConditionMap = new HashMap<>();
			tConditionMap.put(ApiAttributeEnum.apiName.toString(), tApiName);
			tConditionMap.put(ApiAttributeEnum.version.toString(), tVersion);
			if (tApiNode.get(ApiAttributeEnum.tenantId.toString()) != null) {
				tConditionMap.put(ApiAttributeEnum.tenantId.toString(), tApiNode.get(ApiAttributeEnum.tenantId.toString()).asText());
			}
			StandardApiVersion tStandardApiVersion = standardApiVersionDao.fetchApiVersion(tConditionMap);
			StandardApi tStandardApi = tStandardApiVersion.getStandardApi();
			List<StandardApiDataName> tStandardApiDataNames = standardApiDataNameDao.getByApiVerId(tStandardApiVersion.getId(), true);
			Map<String, StandardApiDataName> tAddDataName = convertToMap(tStandardApiDataNames);
			// 取得前一版StandardApiVersion
			String tPreviousVersion = ApiVersionServiceUtil.getPreviousVersion(tVersion);
			Map<String, String> tVersionCondition = new HashMap<>();
			tVersionCondition.put(ApiVersionEnum.API_NAME.toString(), tApiName);
			tVersionCondition.put(ApiVersionEnum.API_VERSION.toString(), tPreviousVersion);
			if (tApiNode.get(ApiAttributeEnum.tenantId.toString()) != null) {
				tVersionCondition.put(ApiVersionEnum.TENANT_ID_FULL_MATCHING.toString(), tApiNode.get(ApiAttributeEnum.tenantId.toString()).asText());
			}
			List<StandardApiVersion> tPreviStandardApiVersionList = standardApiVersionDao.fetch(tVersionCondition);
			List<Map<String, String>> tApiDataNameCheckResult = new ArrayList<>();
			// 檢查列舉值格式
			apiSaveAndReviewVerifyProcessor.checkListEnum(tApiDataNameCheckResult, tAddDataName);
			// 若服務發起方或接收方包含ATHENA，則訊息格式必須為JSON
			if (tStandardApi.getRequester().equalsIgnoreCase("Athena") || tStandardApi.getProvider().equalsIgnoreCase("Athena")) {
				if (!tStandardApi.getMsgFormat().equals(MessageFormatEnum.JSON.getMsgFormat())) {
					Map<String, String> tReqErrorLog = new HashMap<>();
					tReqErrorLog.put(ApimgmtConstant.DESC, "發起方、接收方產品包含「Athena」，則信息格式必須為「JSON」");
					tApiDataNameCheckResult.add(tReqErrorLog);
				}
			}
			// 檢查匯入是否有包含必要字段
			apiSaveAndReviewVerifyProcessor.checkRequiredDataName(tApiDataNameCheckResult, tAddDataName);
			// 檢查「分頁模式」
			if (tStandardApi.getStandardApiPaging().getId() == 2L || tStandardApi.getStandardApiPaging().getId() == 3L) {
				apiSaveAndReviewVerifyProcessor.checkPagingRequiredDataName(tApiDataNameCheckResult, tAddDataName);
			}
			// 除M、D、SD、4D、5D...之外，「為數組」不可為Y
			apiSaveAndReviewVerifyProcessor.checkIsArrayCannotBeY(tApiDataNameCheckResult, tAddDataName);
			// 檢查datakey是否設為Y
			apiSaveAndReviewVerifyProcessor.checkDataKey(tApiDataNameCheckResult, tAddDataName);
			// 確認是否有缺少字段、「為數組」、「必要」欄位是否正確
			if (!tPreviStandardApiVersionList.isEmpty()) {
				List<StandardApiDataName> tPreviApiDataName = standardApiDataNameDao.getByApiVerId(tPreviStandardApiVersionList.get(0).getId(), true);
				apiSaveAndReviewVerifyProcessor.compareDataName(tApiDataNameCheckResult, tPreviApiDataName, tAddDataName);
			}
			// 若調用模式為全異步則訊息格式必須為JSON
			if (tStandardApi.getStandardApiSyncType().getId() == 2L) {
				if (!tStandardApi.getMsgFormat().equals(MessageFormatEnum.JSON.getMsgFormat())) {
					Map<String, String> tReqErrorLog = new HashMap<>();
					tReqErrorLog.put(ApimgmtConstant.DESC, "調用模式為「全異步」，則信息格式必須為「JSON」");
					tApiDataNameCheckResult.add(tReqErrorLog);
				}
			}
			if (!tApiDataNameCheckResult.isEmpty()) {
				String tApiError = tApiName + "/" + tVersion;
				tTotalApiErrorMap.put(tApiError, tApiDataNameCheckResult);
				tApiSpecErrorExist = true;
			} else {
				// 檢查是否有新詞彙，有則自動加入辭彙庫
				checkNewDataName(tStandardApiDataNames, tUserId, tSkipReview);
				ApprovedStatus tApprovedStatus;
				if (tTanantIsSkipReview) {
					// 開發中
					tApprovedStatus = approvedStatusDao.get(3L);
				} else {
					// 審核中
					tApprovedStatus = approvedStatusDao.get(2L);
				}
				tStandardApiVersion.setApprovedStatus(tApprovedStatus);
                tStandardApiVersion.setApprovedTime(LocalDateTime.now());
				standardApiVersionDao.saveOrupdate(tStandardApiVersion);
				Long tApiVerId = tStandardApiVersion.getId();
				if (tVersion.startsWith("1.")){
					tListApiVerIdNeedToSyncMetadata.add(tApiVerId);
				}

				// 同步开发平台
                if (StrUtil.isNotEmpty(ApiMgmtApplicationParameter.SYNC_DEV_URL) && tVersion.startsWith("1.")) {
                    DevelopmentPlatformServiceUtil.sendApiToDevelopmentPlatform(tStandardApiVersion);
                }
			}
		}
		if (tTanantIsSkipReview) {
            if (!tListApiVerIdNeedToSyncMetadata.isEmpty() && CollUtil.isNotEmpty(mdcProp.getReviewUrlList())) {
                String tToken = ApiMgmtServiceContextHolder.getToken();
				//同步MDC元数据
                for (String url : mdcProp.getReviewUrlList()) {
					syncMdcServiceProcessor.runSyncMdcMetadata(
							tListApiVerIdNeedToSyncMetadata, url, tUserId, tToken, routerKey);
				}
			}
		}
		if (tApiSpecErrorExist) {
			return result(tStateCode, tDescription, tTotalApiErrorMap);
		}
		tDescription = tStateCode.getDescription();
		// 組成回傳訊息
		tResponse.put(ApimgmtConstant.CODE, tStateCode.getCode());
		tResponse.put(ApimgmtConstant.DESC, tDescription);
		return tResponse;
	}

	private Map<String, Object> result(StateCode tStateCode, String tDescription,
									   Map<String, List<Map<String, String>>> pTotalApiErrorMap) throws Exception {
		ObjectNode tResponseNode = mapper.createObjectNode();
		ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
		tStateCode = getStateCode(ValidateStateEnum.IMPORT_SPEC_FORMAT_ERROR.getCode());
		tDescription = tStateCode.getDescription();
		// 取得樣板
		HSSFWorkbook tHSSFWorkbook = StandardApiExportServiceUtil.getTemplateFile(false);
		Sheet tErrorSheet = tHSSFWorkbook.createSheet("錯誤明細");
		StandardApiFileReadServiceUtil.writeImportErrorFile(tErrorSheet, pTotalApiErrorMap);
        String tFileName = ApiMgmtApplicationParameter._TEMPLATE_FILE_NAME;
        String tFileExportPath = ApiMgmtApplicationParameter._FILE_EXPORT_PATH + "/" + tFileName;
		FileOutputStream fileOut = new FileOutputStream(tFileExportPath);
		tHSSFWorkbook.write(fileOut);
		fileOut.close();
		int pFailedNum = pTotalApiErrorMap.keySet().size();
		//上傳檔案並提供下載連結
		DmcFileServiceUtil.uploadFileAndGenerateDownloadLink(tResponseNode, tFileExportPath, tFileName, true, pFailedNum);
		tResponseNode.put(ApimgmtConstant.CODE, tStateCode.getCode());
		tResponseNode.put(ApimgmtConstant.DESC, tDescription);
		return converJsonNodeToMap(tResponseJsonNode);
	}

	private Map<String, StandardApiDataName> convertToMap(List<StandardApiDataName> pStandardApiDataNameList) {
		Map<String, StandardApiDataName> tDataNameMap = new HashMap<>();
		for (StandardApiDataName tStandardApiDataName : pStandardApiDataNameList) {
			String tColumnType = tStandardApiDataName.getColumnType();
			String tDataName = tStandardApiDataName.getStandardDataName();
			String tDataType = tStandardApiDataName.getMsgType().toString();
			StandardApiDataName tParent = tStandardApiDataName.getParent();
			String tParentDataName = tParent == null ? "" : tParent.getStandardDataName();
			tDataNameMap.put(tColumnType + tDataName + tDataType + tParentDataName, tStandardApiDataName);
		}
		return tDataNameMap;
	}

	/**
	 * 更新本地缓存-导入+送审+定版都有涉及到
	 * @param pStandardApiDataNames pStandardApiDataNames
	 * @param pUserId pUserId
	 * @param pSkipReview pSkipReview
	 */
	private void checkNewDataName(List<StandardApiDataName> pStandardApiDataNames, String pUserId, boolean pSkipReview){
		// 檢查是否有新詞彙
		List<String> tDataNameAllList = null;
		LOCK.lock();
		try {
			if (!pSkipReview) {
				tDataNameAllList = ApiDataNameServiceUtil.getApiDataNameList();
				Long tDataNameCount = standardDataNameDao.getDataNameCount();
				if (tDataNameAllList.size() != tDataNameCount) {
					tDataNameAllList = standardDataNameDao.getAllDataName();
					ApiDataNameServiceUtil.setApiDataNameList(tDataNameAllList);
				}
			}
		} finally {
			LOCK.unlock();
		}
		if (!pSkipReview) {
			for(StandardApiDataName tStandardApiDataName : pStandardApiDataNames){
				String tDataName = tStandardApiDataName.getStandardDataName();
				String tFlag = tDataNameAllList.stream().filter(dataName -> dataName.contains(tDataName)).findAny().orElse(null);
				// 若詞彙不存在詞彙庫，則自動新增至詞彙庫
				if (tFlag == null) {
                    addDataName(tDataName, tStandardApiDataName.getDescriptionZhTw(),
                            tStandardApiDataName.getDescriptionZhCn(), tStandardApiDataName.getDescriptionEnUs(),
                            pUserId, pUserId, LocalDateTime.now());
				}
			}
		}
	}
	
	private void addDataName(String pDataName, String pDescriptionZhTW, String pDescriptionZhCN, String pDescriptionEnUs,
                             String tBuildAcct, String pLoginAcct, LocalDateTime pBuildTime) {
		StandardDataName tStandardDataName = new StandardDataName();
		tStandardDataName.setDataName(pDataName);
		tStandardDataName.setDescriptionZhTw(pDescriptionZhTW);
		tStandardDataName.setDescriptionZhCn(pDescriptionZhCN);
		tStandardDataName.setDescriptionEnUs(pDescriptionEnUs);
		// 狀態:草稿
		tStandardDataName.setApprovedStatus(ApimgmtConstant.APPROVED_STATUS_DRAFT);
		// 核可人:登入帳號
		tStandardDataName.setApprovedAcct(pLoginAcct);
		// 核可日期
        tStandardDataName.setApprovedTime(LocalDateTime.now());
		tStandardDataName.setBuildTime(pBuildTime);
		tStandardDataName.setBuildAcct(tBuildAcct);
        tStandardDataName.setLastUpdateTime(LocalDateTime.now());
		tStandardDataName.setDirty(false);
		standardDataNameDao.saveOrupdate(tStandardDataName);
	}

}
