package com.digiwin.athena.apimgmt.apiservice;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.constants.ApimgmtSchemaConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtTenantConfigDao;
import com.digiwin.athena.apimgmt.enums.ApiAttributeEnum;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.model.AdvanceSearchValue;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.service.model.Page;
import com.digiwin.athena.apimgmt.service.util.DmcFileServiceUtil;
import com.digiwin.athena.apimgmt.service.util.StandardApiExportWithXssServiceUtil;
import com.digiwin.athena.apimgmt.util.StandardApiMessageUtil;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.FileOutputStream;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 标准api查询导出
 */
@Slf4j
@Service
public class StandardApiQueryExportService extends AbstractApiService {

    @Autowired
    private ApiMgmtTenantConfigDao tenantConfigDao;

    @Autowired
    private ApiMgmtStandardApiDao standardApiDao;

    @Autowired
    private AdvanceSearchApiListGetOtherService advanceSearchApiListGetOtherService;

    public StandardApiQueryExportService() {
        super();
        jsonSchemaFileName = ApimgmtSchemaConstant.STANDARD_API_QUERY_EXPORT_SCHEMA;
    }

    @Override
    protected Map<String, Object> processData(ValidatorResult validatorResult) throws Exception {
        String tLocale = ApiMgmtServiceContextHolder.getLocale();
        // header沒傳語系的話，默認回傳英文
        tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
        StateCode tStateCode = getStateCode(validatorResult.getState().getCode());
        String tDescription = tStateCode.getDescription();
        JsonNode tRequestJsonNode = validatorResult.getJsonContent();
        // 建立response node
        ObjectNode tResponseNode = mapper.createObjectNode();
        // 进阶查询检索条件
        ArrayNode tConditionArrayNode = (ArrayNode) tRequestJsonNode.get(ApimgmtConstant.CONDITION_LIST);
        // 标准API列表查询入参
        JsonNode tCategoryId = tRequestJsonNode.get(ApimgmtConstant.CATEGORY_ID);
        // 产品ID
        JsonNode projectId = tRequestJsonNode.get(ApimgmtConstant.PROJECT_ID);
        //产品版本ID
        JsonNode projectVersionId = tRequestJsonNode.get(ApimgmtConstant.PROJECT_VERSION_ID);
        // 查询类型: true:设计api查询、false:标准api查询
        JsonNode tType = tRequestJsonNode.get(ApimgmtConstant.TYPE_STRING);
        if (null == tType || StringUtil.isEmptyOrSpace(tType.asBoolean())) {
            throw new Exception("请选择查询类型!");
        }
        String tToken = ApiMgmtServiceContextHolder.getToken();
        // 设计API查询-必须登录
        if (tType.asBoolean() && null == tToken) {
            throw new Exception("当前用户未登录,请登录!");
        }
        String tTenantId = ApiMgmtServiceContextHolder.getTenantId();
        String teamType =  ApiMgmtServiceContextHolder.getTeamType();
        // 标准API查询-判断租戶是否可查看全部API
        boolean tViewAllApi = false;
        if (!tType.asBoolean() && !StringUtil.isEmptyOrSpace(tTenantId)) {
            tViewAllApi = tenantConfigDao.viewAllApiTenantExist(tTenantId);
        }
        FileOutputStream fileOut = null;
        try {
            XSSFWorkbook xssfWorkbook = StandardApiExportWithXssServiceUtil.createXSSFWorkbook();
            String tNewFileName = "StdApi_" + DateUtil.format(LocalDateTime.now(), DatePattern.PURE_DATETIME_MS_PATTERN) + ".xlsx";
            String tNewFilePath = ApiMgmtApplicationParameter._FILE_EXPORT_PATH + "/" + tNewFileName;
            fileOut = new FileOutputStream(tNewFilePath);
            // 查询条件
            List<List<AdvanceSearchValue>> arrayList = StandardApiMessageUtil.generateQuery(tConditionArrayNode, tCategoryId, null, tType, projectId, projectVersionId, null);
            Page<StandardApiVersion> standardApiVersionPage = standardApiDao.advanceApiSearch(arrayList, teamType, tTenantId,
                    tType.asBoolean(), tViewAllApi, 1, 50);
            setAdvanceSearchApiResponse(standardApiVersionPage, tLocale, tType.asBoolean(), xssfWorkbook);

            xssfWorkbook.write(fileOut);

            DmcFileServiceUtil.uploadFileAndGenerateDownloadLink(tResponseNode, tNewFilePath, tNewFileName, false, 0);
            boolean b = StandardApiExportWithXssServiceUtil.removeFile(tNewFilePath);
            log.info("导出文件删除:{}" + b);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
            tDescription = tStateCode.getDescription();
        }finally {
            assert fileOut != null;
            fileOut.close();
        }
        ObjectNode tResponseJsonNode = createResponseJsonNode(tStateCode.getCode(), tDescription, tResponseNode);
        return converJsonNodeToMap(tResponseJsonNode);
    }

    private void setAdvanceSearchApiResponse(Page<StandardApiVersion> pStandardApiVersionPage, String tLocale,
                                             boolean tType, XSSFWorkbook xssfWorkbook) {
        int number = 0;
        if (CollUtil.isNotEmpty(pStandardApiVersionPage.getList())) {
            List<StandardApi> standardApiList = pStandardApiVersionPage.getList().stream().map(StandardApiVersion::getStandardApi).collect(Collectors.toList());
            advanceSearchApiListGetOtherService.setProjectAndVersion(standardApiList);
            for (StandardApiVersion tStandardApiVersion : pStandardApiVersionPage.getList()) {
                Map<String, Object> tApiMap = new HashMap<>();
                StandardApi tStandardApi = tStandardApiVersion.getStandardApi();
                // 检视所有版本
                if (!tType) {
                    // 最大1.0版本说明只有1.0，最大不是1.0则查询1.0的版本
                    if (tStandardApiVersion.getVersion().endsWith(".0")) {
                        tApiMap.put(ApimgmtConstant.FIRST_APPLICANT, tStandardApiVersion.getApplicant());
                        tApiMap.put(ApimgmtConstant.LAST_APPLICANT, "");
                    } else {
                        tApiMap.put(ApimgmtConstant.FIRST_APPLICANT, tStandardApi.getFirstApplicant());
                        tApiMap.put(ApimgmtConstant.LAST_APPLICANT, tStandardApiVersion.getApplicant());
                    }
                }
                String tApprovedStatus;
                long tApprovedStatusCode;
                String tDescription;
                String tRemark;
                String tSyncType;
                String tCategory;
                switch (tLocale) {
                    case ApimgmtConstant.ZH_CN:
                        tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameZhCn();
                        tApprovedStatusCode = tStandardApiVersion.getApprovedStatus().getId();
                        tDescription = tStandardApi.getDescriptionZhCn();
                        tRemark = tStandardApi.getRemarkZhCn();
                        tSyncType = tStandardApi.getStandardApiSyncType().getNameZhCn();
                        tCategory = tStandardApi.getStandardApiCategory().getNameZhCn();
                        break;
                    case ApimgmtConstant.ZH_TW:
                        tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameZhTw();
                        tApprovedStatusCode = tStandardApiVersion.getApprovedStatus().getId();
                        tDescription = tStandardApi.getDescriptionZhTw();
                        tRemark = tStandardApi.getRemarkZhTw();
                        tSyncType = tStandardApi.getStandardApiSyncType().getNameZhTw();
                        tCategory = tStandardApi.getStandardApiCategory().getNameZhTw();
                        break;
                    case ApimgmtConstant.EN_US:
                    default:
                        tApprovedStatus = tStandardApiVersion.getApprovedStatus().getNameEnUs();
                        tApprovedStatusCode = tStandardApiVersion.getApprovedStatus().getId();
                        tDescription = tStandardApi.getDescriptionEnUs();
                        tRemark = tStandardApi.getRemarkEnUs();
                        tSyncType = tStandardApi.getStandardApiSyncType().getNameEnUs();
                        tCategory = tStandardApi.getStandardApiCategory().getNameEnUs();
                        break;
                }
                // 組成回傳訊息
                tApiMap.put(ApimgmtConstant.NUMERIC, number);
                tApiMap.put(ApimgmtConstant.API_ID, tStandardApi.getId());
                tApiMap.put(ApimgmtConstant.NAME, tStandardApi.getName());
                tApiMap.put(ApimgmtConstant.VERSION, tStandardApiVersion.getVersion());
                tApiMap.put(ApimgmtConstant.APPROVED_STATUS, tApprovedStatus);
                tApiMap.put(ApiAttributeEnum.approvedTime.toString(), DateUtil.formatLocalDateTime(tStandardApiVersion.getApprovedTime()));
                tApiMap.put(ApimgmtConstant.APPROVED_STATUS_CODE, tApprovedStatusCode);
                tApiMap.put(ApimgmtConstant.DESCRIPTION, tDescription);
                tApiMap.put(ApimgmtConstant.REMARK, tRemark);
                tApiMap.put(ApimgmtConstant.SYNC_TYPE, tSyncType);
                tApiMap.put(ApimgmtConstant.CATEGORY_ID, tStandardApi.getStandardApiCategory().getId());
                tApiMap.put(ApimgmtConstant.CATEGORY_NAME, tCategory);
                tApiMap.put(ApimgmtConstant.MSG_FORMAT, tStandardApi.getMsgFormat());
                tApiMap.put(ApimgmtConstant.REQUESTER, tStandardApi.getProvider());
                tApiMap.put(ApimgmtConstant.PROVIDER, tStandardApi.getProvider());
                tApiMap.put(ApimgmtConstant.TENANT_ID, tStandardApi.getTenantId());
                tApiMap.put(ApimgmtConstant.API_BUILD_ACCT, tStandardApi.getBuildAcct());
                tApiMap.put(ApimgmtConstant.API_BUILD_TIME, DateUtil.formatLocalDateTime(tStandardApi.getBuildTime()));
                tApiMap.put(ApimgmtConstant.DESIGN_TENANT_ID, tStandardApiVersion.getStandardApi().getDesignTenantId());
                tApiMap.put(ApimgmtConstant.TEAM_ID, tStandardApiVersion.getStandardApi().getTeamId());
                tApiMap.put(ApimgmtConstant.STATUS_ID, tStandardApiVersion.getApprovedStatus().getId());
                tApiMap.put(ApimgmtConstant.API_TYPE, tStandardApiVersion.getStandardApi().getApiType());
                tApiMap.put(ApimgmtConstant.PROJECT_NAME, tStandardApiVersion.getStandardApi().getProjectName());
                if (tStandardApiVersion.getStandardApi().getProjectVersionNameList() != null) {
                    ArrayNode projectVersionNameListNode = mapper.createArrayNode();
                    tStandardApiVersion.getStandardApi().getProjectVersionNameList().forEach(name -> projectVersionNameListNode.add(name));
                    tApiMap.put(ApimgmtConstant.PROJECT_VERSION_NAME_LIST, projectVersionNameListNode);
                    tApiMap.put(ApimgmtConstant.PROJECT_VERSION_NAME_LIST_STR, String.join(ApimgmtConstant.SPLITER, tStandardApiVersion.getStandardApi().getProjectVersionNameList()));
                }
                //设置api简易信息
                StandardApiExportWithXssServiceUtil.setEasyApiInfo(xssfWorkbook.getSheetAt(0), tApiMap);
                //设置api详情信息
                StandardApiExportWithXssServiceUtil.createXSSFSheet(xssfWorkbook, tStandardApiVersion, number);
                number++;
            }
        }
    }
}
