package com.digiwin.athena.apimgmt.services;

import com.digiwin.athena.apimgmt.enums.CommunicationEnum;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtMailProp;
import com.digiwin.athena.apimgmt.util.StringUtil;
import jakarta.mail.MessagingException;
import jakarta.mail.internet.MimeMessage;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Component;

import java.util.Properties;

@Slf4j
@Component("com.digiwin.dsms.service.MailService")
public class ApiMgmtMailService implements InitializingBean {

	private final JavaMailSenderImpl mailSender = new JavaMailSenderImpl();

    @Autowired
    private ApiMgmtMailProp mailProp;

    public ApiMgmtMailService() {
	}

	/**
	 * 傳送Mail訊息
	 * 
	 * @param pTo 收件者的 email address
	 * @param pMailContent 信件內文
	 * @param pSubject 信件標題(主旨)
	 * @throws MessagingException MessagingException
	 */
	public void sendMail(String pTo, String pMailContent, String pSubject) throws MessagingException {
        if (!mailProp.getEnabled()) {
            log.warn("邮件通知已关闭！to:{},subject:{},content:{}", pTo, pSubject, pMailContent);
            return;
        }

        // 建立郵件訊息
		MimeMessage tMailMessage = mailSender.createMimeMessage();
		MimeMessageHelper tMessageHelper = new MimeMessageHelper(tMailMessage, true, "utf-8");
		tMessageHelper.setTo(pTo);
        tMessageHelper.setFrom(mailProp.getFrom());
		tMessageHelper.setSubject(pSubject);
        // 設定信件內文.
		tMessageHelper.setText(pMailContent, true);
		// 傳送郵件
		log.info("Send email to: " + pTo);
		mailSender.send(tMailMessage);
	}
	
	public void sendMailToAdmin(String pMailContent, String pSubject) {
        String pTo = mailProp.getAlertAddress();
		try {
			sendMail(pTo, pMailContent, pSubject);
		} catch (Exception e) {
			log.error("Send Email message error.", e);
		}
	}

    protected void init() {
        log.info("Mail service start");
        // 設定 Mail Server
        mailSender.setHost(mailProp.getServer().getHostname());
        mailSender.setPort(mailProp.getServer().getPort());
        mailSender.setUsername(mailProp.getServer().getUsername());
        mailSender.setPassword(mailProp.getServer().getPassword());
        if (StringUtil.isEmptyOrSpace(mailProp.getServer().getUsername())) {
            mailSender.setJavaMailProperties(new MailServiceProperties(mailProp, false));
        } else {
            mailSender.setJavaMailProperties(new MailServiceProperties(mailProp, true));
        }
    }

    @Override
    public void afterPropertiesSet() {
        init();
    }

    static class MailServiceProperties extends Properties {
		private static final long serialVersionUID = -6985941591953133952L;

        public MailServiceProperties(ApiMgmtMailProp mailProp, boolean pIsSmtpAuth) {
			super.setProperty("mail.smtp.auth", String.valueOf(pIsSmtpAuth));
			// TLS
            String authentication = mailProp.getServer().getAuthentication();
            if (CommunicationEnum.TLS.getCode().equals(authentication)) {
				super.setProperty("mail.smtp.starttls.enable", "true");
			}
            if (CommunicationEnum.SSL.getCode().equals(authentication)) {
				super.setProperty("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
			}
		}
	}
}