package com.digiwin.athena.apimgmt.dao.impl.mybatis;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Pair;
import cn.hutool.core.stream.CollectorUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDataNameDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.dao.impl.mybatis.mapper.ApiMgmtStandardApiVersionMapper;
import com.digiwin.athena.apimgmt.enums.ApiVersionEnum;
import com.digiwin.athena.apimgmt.model.ApiCountKeyValue;
import com.digiwin.athena.apimgmt.model.DirtyData;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.digiwin.athena.apimgmt.model.StandardApiVersion.Status.DEVELOPING;

@Primary
@Repository
@RequiredArgsConstructor
public class ApiMgmtStandardApiVersionDaoMybatisImpl extends GenericDaoMybatisImpl<ApiMgmtStandardApiVersionMapper, StandardApiVersion, Long>
        implements ApiMgmtStandardApiVersionDao {

    private ApiMgmtStandardApiDao apiDao;
    private ApiMgmtStandardApiDataNameDao apiDataNameDao;

    @Override
    public List<StandardApiVersion> fetch(Map<String, String> pConditions) {
        MpPage<?> page = MpPage.from(pConditions);
        List<StandardApiVersion> verList = baseMapper.fetchByConditions(pConditions, page);

        // mybatis 懒加载，懒加载语句某种情况下存在未被Plugin代理，导致拦截器失效问题，因此动态表名插件失效
        apiDao.fillApi(verList);

        return verList;
    }

    @Override
    public List<StandardApiVersion> getVersionByApiId(Long pId, String pUserId, String pStatus) {
        return baseMapper.getVersionByApiId(pId, pUserId, pStatus);
    }

    @Override
    public List<StandardApiVersion> getVersionListByApiId(Long pId) {
        LambdaQueryWrapper<StandardApiVersion> wrapper = Wrappers.lambdaQuery(StandardApiVersion.class)
                .eq(StandardApiVersion::getApiId, pId);
        return baseMapper.selectList(wrapper);
    }

    @Override
    public Map<Long, Long> advanceStandardApiCountByVersionIds(List<Long> versionIds) {
        return baseMapper.advanceStandardApiCountByVersionIds(versionIds).stream()
                .collect(CollectorUtil.toMap(
                        v -> NumberUtil.parseLong(StrUtil.toStringOrNull(v.getKey())),
                        Pair::getValue,
                        Long::sum
                ));
    }

    @Override
    public void batchSave(List<StandardApiVersion> standardApiVersions) {
        batchService.saveBatch(standardApiVersions, 100);

        apiDataNameDao.batchSave(
                standardApiVersions.stream()
                        .flatMap(v -> v.getStandardApiDataNames().stream())
                        .collect(Collectors.toList())
        );
    }

    @Override
    @Transactional
    public void delete(Long pId, boolean pRemoveApi) {
        if (pRemoveApi) {
            StandardApiVersion version = get(pId);
            apiDao.delete(version.getApiId());
        }

        apiDataNameDao.deleteByApiVerId(pId);
        baseMapper.deleteById(pId);
    }

    @Override
    public List<StandardApiVersion> fetchApiVersionsByName(String pApiName, String branch, String pTenantId) {
        return baseMapper.fetchApiVersionsByNameSimple(pApiName, branch, pTenantId);
    }

    @Override
    public List<StandardApiVersion> fetchApiVersionsByName(String pApiName, String branch, String pTenantId,
                                                           String tTenantId, String teamType, boolean pViewAllApi) {
        return baseMapper.fetchApiVersionsByNameWithAuth(pApiName, branch, tTenantId, pTenantId, teamType, pViewAllApi);
    }

    @Override
    public List<StandardApiVersion> getByStatus(Long pStatus, String pUserId, String pTenantId,
                                                boolean pIsVerifyAcct) {
        return baseMapper.getByStatus(pStatus, pUserId, pTenantId, pIsVerifyAcct);
    }

    @Override
    public List<ApiCountKeyValue> getApiCountByStatus(String pUserId, String pTenantId, boolean pIsVerifyAcct) {
        List<Integer> approvedStatus = pIsVerifyAcct ?
                StandardApiVersion.Status.needVerifyAcct()
                : StandardApiVersion.Status.notNeedVerifyAcct();
        return baseMapper.getApiCountByStatus(pUserId, pTenantId, pIsVerifyAcct, approvedStatus);
    }

    @Override
    public StandardApiVersion fetchApiVersion(Map<String, String> pConditions) {
        return baseMapper.fetchApiVersion(pConditions);
    }

    @Override
    public List<StandardApiVersion> fetchApiVersions(String pApiName, String pVersion) {
        return fetchApiVersions(pApiName, "", pVersion);
    }

    @Override
    public List<StandardApiVersion> fetchApiVersions(String pApiName, String pTenantId, String pVersion) {
        Map<String, String> conditions = new HashMap<>();
        conditions.put(ApiVersionEnum.API_NAME.toString(), pApiName);
        conditions.put(ApiVersionEnum.TENANT_ID_FULL_MATCHING.toString(), StringUtils.defaultIfBlank(pTenantId, ""));
        conditions.put(ApiVersionEnum.API_VERSION.toString(), pVersion);
        conditions.put(ApiVersionEnum.APPROVED_STATUS.toString(), String.valueOf(DEVELOPING));
        return fetch(conditions);
    }

    @Override
    public void fillApiLatestVersion(List<StandardApi> apiList) {
        if (CollUtil.isNotEmpty(apiList)) {
            return;
        }

        Map<Long, StandardApiVersion> apiIdMap = baseMapper.selectLatestByApiId(
                apiList.stream().map(StandardApi::getId).collect(Collectors.toList())
        );

        apiList.forEach(ver ->
                ver.setStandardApiVersions(CollUtil.toList(apiIdMap.get(ver.getId())))
        );
    }

    @Override
    public List<StandardApiVersion> getExistApiVersion(String pApiName, String pTenantId, String pVersion) {
        Map<String, String> conditions = new HashMap<>();
        conditions.put(ApiVersionEnum.API_NAME.toString(), pApiName);
        conditions.put(ApiVersionEnum.TENANT_ID_FULL_MATCHING.toString(), pTenantId);
        conditions.put(ApiVersionEnum.API_VERSION.toString(), pVersion);
        return fetch(conditions);
    }

    @Override
    public List<StandardApiVersion> getByCategory(Long pCategory, String pTenantId, String teamType,
                                                  boolean pViewAllApi) {
        List<StandardApiVersion> list = baseMapper.getByCategory(pCategory, pTenantId, teamType, pViewAllApi);
        apiDao.fillApi(list);
        return list;
    }

    @Override
    public List<DirtyData> queryDirtyData() {
        return baseMapper.queryDirtyData();
    }

    @Autowired
    @Lazy
    public void setApiDao(ApiMgmtStandardApiDao apiDao) {
        this.apiDao = apiDao;
    }

    @Autowired
    @Lazy
    public void setApiDataNameDao(ApiMgmtStandardApiDataNameDao apiDataNameDao) {
        this.apiDataNameDao = apiDataNameDao;
    }
}
