package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.apimgmt.ApiMgmtApplicationParameter;
import com.digiwin.athena.apimgmt.common.model.dto.FileDto;
import com.digiwin.athena.apimgmt.constants.ApiTypeConstants;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.*;
import com.digiwin.athena.apimgmt.enums.APIExportImportEnums;
import com.digiwin.athena.apimgmt.enums.LocaleEnum;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.exception.ImportApiNotFindValidationException;
import com.digiwin.athena.apimgmt.exception.TeamIdNotFindValidationException;
import com.digiwin.athena.apimgmt.infra.context.ApiMgmtServiceContextHolder;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtProp;
import com.digiwin.athena.apimgmt.model.*;
import com.digiwin.athena.apimgmt.service.util.*;
import com.digiwin.athena.apimgmt.services.ApiMgmtApiVersionService;
import com.digiwin.athena.apimgmt.services.ApiMgmtProductCategoryService;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FilenameUtils;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * API導入
 * /restful/standard/apimgmt/ApiImport/Update
 */
@Slf4j
@Service
public class ApiImportService {

    ObjectMapper mapper = new ObjectMapper();

    private static final Lock LOCK = new ReentrantLock(true);


    @Autowired
    ApiMgmtTenantConfigDao tenantConfigDao;
    @Autowired
    private ApiMgmtStateCodeDao stateCodeDao;
    @Autowired
    private ApiMgmtStandardApiDao standardApiDao;
    @Autowired
    private ApiMgmtStandardApiVersionDao standardApiVersionDao;
    @Autowired
    private ApiMgmtStandardDataNameDao standardDataNameDao;
    @Autowired
    private ApiMgmtProductCategoryService productCategoryService;

    @Autowired
    private ApiMgmtApiVersionService apiVersionService;

    @Autowired
    private ApiMgmtProp prop;

    public ApiImportService() {
        super();
    }

    public Map<String, Object> execute(FileDto file) throws Exception {
        if (prop.isEnvType()) {
            throw new Exception("此操作不对外开放,请联系管理员!");
        }
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口：" + this.getClass());
        // 建立response node
        ObjectNode tResponseNode = mapper.createObjectNode();
        StateCode tStateCode = stateCodeDao.getStateCodeByCode(ValidateStateEnum.SUCCESS.getCode());
        String tDescription = tStateCode.getDescription();
        StringBuilder tDescriptionBuilder = new StringBuilder();
        try {
            // 取得語系
            String tLocale = ApiMgmtServiceContextHolder.getLocale();
            // header沒傳語系的話，默認回傳英文
            tLocale = tLocale == null ? LocaleEnum.EN_US.getType() : tLocale;
            // 新增租户类型ID 如果不存在会提示错误
            String tTeamId = ApiMgmtServiceContextHolder.getTeamId();
            String teamType = ApiMgmtServiceContextHolder.getTeamType();
            if (null == teamType) {
                throw new TeamIdNotFindValidationException(getClass());
            }
            // 取得用戶資訊
            String tTenant = ApiMgmtServiceContextHolder.getTenantId();
            String tUserId = ApiMgmtServiceContextHolder.getUserId();
            String tUserName = ApiMgmtServiceContextHolder.getUserName();
            // 把檔案轉乘inputstream
            InputStream fileIn = file.getInputStream();
            // 取得副檔名
            String fileType = FilenameUtils.getExtension(file.getFileName());
            // 轉成Workbook
            Workbook tWorkbook = StandardApiFileReadServiceUtil.getWorkBook(fileType, fileIn);
            boolean tApiSpecErrorExist = false;
            log.info("导入的Excel有" + tWorkbook.getNumberOfSheets() + "个Sheet页");
            ArrayList<Integer> tApiSheetIndex = new ArrayList<>();
            for (int tSheetNumber = 0; tSheetNumber < tWorkbook.getNumberOfSheets(); tSheetNumber++) {
                Sheet tSheet = tWorkbook.getSheetAt(tSheetNumber);
                String tSheetName = tSheet.getSheetName().trim();
                if (StandardApi.validateName(ApiTypeConstants.ESP_CODE, tSheetName)
                        || StandardApi.validateName(ApiTypeConstants.OPENAPI_CODE, tSheetName)) {
                    tApiSheetIndex.add(tSheetNumber);
                }
            }
            // 如果apiSheet没有需要导入直接提示异常
            if (CollUtil.isEmpty(tApiSheetIndex)) {
                throw new ImportApiNotFindValidationException(getClass());
            }
            // 如果Tenant被設定為略過不檢查字段，則不進記憶體or資料庫查Data Name清單
            boolean tSkipReview = tenantConfigDao.getSkipReviewDataNameTenant(tTenant);
            List<String> tDataNameAllList;
            LOCK.lock();
            try {
                if (!tSkipReview) {
                    tDataNameAllList = ApiDataNameServiceUtil.getApiDataNameList();
                    Long tDataNameCount = standardDataNameDao.getDataNameCount();
                    if (tDataNameAllList.size() != tDataNameCount) {
                        tDataNameAllList = standardDataNameDao.getAllDataName();
                        ApiDataNameServiceUtil.setApiDataNameList(tDataNameAllList);
                    }
                }
            } finally {
                LOCK.unlock();
            }
            // 全部API的錯誤總Map
            Map<String, List<Map<String, String>>> tTotalApiErrorMap = new HashMap<>();
            // 讀取模版
            for (int tSheetNumber : tApiSheetIndex) {
                log.info("检查样板Sheet索引为: " + tSheetNumber);
                // 隱藏的sheet不讀取
                if (tWorkbook.isSheetHidden(tSheetNumber)) {
                    continue;
                }
                StandardApi tStandardApi = new StandardApi();
                Sheet tSheet = tWorkbook.getSheetAt(tSheetNumber);
                // 讀取StandardAPI資訊
                List<Map<String, String>> tSetStdApiResult;
                tSetStdApiResult = StandardApiFileReadServiceUtil.setStandardApi(tStandardApi, tUserName, tSheet, tLocale);
                String tApiName = tStandardApi.getName();
                String tTenantId = tStandardApi.getTenantId();
                String tVersion = Optional.ofNullable(CollUtil.getFirst(tStandardApi.getStandardApiVersions()))
                        .map(StandardApiVersion::getVersion).orElse("");
                String branch = tStandardApi.getBranch();
                if (CollUtil.isNotEmpty(tSetStdApiResult)) {
                    String tApiError = tApiName + "/" + tVersion;
                    tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                    tApiSpecErrorExist = true;
                    continue;
                }
                tStandardApi.setTeamId(tTeamId);
                tStandardApi.setTeamType(teamType);
                tStandardApi.setDesignTenantId(tTenant);
                // API已存在，略過執行下一筆
                List<StandardApiVersion> tExistStandardApiVersions = standardApiVersionDao.getExistApiVersion(tApiName, tTenantId, tVersion);
                if (!tExistStandardApiVersions.isEmpty()) {
                    StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.API_VERSION_IS_EXIST.toString(), null, null, 0, 0, tLocale);
                    String tApiError = tApiName + "/" + tVersion;
                    tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                    tApiSpecErrorExist = true;
                    continue;
                }
                // API不存在，但是有發生規格異常，將異常寫到excel檔案
                if (tSetStdApiResult.size() != 0) {
                    String tApiError = tApiName + "/" + tVersion;
                    tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                    tApiSpecErrorExist = true;
                    continue;
                }
                for (StandardApiVersion tapiVersion : tStandardApi.getStandardApiVersions()) {
                    tapiVersion.setApprovedAcct(tUserId);
                    tapiVersion.setUserId(tUserId);
                }
                tStandardApi.setBuildAcct(tUserId);
                // 檢查前一版是否匯入，除了1.0不檢查
                //if (Float.parseFloat(tVersion) > Float.parseFloat(ApimgmtConstant.API_VER)) {
                if (!apiVersionService.judgeNewBranch(tVersion)) {
                    String prevVersion = ApiVersionServiceUtil.getPreviousVersion(tVersion);
                    // 检查上一版本/分支是否存在，不允许跳版
                    List<StandardApiVersion> tPreviStandardApiVersions = standardApiVersionDao.getExistApiVersion(tApiName, tTenantId, prevVersion);
                    if (tPreviStandardApiVersions.isEmpty()) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PREVI_API_VERSION_IS_EXIST.toString(),
                                null, null, 0, 0, tLocale);
                        String tApiError = tApiName + "/" + tVersion;
                        tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                        tApiSpecErrorExist = true;
                        continue;
                    }

                    Long approvedStatus = Optional.ofNullable(CollUtil.getFirst(tPreviStandardApiVersions))
                            .map(StandardApiVersion::getApprovedStatus)
                            .map(ApprovedStatus::getId).orElse(null);
                    if (!Objects.equals(approvedStatus, ApimgmtConstant.APPROVED_STATUS_COMFIRMED)) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PREVI_IS_DEVELOPING.toString(), null, null, 0, 0, tLocale);
                        String tApiError = tApiName + "/" + tVersion;
                        tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                        tApiSpecErrorExist = true;
                        continue;
                    }

                    // 檢查「須分包」是否有異動
                    List<StandardApi> tPreviStandardApis = standardApiDao.getListByNameAndBranchAndTenantId(tApiName, branch, tTenantId);
                    if (CollUtil.isNotEmpty(tPreviStandardApis)) {
                        boolean tBatchIsChanged = ValidateApiDataNameServiceUtil
                                .checkBatchIsChanged(tPreviStandardApis.get(0).getIsBatch(), tStandardApi.getIsBatch());
                        if (tBatchIsChanged) {
                            StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.IS_BATCH_CAN_NOT_CHANGE.toString(), null, null, 0, 0, tLocale);
                            String tApiError = tApiName + "/" + tVersion;
                            tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                            tApiSpecErrorExist = true;
                            continue;
                        }
                        // 檢查「調用模式」是否有異動
                        boolean tSyncTypeIsChanged = ValidateApiDataNameServiceUtil
                                .checkSyncTypeIsChanged(tPreviStandardApis.get(0).getStandardApiSyncType().getId(), tStandardApi.getStandardApiSyncType().getId());
                        if (tSyncTypeIsChanged) {
                            StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.SYNC_TYPE_CAN_NOT_CHANGE.toString(), null, null, 0, 0, tLocale);
                            String tApiError = tApiName + "/" + tVersion;
                            tTotalApiErrorMap.put(tApiError, tSetStdApiResult);
                            tApiSpecErrorExist = true;
                            continue;
                        }
                    }
                } else if (!ApimgmtConstant.API_VER.equals(tVersion)) {
                    String prevBranch = ApiVersionServiceUtil.getPrevBranchByVersion(tVersion);
                    // 检查上一版本/分支是否存在，不允许跳版
                    List<StandardApiVersion> tPreviStandardApiVersions = standardApiVersionDao.getExistApiVersion(tApiName, tTenantId, prevBranch);
                    if (tPreviStandardApiVersions.isEmpty()) {
                        StandardApiFileReadServiceUtil.uploadCheckResult(tSetStdApiResult, APIExportImportEnums.PREV_API_BRANCH_NOT_EXIST.toString(),
                                null, null, 0, 0, tLocale);
                        tTotalApiErrorMap.put(tApiName + "/" + tVersion, tSetStdApiResult);
                        tApiSpecErrorExist = true;
                        continue;
                    }
                }

                Map<String, StandardApiDataName> tAddDataName = new HashMap<>();
                // 讀取StandardApiDataName資訊
                List<Map<String, String>> tSetStdApiDataNameResult = StandardApiFileReadServiceUtil.setStandardApiDataName(tStandardApi, tAddDataName, tSheet, true, tLocale, tSkipReview, tUserId);
                if (!tSetStdApiDataNameResult.isEmpty()) {
                    String tApiError = tApiName + "/" + tVersion;
                    tTotalApiErrorMap.put(tApiError, tSetStdApiDataNameResult);
                    tApiSpecErrorExist = true;
                    continue;
                }
                // 該sheet的列跑完，save StandardApi
                standardApiDao.save(tStandardApi);
                //保存接收者信息
                productCategoryService.saveProductCategory(tStandardApi.getProvider());
            }
            if (tApiSpecErrorExist) {
                tStateCode = stateCodeDao.getStateCodeByCode(ValidateStateEnum.IMPORT_SPEC_FORMAT_ERROR.getCode());
                tDescription = tStateCode.getDescription();
                // 避免使用者匯入的excel有「錯誤明細」sheet，故先刪除再建立
                int tErrorSheetIndex = tWorkbook.getSheetIndex("錯誤明細");
                if (tErrorSheetIndex >= 0) {
                    tWorkbook.removeSheetAt(tErrorSheetIndex);
                }
                Sheet tErrorSheet = tWorkbook.createSheet("錯誤明細");
                StandardApiFileReadServiceUtil.writeImportErrorFile(tErrorSheet, tTotalApiErrorMap);
                // 提供詳細的下載連結
                String tDetailFilePath = ApiMgmtApplicationParameter._FILE_EXPORT_PATH + "/" + file.getFileName();
                String tDetailFileName = file.getFileName();
                FileOutputStream fileOut = new FileOutputStream(tDetailFilePath);
                tWorkbook.write(fileOut);
                fileOut.close();
                int pFailedNum = tTotalApiErrorMap.keySet().size();
                // 上傳檔案並提供下載連結
                DmcFileServiceUtil.uploadFileAndGenerateDownloadLink(tResponseNode, tDetailFilePath, tDetailFileName, true, pFailedNum);
            }
        } catch (BaseException e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(e.getStateEnum().getCode());
            tDescriptionBuilder.append(tStateCode.getDescription());
            if (!StringUtil.isEmptyOrSpace(e.getMessage())) {
                tDescriptionBuilder.append(ApimgmtConstant.BLANK).append(e.getMessage());
            }
            tDescription = tDescriptionBuilder.toString();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
            tDescriptionBuilder.append(tStateCode.getDescription()).append(ApimgmtConstant.BLANK).append(e.getClass().toString()).append(":").append(e.getMessage());
            tDescription = tDescriptionBuilder.toString();
        }
        tResponseNode.put("code", tStateCode.getCode());
        tResponseNode.put("description", tDescription);
        log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口：" + this.getClass() + "，回傳訊息：" + tResponseNode.toString());
        return mapper.convertValue(tResponseNode, new TypeReference<Map<String, Object>>() {
        });
    }

    protected StateCode getStateCode(String pStateCode) {
        return stateCodeDao.getStateCodeByCode(pStateCode);
    }
}
