package com.digiwin.athena.apimgmt.apiservice;

import cn.hutool.extra.spring.SpringUtil;
import com.digiwin.athena.apimgmt.annotate.OperateAuthorityVerify;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStateCodeDao;
import com.digiwin.athena.apimgmt.enums.ValidateStateEnum;
import com.digiwin.athena.apimgmt.exception.BaseException;
import com.digiwin.athena.apimgmt.infra.prop.ApiMgmtProp;
import com.digiwin.athena.apimgmt.model.StateCode;
import com.digiwin.athena.apimgmt.util.StringUtil;
import com.digiwin.athena.apimgmt.validator.JsonSchemaValidator;
import com.digiwin.athena.apimgmt.validator.ValidatorResult;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.lang.annotation.Annotation;
import java.util.Map;

@Slf4j
public abstract class AbstractApiService {

    /**
     * json訊息驗證器
     */
    final JsonSchemaValidator jsonSchemaValidator = new JsonSchemaValidator();

    /**
     * mapper(parse or deserialize JSON content)
     */
    ObjectMapper mapper = new ObjectMapper();

    protected String jsonSchemaFileName;

    protected String requestMsg;

    @Autowired
    private ApiMgmtProp prop;

    @Autowired
    private ApiMgmtStateCodeDao stateCodeDao;

    public Map<String, Object> execute(String pRequestMsg) throws Exception {
        StateCode tStateCode;
        StringBuilder tDescription = new StringBuilder();
        requestMsg = pRequestMsg;
        try {
            log.info("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
                    + ", 原始訊息: " + pRequestMsg);
            if (prop.isEnvType()) {
                for (Annotation annotation : this.getClass().getAnnotations()) {
                    if (OperateAuthorityVerify.class.isAssignableFrom(annotation.annotationType())) {
                        throw new Exception("此操作不对外开放,请联系管理员!");
                    }
                }
            }
            ValidatorResult tValidatorResult = validate(pRequestMsg);

            AbstractApiService thisProxy = SpringUtil.getBean(getClass());
            return thisProxy.processData(tValidatorResult);
        } catch (BaseException e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(e.getStateEnum().getCode());
            tDescription.append(tStateCode.getDescription());
            if (!StringUtil.isEmptyOrSpace(e.getMessage())) {
                tDescription.append(ApimgmtConstant.BLANK).append(e.getMessage());
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            tStateCode = getStateCode(ValidateStateEnum.UNEXPECTED_ERROR.getCode());
            tDescription.append(tStateCode.getDescription()).append(ApimgmtConstant.BLANK).append(e.getClass().toString()).append(":")
                    .append(e.getMessage());
        }
        return converJsonNodeToMap(createResponseJsonNode(tStateCode.getCode(), tDescription.toString()));
    }

    protected Map<String, Object> converJsonNodeToMap(JsonNode pJsonNode) {
        log.debug("[Thread.id " + Thread.currentThread().getId() + "]" + "接口: " + this.getClass()
                + ", 回傳訊息: " + pJsonNode.toString());
        return mapper.convertValue(pJsonNode, new TypeReference<Map<String, Object>>() {
        });
    }

    protected StateCode getStateCode(String pStateCode) {
        return stateCodeDao.getStateCodeByCode(pStateCode);
    }

    protected ObjectNode createResponseJsonNode(String pCode, String pDescription, ObjectNode pResponseNode) {
        ObjectNode tResponseNode;
        if (pResponseNode == null) {
            tResponseNode = mapper.createObjectNode();
        } else {
            tResponseNode = pResponseNode;
        }
        tResponseNode.put(ApimgmtConstant.CODE, pCode);
        if (!StringUtil.isEmptyOrSpace(pDescription)) {
            tResponseNode.put(ApimgmtConstant.DESCRIPTION, pDescription);
        }
        return tResponseNode;
    }

    /**
     * 钩子方法
     *
     * @param validatorResult validatorResult
     * @return validatorResult
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    protected abstract Map<String, Object> processData(ValidatorResult validatorResult) throws BaseException, Exception;

    private ObjectNode createResponseJsonNode(String pCode, String pDescription) {
        return createResponseJsonNode(pCode, pDescription, null);
    }

    /**
     * 验证
     *
     * @param pMsg pMsg
     * @return ValidatorResult
     * @throws BaseException BaseException
     * @throws Exception     Exception
     */
    private ValidatorResult validate(String pMsg) throws BaseException, Exception {
        return jsonSchemaValidator.validate(pMsg, jsonSchemaFileName);
    }
}
