package com.digiwin.athena.apimgmt.dao.impl.mybatis;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.apimgmt.apiservice.ApiMgmtProjectService;
import com.digiwin.athena.apimgmt.constants.ApimgmtConstant;
import com.digiwin.athena.apimgmt.dao.ApiMgmtProjectVersionRelationDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiDataNameDao;
import com.digiwin.athena.apimgmt.dao.ApiMgmtStandardApiVersionDao;
import com.digiwin.athena.apimgmt.dao.impl.mybatis.mapper.ApiMgmtStandardApiMapper;
import com.digiwin.athena.apimgmt.facade.dto.ApiMgmtApiInfoDTO;
import com.digiwin.athena.apimgmt.facade.dto.ApiMgmtApiKeyDTO;
import com.digiwin.athena.apimgmt.model.AdvanceSearchValue;
import com.digiwin.athena.apimgmt.model.StandardApi;
import com.digiwin.athena.apimgmt.model.StandardApiVersion;
import com.digiwin.athena.apimgmt.service.model.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Primary
@Slf4j
@Repository
public class ApiMgmtStandardApiDaoMybatisImpl extends GenericDaoMybatisImpl<ApiMgmtStandardApiMapper, StandardApi, Long>
        implements ApiMgmtStandardApiDao {
    private ApiMgmtProjectVersionRelationDao projectVersionRelationDao;
    private ApiMgmtStandardApiDataNameDao dataNameDao;
    private ApiMgmtStandardApiVersionDao versionDao;

    public ApiMgmtStandardApiDaoMybatisImpl() {
        super();
    }

    @Override
    public List<StandardApi> advanceSearch(ArrayList<Map<String, String>> pConditionList, String teamType, boolean searchType, boolean tViewAllApi) {
        List<WhereCondition> conditionList = DaoConverter.INSTANCE.toWhereCondition(pConditionList);

        String tenantId = pConditionList.stream()
                .map(v -> v.get(ApimgmtConstant.TENANT))
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(null);

        return baseMapper.advanceSearch(conditionList, tenantId, teamType, searchType, tViewAllApi);
    }

    @Override
    public List<StandardApi> getByNameAndTenantId(String pName, String pTenantId) {
        LambdaQueryWrapper<StandardApi> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(StandardApi::getName, pName)
                .eq(StandardApi::getTenantId, pTenantId);
        return baseMapper.selectList(wrapper);
    }

    @Override
    public List<StandardApi> getListByNameAndBranchAndTenantId(String pName, String branch, String pTenantId) {
        LambdaQueryWrapper<StandardApi> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(StandardApi::getName, pName)
                .eq(StandardApi::getBranch, branch)
                .eq(StandardApi::getTenantId, pTenantId);
        return baseMapper.selectList(wrapper);
    }

    @Override
    public StandardApi getByNameAndBranchAndTenantId(String pName, String branch, String pTenantId) {
        LambdaQueryWrapper<StandardApi> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(StandardApi::getName, pName)
                .eq(StandardApi::getBranch, branch)
                .eq(StandardApi::getTenantId, pTenantId)
                .last("LIMIT 1");
        return baseMapper.selectOne(wrapper);
    }


    @Override
    public List<ApiMgmtApiInfoDTO> queryByKey(List<ApiMgmtApiKeyDTO> apiKeys) {
        if (CollUtil.isEmpty(apiKeys)) {
            return CollUtil.newArrayList();
        }

        return baseMapper.queryByKey(apiKeys);
    }

    @Override
    public Page<StandardApiVersion> advanceApiSearch(List<List<AdvanceSearchValue>> pList, String teamType,
                                                     String tenantId, boolean searchType, boolean tViewAllApi,
                                                     Integer pageNum, Integer pageSize) {
        return advanceApiSearch(pList, teamType, tenantId, searchType, tViewAllApi, pageNum, pageSize, false);
    }

    @Override
    public Page<StandardApiVersion> advanceApiSearch(List<List<AdvanceSearchValue>> pList, String teamType,
                                                     String tenantId, boolean searchType, boolean tViewAllApi,
                                                     Integer pageNum, Integer pageSize, boolean queryDataName) {
        Page<StandardApiVersion> page = new Page<>();
        page.setPageNum(pageNum);
        page.setPageSize(pageSize);

        List<WhereCondition> conditions = DaoConverter.INSTANCE.convertToWhereCondition(pList);


        MpPage<?> mpPage = new MpPage<>(pageNum, pageSize);
        List<Long> versionIds = baseMapper.advanceApiSearchVersionIds(conditions, teamType,
                tenantId, searchType, tViewAllApi, mpPage);

        page.setPageTotal(mpPage.getTotal());
        if (versionIds != null && !versionIds.isEmpty()) {
            List<StandardApiVersion> versions = baseMapper.selectVersionsByIds(versionIds);
            fillApi(versions);
            if (queryDataName) {
                dataNameDao.fillStandardApiDataNames(versions);
            }
            projectVersionRelationDao.fillRelationToVersion(versions);
            page.setList(versions);
        }

        return page;
    }

    @Override
    public List<Long> advanceStandardApiCountSearch(List<List<AdvanceSearchValue>> pList, String teamType,
                                                    String pTenantId, boolean tViewAllApi) {
        List<WhereCondition> conditions = DaoConverter.INSTANCE.convertToWhereCondition(pList);
        return baseMapper.advanceStandardApiCountSearch(conditions, teamType, pTenantId, false, tViewAllApi);
    }

    @Override
    @Transactional
    public StandardApi save(StandardApi pStandardApi) {
        return save(pStandardApi, false);
    }

    @Override
    @Transactional
    public StandardApi save(StandardApi pStandardApi, boolean cleanProjectRelation) {
        String tApiName = pStandardApi.getName();
        String tTenantId = pStandardApi.getTenantId();
        String branch = pStandardApi.getBranch();
        StandardApi existingApi = getByNameAndBranchAndTenantId(tApiName, branch, tTenantId);

        if (existingApi != null) {
            pStandardApi.setId(existingApi.getId());
            pStandardApi.setMsgMdata(existingApi.getMsgMdata());

            if (cleanProjectRelation) {
                projectVersionRelationDao.deleteByApiId(pStandardApi.getId());
            }
        }

        for (StandardApiVersion tVersion : pStandardApi.getStandardApiVersions()) {
            tVersion.setApprovedTime(LocalDateTime.now());
        }
        pStandardApi.setBuildTime(LocalDateTime.now());

        if (pStandardApi.getId() == null) {
            baseMapper.insert(pStandardApi);
        } else {
            baseMapper.updateById(pStandardApi);
        }

        versionDao.batchSave(pStandardApi.getStandardApiVersions());
        projectVersionRelationDao.batchSave(pStandardApi.getProjectVersionRelations());

        log.info("儲存API. API名稱: {}", pStandardApi.getName());
        return pStandardApi;
    }

    @Override
    public void delete(Long pId) {
        baseMapper.deleteById(pId);
    }


    @Override
    public void fillApi(List<StandardApiVersion> versions) {
        if (CollUtil.isEmpty(versions)) {
            return;
        }

        CollUtil.split(versions, 500).forEach(subList -> {
            List<StandardApi> standardApis = baseMapper.selectBatchIds(
                    subList.stream().map(StandardApiVersion::getApiId).distinct().collect(Collectors.toList())
            );

            Map<Long, StandardApi> apiIdMap = CollUtil.toMap(standardApis, MapUtil.newHashMap(standardApis.size()),
                    StandardApi::getId);
            subList.forEach(ver -> ver.setStandardApi(apiIdMap.get(ver.getApiId())));
        });
    }

    @Autowired
    public void setProjectVersionRelationDao(ApiMgmtProjectVersionRelationDao projectVersionRelationDao) {
        this.projectVersionRelationDao = projectVersionRelationDao;
    }

    @Autowired
    public void setDataNameDao(ApiMgmtStandardApiDataNameDao dataNameDao) {
        this.dataNameDao = dataNameDao;
    }

    @Autowired
    @Lazy
    public void setVersionDao(ApiMgmtStandardApiVersionDao versionDao) {
        this.versionDao = versionDao;
    }

    @Autowired
    @Lazy
    public void setProjectService(ApiMgmtProjectService projectService) {
        DaoConverter.projectService = projectService;
    }

    public static class DaoConverter {
        private static final Map<String, WhereCondition> sqlWhereMapping = new HashMap<>();
        public static DaoConverter INSTANCE = new DaoConverter();
        protected static ApiMgmtProjectService projectService;

        static {
            WhereCondition projectCodeCondition = WhereCondition.of("pvr.project_id", WhereCondition.Type.STRING);
            projectCodeCondition.setValMapping(projectCode -> projectService.getProjectIdByCodeOrNull(StrUtil.toStringOrNull(projectCode)));

            sqlWhereMapping.put("apiName", WhereCondition.of("stdapi.name", WhereCondition.Type.STRING));
            sqlWhereMapping.put("apiType", WhereCondition.of("stdapi.api_type", WhereCondition.Type.STRING));
            sqlWhereMapping.put("projectId", WhereCondition.of("pvr.project_id", WhereCondition.Type.STRING));
            sqlWhereMapping.put("projectVersionId", WhereCondition.of("pvr.project_version_id", WhereCondition.Type.STRING));
            sqlWhereMapping.put("projectCode", projectCodeCondition);
            sqlWhereMapping.put("apiStatus", WhereCondition.of("astat.id", WhereCondition.Type.STRING));
            sqlWhereMapping.put("apiBranch", WhereCondition.of("stdapi.branch", WhereCondition.Type.STRING));
            sqlWhereMapping.put("branch", WhereCondition.of("stdapi.branch", WhereCondition.Type.STRING));
            sqlWhereMapping.put("apiTag", WhereCondition.of("stdapi.tag", WhereCondition.Type.MULTI_LANG));
            WhereCondition dataName = WhereCondition.of("stdapidataname.data_name", WhereCondition.Type.STRING);
            dataName.setType(WhereCondition.Type.IGNORE);
            sqlWhereMapping.put("apiDataName", dataName);
            sqlWhereMapping.put("apiDescription", WhereCondition.of("stdapi.description", WhereCondition.Type.MULTI_LANG));
            sqlWhereMapping.put("apiRemark", WhereCondition.of("stdapi.remark", WhereCondition.Type.MULTI_LANG));
            sqlWhereMapping.put("apiSyncType", WhereCondition.of("stdapi.sync_type", WhereCondition.Type.STRING));
            sqlWhereMapping.put("apiCategory", WhereCondition.of("stdapi.category", WhereCondition.Type.STRING));
            sqlWhereMapping.put("provider", WhereCondition.of("stdapi.provider", WhereCondition.Type.STRING));
            sqlWhereMapping.put("requester", WhereCondition.of("stdapi.requester", WhereCondition.Type.STRING));
            sqlWhereMapping.put("tenantId", WhereCondition.of("stdapi.tenant_id", WhereCondition.Type.STRING, true));
            sqlWhereMapping.put("apiBuildTime", WhereCondition.of("stdapi.build_time", WhereCondition.Type.DATE));
            sqlWhereMapping.put("userId", WhereCondition.of("stdapiver.user_id", WhereCondition.Type.STRING));
        }

        public List<WhereCondition> convertToWhereCondition(List<List<AdvanceSearchValue>> pList) {
            // 转换查询条件List<List>中每个子List为一各条件组，使用第一个元素的AND|OR作为整组的条件
            return pList.stream().flatMap(segmentList -> {
                List<WhereCondition> child = segmentList.stream()
                        .map(v -> WhereCondition.from(sqlWhereMapping.get(v.getField()), v))
                        .filter(Objects::nonNull).collect(Collectors.toList());

                if (child.isEmpty()) {
                    return Stream.empty();
                }

                return Stream.of(WhereCondition.segment(child.get(0).isAnd(), child));
            }).collect(Collectors.toList());
        }

        public List<WhereCondition> toWhereCondition(ArrayList<Map<String, String>> list) {
            return list.stream()
                    // 应该反转map key遍历，便利inMap的key，次数更少
                    .flatMap(inMap -> sqlWhereMapping.keySet().stream()
                            .map(field -> {
                                String val = inMap.get(field);
                                if (val == null) {
                                    return null;
                                }

                                String ops = inMap.get(field + "_ops");
                                if ("provider".equals(field)) {
                                    ops = ops == null ? WhereCondition.Operator.LIKE.getOp() : ops;
                                }

                                AdvanceSearchValue searchValue = new AdvanceSearchValue(val,
                                        ops,
                                        ApimgmtConstant.MARK_AND,
                                        field,
                                        0);
                                return WhereCondition.from(
                                        sqlWhereMapping.get(searchValue.getField()),
                                        searchValue);
                            })
                            .filter(Objects::nonNull)
                    )
                    .collect(Collectors.toList());
        }

    }
}
