package com.digiwin.athena.apimgmt.facade;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.apimgmt.facade.dto.ApiMgmtApiInfoDTO;
import com.digiwin.athena.apimgmt.facade.dto.ApiMgmtApiKeyDTO;
import com.digiwin.athena.apimgmt.services.ApiMgmtStandardApiService;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
@Service
@RequiredArgsConstructor
public class ApiInfoFacadeService implements IApiInfoFacadeService {

    private final ApiMgmtStandardApiService standardApiService;

    @Override
    public List<ApiMgmtApiInfoDTO> queryByKey(List<ApiMgmtApiKeyDTO> apiKeys) {
        return CollUtil.split(apiKeys, 100).stream()
                .flatMap(subKeys -> {
                    if (CollUtil.isEmpty(subKeys)) {
                        return Stream.empty();
                    }

                    return standardApiService.queryByKey(subKeys).stream();
                })
                .collect(Collectors.toList());
    }


    @Override
    public Map<ApiMgmtApiKeyDTO, ApiMgmtApiInfoDTO> queryApiInfoMapByKeys(List<ApiMgmtApiKeyDTO> apiKeys, boolean userTenant) {
        if (CollUtil.isEmpty(apiKeys)) {
            return Collections.emptyMap();
        }

        if (!userTenant) {
            apiKeys.forEach(v -> v.setTenantId(null));
        }

        List<ApiMgmtApiInfoDTO> apiInfoList = queryByKey(apiKeys);
        return transToKeyMap(apiInfoList, userTenant);
    }

    protected @NonNull Map<ApiMgmtApiKeyDTO, ApiMgmtApiInfoDTO> transToKeyMap(List<ApiMgmtApiInfoDTO> apiInfoList, boolean useTenant) {
        if (CollUtil.isEmpty(apiInfoList)) {
            return Collections.emptyMap();
        }

        Map<ApiMgmtApiKeyDTO, ApiMgmtApiInfoDTO> resultMap = MapUtil.newHashMap(apiInfoList.size() * 2);
        for (ApiMgmtApiInfoDTO apiInfo : apiInfoList) {
            String apiName = apiInfo.getName();
            String tenantId = useTenant ? apiInfo.getTenantId() : null;
            String version = apiInfo.getVersion();

            ApiMgmtApiKeyDTO keyWithVersion = ApiMgmtApiKeyDTO.of(apiName, tenantId, version);
            resultMap.put(keyWithVersion, apiInfo);

            ApiMgmtApiKeyDTO keyWithoutVersion = ApiMgmtApiKeyDTO.of(apiName, tenantId);
            resultMap.put(keyWithoutVersion, apiInfo);
        }

        return resultMap;
    }

    @Override
    public String getProjectCode(Map<ApiMgmtApiKeyDTO, ApiMgmtApiInfoDTO> apiInfoMap, ApiMgmtApiKeyDTO key) {
        key.setTenantId(null);
        ApiMgmtApiInfoDTO apiInfo = apiInfoMap.get(key);

        if (apiInfo == null) {
            ApiMgmtApiKeyDTO keyWithoutVersion = ApiMgmtApiKeyDTO.ofName(key.getApiName());
            apiInfo = apiInfoMap.get(keyWithoutVersion);
        }


        return apiInfo != null ? apiInfo.getProjectCode() : null;
    }
}
