package com.digiwin.athena.apimgmt.infra.spring.http;

import cn.hutool.core.io.IoUtil;
import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.apimgmt.common.util.JsonUtil;
import com.digiwin.athena.apimgmt.infra.http.*;
import com.digiwin.athena.apimgmt.infra.http.HttpRequest;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.InputStreamResource;
import org.springframework.http.*;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.ResponseExtractor;
import org.springframework.web.client.RestClientResponseException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.DefaultUriBuilderFactory;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Spring HTTP客户端实现
 * 实现IHttpClient接口，使用RestTemplate发送HTTP请求
 */
@Slf4j
public class SpringApiMgmtHttpClientWrapper implements ApiMgmtHttpClientWrapper {

    private final RestTemplate restTemplate;
    private final Map<Integer, RestTemplate> CACHE = new ConcurrentHashMap<>();

    public SpringApiMgmtHttpClientWrapper(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    public static RestTemplate copyWithTimeout(RestTemplate source, int timeout) {
        RestTemplate newTemplate = new RestTemplate();

        // 复制配置（拦截器、转换器、错误处理等）
        newTemplate.setInterceptors(source.getInterceptors());
        newTemplate.setMessageConverters(source.getMessageConverters());
        newTemplate.setErrorHandler(source.getErrorHandler());
        newTemplate.setUriTemplateHandler(source.getUriTemplateHandler());
        newTemplate.setRequestFactory(source.getRequestFactory());
        if (source.getUriTemplateHandler() instanceof DefaultUriBuilderFactory) {
            newTemplate.setDefaultUriVariables(((DefaultUriBuilderFactory) source.getUriTemplateHandler()).getDefaultUriVariables());
        }

        // 替换工厂（根据原来的类型判断）
        ClientHttpRequestFactory factory = source.getRequestFactory();

        if (factory instanceof SimpleClientHttpRequestFactory) {
            SimpleClientHttpRequestFactory simple = (SimpleClientHttpRequestFactory) factory;
            simple.setConnectTimeout(timeout);
            simple.setReadTimeout(timeout);
            newTemplate.setRequestFactory(simple);
        } else if (factory instanceof HttpComponentsClientHttpRequestFactory) {
            HttpComponentsClientHttpRequestFactory http = (HttpComponentsClientHttpRequestFactory) factory;
            http.setConnectTimeout(timeout);
            http.setReadTimeout(timeout);
            newTemplate.setRequestFactory(http);
        } else {
            // 兜底方案，不支持的实现类型直接替换新建
            SimpleClientHttpRequestFactory fallback = new SimpleClientHttpRequestFactory();
            fallback.setConnectTimeout(timeout);
            fallback.setReadTimeout(timeout);
            newTemplate.setRequestFactory(fallback);
        }

        return newTemplate;
    }

    private static ResponseExtractor<HttpResponse> transformResp() {
        return resp -> HttpResponse.builder()
                .statusCode(resp.getStatusCode().value())
                .header(resp.getHeaders().toSingleValueMap())
                .is(resp.getBody())
                .bodyConverter(() -> {
                    try {
                        MediaType contentType = resp.getHeaders().getContentType();
                        return IoUtil.read(resp.getBody(),
                                contentType != null ? contentType.getCharset() : StandardCharsets.UTF_8);
                    } catch (IOException e) {
                        throw new SpringHttpFailedException("读取响应流异常!", e);
                    }
                })
                .build();
    }

    @Override
    public <T> T execute(HttpRequest request, Class<T> responseType) {
        return execute(request, responseType, null);
    }

    @Override
    public <T> T execute(HttpRequest request, Class<T> responseType, RequestOption options) {
        RestTemplate template = setOptions(options);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        MapUtil.emptyIfNull(request.getHeaders()).forEach(headers::add);

        Object body = request.getBody();
        String bodyStr = null;
        if (body != null) {
            if (body instanceof String) {
                bodyStr = (String) body;
            } else {
                bodyStr = JsonUtil.serialize(body);
            }
        }

        HttpEntity<String> httpEntity = new HttpEntity<>(bodyStr, headers);
        HttpMethod method = HttpMethod.valueOf(request.getMethod());
        if (method == null) {
            throw new IllegalArgumentException("Unsupported HTTP method: " + request.getMethod());
        }

        return execute(request, responseType, method, httpEntity, template);
    }

    protected RestTemplate setOptions(RequestOption options) {
        RestTemplate template = restTemplate;

        if (options != null && options.getTimeout() != null) {
            template = CACHE.computeIfAbsent(options.getTimeout(),
                    k -> copyWithTimeout(restTemplate, options.getTimeout()));
        }

        return template;
    }

    @Override
    public <T> T uploadFile(HttpRequest request, MultipartFile file, Class<T> responseType) {
        return uploadFile(request, file, responseType, null);
    }

    @Override
    public <T> T uploadFile(HttpRequest request, MultipartFile file, Class<T> responseType, RequestOption options) {
        RestTemplate template = setOptions(options);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);
        MapUtil.emptyIfNull(request.getHeaders()).forEach(headers::add);


        MultiValueMap<String, Object> reqBody = new LinkedMultiValueMap<>();
        getBodyMap(request).forEach((k, v) -> reqBody.add(k, JsonUtil.serialize(v)));
        reqBody.add(file.getName(), buildFileEntity(file));

        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<>(reqBody, headers);

        HttpMethod method = HttpMethod.valueOf(request.getMethod());
        if (method == null) {
            throw new IllegalArgumentException("Unsupported HTTP method: " + request.getMethod());
        }

        return execute(request, responseType, method, httpEntity, template);
    }

    private <T> T execute(HttpRequest request, Class<T> responseType, HttpMethod method,
                          HttpEntity<?> httpEntity, RestTemplate template) {
        try {
            if (responseType == null || responseType == HttpResponse.class) {
                //noinspection unchecked
                return (T) restTemplate.execute(request.getUrl(),
                        method,
                        restTemplate.httpEntityCallback(httpEntity),
                        transformResp()
                );
            }


            ResponseEntity<T> response = template.exchange(
                    request.getUrl(),
                    method,
                    httpEntity,
                    responseType
            );

            return response.getBody();
        } catch (RestClientResponseException e) {
            throw new SpringHttpFailedException(e);
        } catch (Exception e) {
            throw new SpringHttpFailedException("HTTP file upload failed: " + e.getMessage(), e);
        }
    }

    protected HttpEntity<InputStreamResource> buildFileEntity(MultipartFile file) {
        InputStreamResource fileResource = new InputStreamResource(file.getInputStream()) {
            @Override
            public String getFilename() {
                return file.getFileName();
            }

            @Override
            public long contentLength() {
                return file.getSize();
            }
        };

        HttpHeaders fileHeader = new HttpHeaders();
        fileHeader.setContentType(MediaType.valueOf(file.getContentType()));

        return new HttpEntity<>(fileResource, fileHeader);
    }
}
