package com.digiwin.athena.apimgmt.infra.spring.http;

import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpRequest;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.lang.NonNull;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.Assert;
import org.springframework.web.client.DefaultResponseErrorHandler;

import java.io.IOException;
import java.net.URI;

@Slf4j
public class HttpRetryInterceptor implements ClientHttpRequestInterceptor {

    private final RetryProp retryProp;
    private final AntPathMatcher pathMatcher = new AntPathMatcher();
    private final DefaultResponseErrorHandler errorHandler;

    public HttpRetryInterceptor(RetryProp retryProp) {
        this.retryProp = retryProp;
        this.errorHandler = new DefaultResponseErrorHandler();
    }


    @Override
    public @NonNull ClientHttpResponse intercept(final HttpRequest request, final @NonNull byte[] body,
                                        final @NonNull ClientHttpRequestExecution execution) throws IOException {
        final URI originalUri = request.getURI();
        final String serviceName = originalUri.getHost();
        Assert.state(serviceName != null, "Request URI does not contain a valid hostname: " + originalUri);

        RetryProp.Entry retry = retryProp.getHttp().stream()
                .filter(pattern -> pathMatcher.match(pattern.getPattern(), originalUri.getPath()))
                .findFirst()
                .orElse(RetryProp.HTTP_DEFAULT);

        if (!retry.isEnabled()) {
            if (log.isTraceEnabled()) {
                log.trace("Retry is disabled for {}", retry.getPattern());
            }

            return execution.execute(request, body);
        }

        RetryTemplate template = createRetryTemplate(retry);

        return template.execute(context -> {
            try (ClientHttpResponse resp = execution.execute(request, body)) {
                HttpStatusCode status = resp.getStatusCode();
                if (retry.needRetry(status.value())) {
                    errorHandler.handleError(resp);
                }

                return resp;
            } catch (IOException e) {
                throw new SpringHttpFailedException("http请求异常！", e);
            }
        });
    }

    private RetryTemplate createRetryTemplate(RetryProp.Entry retry) {
        RetryTemplate template = new RetryTemplate();

        template.setRetryPolicy(new SimpleRetryPolicy(retry.getMaxRetryCount()));
        template.setBackOffPolicy(new ExponentialBackOffPolicy());
        template.setThrowLastExceptionOnExhausted(true);

        return template;
    }
}