package com.digiwin.athena.auth.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.auth.domain.QueryAuthParam;
import com.digiwin.athena.auth.metadata.constants.AuthConstant;
import com.digiwin.athena.auth.domain.AuthBatchQuery;
import com.digiwin.athena.auth.domain.AuthQueryCondition;
import com.digiwin.athena.auth.domain.AuthSingleQuery;
import com.digiwin.athena.auth.metadata.domain.AuthorityPolicy;
import com.digiwin.athena.auth.metadata.enums.AuthModeEnum;
import com.digiwin.athena.auth.metadata.enums.CombineTypeEnum;
import com.digiwin.athena.auth.service.AuthService;
import com.digiwin.athena.auth.service.AuthApiService;
import com.digiwin.athena.auth.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
public class AuthApiServiceImpl implements AuthApiService {

    @Autowired(required = false)
    @Lazy
    private AuthService authService;

    @Autowired
    @Lazy
    private UserService userService;

    @Value("${designer.auth.mode:FORBID}")
    private String authMode;

    @Value("${designer.domain:default}")
    private String designerDomain;

    @Override
    public Map<String, Boolean> funcAuthVerifyBatch(AuthBatchQuery authBatchQuery){
        List<String> resourceIds = authBatchQuery.getResourceIds();
        String[] requiredActions = authBatchQuery.getRequiredActions();
        String resourceType = authBatchQuery.getResourceType();

        Map<String, Boolean> result = new HashMap<>();
        if (AuthModeEnum.FORBID.name().equals(authMode)) {
            //禁止使用鉴权功能
            return resourceIds.stream().collect(Collectors.toMap(Function.identity(), k -> true));
        }

        List<AuthorityPolicy> policyList;
        if (AuthModeEnum.LOCAL.name().equals(authMode)) {
            //本地鉴权
            QueryAuthParam queryAuthParam = new QueryAuthParam()
                    .setUserId(userService.getCurrentUser())
                    .setResourceType(resourceType)
                    .setResourceId(String.join(";", resourceIds));
            policyList = authService.queryFuncAuthPolicy(queryAuthParam);
        } else {
            //远程鉴权
            if (userService == null) {
                throw new RuntimeException("请实现com.digiwin.athena.auth.service.UserService接口");
            }
            QueryAuthParam queryAuthParam = new QueryAuthParam()
                    .setUserId(userService.getCurrentUser())
                    .setResourceType(resourceType)
                    .setResourceId(String.join(";", resourceIds));
            String response = HttpUtil.createPost(designerDomain + "/athena-designer/auth/queryAuthPolicy").body(JSON.toJSONString(queryAuthParam)).addHeaders(MapUtil.builder("digi-middleware-auth-user", userService.getToken()).build()).execute().body();
            JSONObject jsonObject = JSON.parseObject(response, JSONObject.class);
            Integer code = jsonObject.getInteger("code");
            if (code == 0) {
                JSONArray data = jsonObject.getJSONArray("data");
                policyList = data.toJavaList(AuthorityPolicy.class);
            } else {
                log.error("查询权限失败：" + response);
                throw new RuntimeException("查询权限失败");
            }
        }

        String tenantId = userService.getTenantId();

        resourceIds.forEach(resourceId -> {
            List<AuthorityPolicy> thePolicyList = policyList.stream()
                    .filter(policy -> policy.getResourceId() != null && (policy.getResourceId().equals(resourceId) || policy.getResourceId().equals(tenantId))).collect(Collectors.toList());
            if (CollUtil.isNotEmpty(thePolicyList)) {
                Boolean hasAuth = verifyAuthByQueryPolicy(thePolicyList, requiredActions, resourceType, tenantId);
                result.put(resourceId, hasAuth);
            } else {
                result.put(resourceId, false);
            }
        });
        return result;
    }

    @Override
    public Boolean funcAuthVerifySingle(AuthQueryCondition authQueryCondition) {
        CombineTypeEnum combineType = authQueryCondition.getCombineType();
        List<AuthSingleQuery> authSingleQueryList = authQueryCondition.getAuthSingleQueryList();
        switch (combineType) {
            case AND:
                for (AuthSingleQuery authSingleQuery : authSingleQueryList) {
                    AuthBatchQuery authBatchQuery = new AuthBatchQuery().setResourceIds(Arrays.asList(authSingleQuery.getResourceId()))
                            .setRequiredActions(authSingleQuery.getRequiredActions())
                            .setResourceType(authSingleQuery.getResourceType());
                    Map<String, Boolean> authRe = this.funcAuthVerifyBatch(authBatchQuery);
                    if (!authRe.get(authSingleQuery.getResourceId())) {
                        // 存在一个无权限的情况 直接false
                        return false;
                    }
                }
                return true;
            case OR:
                for (AuthSingleQuery authSingleQuery : authSingleQueryList) {
                    AuthBatchQuery authBatchQuery = new AuthBatchQuery().setResourceIds(Arrays.asList(authSingleQuery.getResourceId()))
                            .setRequiredActions(authSingleQuery.getRequiredActions())
                            .setResourceType(authSingleQuery.getResourceType());
                    Map<String, Boolean> authRe = this.funcAuthVerifyBatch(authBatchQuery);
                    if (authRe.get(authSingleQuery.getResourceId())) {
                        // 存在一个有权限的情况 直接true
                        return true;
                    }
                }
                return false;
            default:
                log.error("权限入参格式不对：" + JSON.toJSONString(authQueryCondition));
                return false;
        }
    }

    // 通过查询的拒绝和允许action判断权限
    @Override
    public Boolean verifyAuthByQueryPolicy(List<AuthorityPolicy> policyList, String[] requiredActions, String resourceType, String tenantId){
        // 租户有权限就直接返回
        List<AuthorityPolicy> tenantPolicyList = policyList.stream().filter(item -> tenantId.equals(item.getResourceId())).collect(Collectors.toList());
        Boolean tenantAuth = generalAuthLogic(tenantPolicyList, requiredActions, resourceType);
        if(tenantAuth){
            return true;
        }
        // 没有租户权限再判断其他类型权限
        List<AuthorityPolicy> otherPolicyList = policyList.stream().filter(item -> !tenantId.equals(item.getResourceId())).collect(Collectors.toList());
        Boolean otherAuth = generalAuthLogic(otherPolicyList, requiredActions, resourceType);
        if(otherAuth){
            return true;
        }

        return false;
    }

    private Boolean generalAuthLogic(List<AuthorityPolicy> policyList, String[] requiredActions, String resourceType){
        List<AuthorityPolicy> denyPolicyList = policyList.stream().filter(item -> AuthConstant.Effect.DENY.equals(item.getEffect())).collect(Collectors.toList());
        boolean presentDeny = denyPolicyList.stream().filter(item -> item.getAction().stream().anyMatch(action -> Arrays.stream(requiredActions).anyMatch(requiredAction -> (resourceType + requiredAction).equals(action)))).findAny().isPresent();
        if (presentDeny) {
            return false;
        }

        List<AuthorityPolicy> allowPolicyList = policyList.stream().filter(item -> AuthConstant.Effect.ALLOW.equals(item.getEffect())).collect(Collectors.toList());
        if (allowPolicyList.stream().filter(item -> item.getAction().stream().anyMatch(action -> action.endsWith(AuthConstant.Action.ALL))).findAny().isPresent()) {
            return true;
        }

        for (AuthorityPolicy authorityPolicy : allowPolicyList) {
            List<String> action = authorityPolicy.getAction();
            for (String item : action) {
                if (Arrays.stream(requiredActions).anyMatch(requiredAction -> (resourceType + requiredAction).equals(item))) {
                    return true;
                }
            }
        }

        return false;
    }

}
