package com.digiwin.athena.auth.aspect;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.digiwin.athena.auth.annotation.DataAuth;
import com.digiwin.athena.auth.context.DataAuthContext;
import com.digiwin.athena.auth.domain.QueryAuthParam;
import com.digiwin.athena.auth.metadata.constants.AuthConstant;
import com.digiwin.athena.auth.metadata.domain.AuthorityPolicy;
import com.digiwin.athena.auth.metadata.domain.ResourceCondition;
import com.digiwin.athena.auth.metadata.enums.AuthModeEnum;
import com.digiwin.athena.auth.service.AuthService;
import com.digiwin.athena.auth.service.DataCountService;
import com.digiwin.athena.auth.service.UserService;
import com.digiwin.athena.auth.util.AuthHelper;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Aspect
@Component
public class DataAuthAspect {

    @Autowired(required = false)
    @Lazy
    private AuthService authService;

    @Autowired(required = false)
    @Lazy
    private DataCountService dataCountService;

    @Autowired
    @Lazy
    private UserService userService;

    @Autowired
    @Lazy
    private AuthHelper authHelper;

    @Value("${designer.auth.mode:FORBID}")
    private String authMode;

    @Value("${designer.auth.dbType:MONGO}")
    private String authDbType;

    @Value("${designer.domain:default}")
    private String designerDomain;

    @Pointcut("execution(* org.springframework.data.mongodb.core.MongoTemplate.find(..))")
    public void executeFind() {
    }

    @Before("@annotation(dataAuth)")
    public void dataAuthCheck(JoinPoint joinPoint, DataAuth dataAuth) {
        if (AuthModeEnum.FORBID.name().equals(authMode)) {
            //禁止使用鉴权功能
            return;
        }
        if (!authDbType.equals(AuthConstant.DBType.MONGO)) {
            throw new RuntimeException("仅支持MongoDB数据源使用数据权限!");
        }

        String resourceType = dataAuth.resourceType();
        String resourceId = dataAuth.resourceId();
        Object[] args = joinPoint.getArgs();

        if (resourceId.startsWith(AuthHelper.JSON_PREFIX)) {
            resourceId = authHelper.getJSONValue(resourceId, args);
        } else {
            resourceId = authHelper.getVarValue(args, resourceId);
        }

        List<AuthorityPolicy> policyList;
        if (AuthModeEnum.LOCAL.name().equals(authMode)) {
            policyList = authService.queryDataAuthPolicy(userService.getCurrentUser(), resourceType, resourceId);
        } else {
            if (userService == null) {
                throw new RuntimeException("请实现com.digiwin.athena.auth.service.UserService接口");
            }
            QueryAuthParam queryAuthParam = new QueryAuthParam()
                    .setUserId(userService.getCurrentUser())
                    .setResourceId(resourceId)
                    .setResourceType(resourceType);
            String response = HttpUtil.createPost(designerDomain + "/athena-designer/auth/queryDataAuthPolicy").body(JSON.toJSONString(queryAuthParam)).addHeaders(MapUtil.builder("digi-middleware-auth-user", userService.getToken()).build()).execute().body();
            JSONObject jsonObject = JSON.parseObject(response, JSONObject.class);
            Integer code = jsonObject.getInteger("code");
            if (code == 0) {
                JSONArray data = jsonObject.getJSONArray("data");
                policyList = data.toJavaList(AuthorityPolicy.class);
            } else {
                throw new RuntimeException("查询权限失败");
            }
        }

        if (CollUtil.isEmpty(policyList)) {
            // 拼接始终为false的条件
            DataAuthContext.setMongoCondition(MapUtil.builder(resourceType, Criteria.where("1").is("2")).build());
            return;
        }

        Criteria criteria = new Criteria();
        Boolean needCondition = true;
        List<Criteria> queryCondition = new ArrayList<>();
        for (AuthorityPolicy authorityPolicy : policyList) {
            List<ResourceCondition> conditions = authorityPolicy.getConditions();
            if (CollUtil.isEmpty(conditions)) {
                needCondition = false;
                break;
            }
            for (ResourceCondition condition : conditions) {
                queryCondition.add(Criteria.where(condition.getField()).is(condition.getValue()));
            }
        }
        if (!needCondition) {
            criteria = null;
        } else {
            criteria.orOperator(ArrayUtil.toArray(queryCondition, Criteria.class));
        }
        DataAuthContext.setMongoCondition(MapUtil.builder(resourceType, criteria).build());

        boolean throwException = dataAuth.throwException();
        if (throwException && needCondition) {
            String pkValue = dataAuth.pkValue();
            pkValue = authHelper.getJSONValue(pkValue, args);
            Long count = dataCountService.count(resourceType, pkValue, criteria);
            if (count <= 0) {
                throw new RuntimeException("无权访问改接口!");
            }
        }

    }

    @Around("executeFind()")
    public Object aroundExecuteFind(ProceedingJoinPoint joinPoint) throws Throwable {
        Object[] args = joinPoint.getArgs();
        int length = args.length;
        if (length == 3) {
            String collection = String.valueOf(args[2]);
            Query query = (Query) args[0];
            Map<String, Criteria> mongoCondition = DataAuthContext.getMongoCondition();
            if (mongoCondition == null) {
                DataAuthContext.removeMongoCondition();
                return joinPoint.proceed();
            }
            Criteria criteria = mongoCondition.get(collection);
            if (criteria != null) {
                query.addCriteria(criteria);
            }
            args[0] = query;
            DataAuthContext.removeMongoCondition();
            return joinPoint.proceed(args);
        }
        DataAuthContext.removeMongoCondition();
        return joinPoint.proceed();

    }
}
