package com.digiwin.athena.auth.aspect;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.auth.annotation.FuncAuth;
import com.digiwin.athena.auth.domain.QueryAuthParam;
import com.digiwin.athena.auth.metadata.domain.AuthorityPolicy;
import com.digiwin.athena.auth.metadata.enums.AuthModeEnum;
import com.digiwin.athena.auth.metadata.enums.PermissionPolicyEnum;
import com.digiwin.athena.auth.service.AuthService;
import com.digiwin.athena.auth.service.AuthApiService;
import com.digiwin.athena.auth.service.MongoDataService;
import com.digiwin.athena.auth.service.UserService;
import com.digiwin.athena.auth.util.AuthHelper;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

@Aspect
@Component
@Slf4j
public class FuncAuthAspect {

    @Autowired(required = false)
    @Lazy
    private AuthService authService;

    @Autowired(required = false)
    @Lazy
    private MongoDataService mongoDataService;

    @Autowired
    @Lazy
    private UserService userService;

    @Autowired
    @Lazy
    private AuthHelper authHelper;

    @Value("${designer.auth.mode:FORBID}")
    private String authMode;

    @Value("${designer.domain:default}")
    private String designerDomain;

    @Autowired
    private AuthApiService authApiService;

    @Before("@annotation(funcAuth)")
    public void funcAuthCheck(JoinPoint joinPoint, FuncAuth funcAuth) {
        if (AuthModeEnum.FORBID.name().equals(authMode)) {
            //禁止使用鉴权功能
            return;
        }
        String resourceId = funcAuth.resourceId();
        String[] requiredActions = funcAuth.actions();
        Object[] args = joinPoint.getArgs();
        String resourceType = getResourceType(funcAuth, args);

        if (StrUtil.isNumeric(resourceId)) {
            //根据参数index获取resourceId
            if (Arrays.stream(args).filter(item -> item != null).collect(Collectors.toList()).size() > Convert.toInt(resourceId)) {
                resourceId = String.valueOf(args[Convert.toInt(resourceId)]);
            } else {
                resourceId = null;
            }
        } else if (resourceId.startsWith(AuthHelper.VAR_PREFIX)) {
            resourceId = authHelper.getVarValue(args, resourceId);
        } else if (resourceId.startsWith(AuthHelper.PRIMARY_PREFIX)) {
            resourceId = getPkValue(args, resourceId, funcAuth.primaryKey());
        } else if (resourceId.startsWith(AuthHelper.JSON_PREFIX)) {
            resourceId = authHelper.getJSONValue(resourceId, args);
        }

        List<AuthorityPolicy> policyList;
        if (AuthModeEnum.LOCAL.name().equals(authMode)) {
            //本地鉴权
            QueryAuthParam queryAuthParam = new QueryAuthParam()
                    .setUserId(userService.getCurrentUser())
                    .setResourceType(resourceType)
                    .setResourceId(resourceId);
            policyList = authService.queryFuncAuthPolicy(queryAuthParam);
        } else {
            //远程鉴权
            if (userService == null) {
                throw new RuntimeException("请实现com.digiwin.athena.auth.service.UserService接口");
            }
            QueryAuthParam queryAuthParam = new QueryAuthParam()
                    .setUserId(userService.getCurrentUser())
                    .setResourceType(resourceType)
                    .setResourceId(resourceId);
            String response = HttpUtil.createPost(designerDomain + "/athena-designer/auth/queryAuthPolicy").body(JSON.toJSONString(queryAuthParam)).addHeaders(MapUtil.builder("digi-middleware-auth-user", userService.getToken()).build()).execute().body();
            JSONObject jsonObject = JSON.parseObject(response, JSONObject.class);
            Integer code = jsonObject.getInteger("code");
            if (code == 0) {
                JSONArray data = jsonObject.getJSONArray("data");
                policyList = data.toJavaList(AuthorityPolicy.class);
            } else {
                log.error("查询权限失败：" + response);
                throw new RuntimeException("查询权限失败");
            }

        }
        String tenantId = userService.getTenantId();
        // 校验具体权限
        Boolean verified = authApiService.verifyAuthByQueryPolicy(policyList, requiredActions, resourceType, tenantId);
        if(!verified){
            throw new RuntimeException("无权访问该接口!");
        }
    }

    private String getEntityPkValue(String primaryKey, Object[] args) {
        String primaryKeyValue;
        if (primaryKey.startsWith(AuthHelper.VAR_PREFIX)) {
            primaryKeyValue = authHelper.getVarValue(args, primaryKey);
        } else if (primaryKey.startsWith(AuthHelper.JSON_PREFIX)) {
            primaryKeyValue = authHelper.getJSONValue(primaryKey, args);
        } else {
            primaryKeyValue = String.valueOf(args[Convert.toInt(primaryKey)]);
        }
        return primaryKeyValue;
    }

    private String getResourceType(FuncAuth funcAuth, Object[] args) {
        String resourceType = funcAuth.resourceType();
        if (resourceType.startsWith(AuthHelper.VAR_PREFIX)) {
            String var = resourceType.substring(resourceType.indexOf(":") + 1, resourceType.length());
            String[] split = var.split(AuthHelper.SEPARATOR);
            String clazz = split[0];
            String field = split[1];
            for (Object arg : args) {
                if (arg.getClass().getSimpleName().equals(clazz)) {
                    Object fieldValue = ReflectUtil.getFieldValue(arg, field);
                    resourceType = PermissionPolicyEnum.getResourceTypeByRole(String.valueOf(fieldValue));
                    break;
                }
            }
        } else if (StrUtil.isNumeric(resourceType)) {
            resourceType = String.valueOf(args[Convert.toInt(resourceType)]);
        } else if (resourceType.startsWith(AuthHelper.JSON_PREFIX)) {
            resourceType = authHelper.getJSONValue(resourceType, args);
            resourceType = PermissionPolicyEnum.getResourceTypeByRole(resourceType);
        }
        return resourceType;
    }

    private String getPkValue(Object[] args, String var, String primaryKey) {
        if (mongoDataService == null) {
            throw new RuntimeException("请实现com.digiwin.athena.auth.service.MongoDataService接口!");
        }
        String primaryKeyValue = getEntityPkValue(primaryKey, args);
        var = var.substring(var.indexOf(":") + 1, var.length());
        String[] split = var.split(AuthHelper.OR_OPERATOR);
        String result = null;
        for (String collectionInfo : split) {
            String[] arr = collectionInfo.split(AuthHelper.SEPARATOR);
            String collection = arr[0];
            String primaryKeyField = arr[1];
            String applicationField = arr[2];
            JSONObject jsonObject = mongoDataService.queryOne(collection, primaryKeyField, primaryKeyValue);
            if (jsonObject != null) {
                result = jsonObject.getString(applicationField);
                break;
            }
        }
        if (StrUtil.isBlank(result)) {
            throw new RuntimeException("找不到该实体对应的应用!");
        }
        return result;
    }
}
