package com.digiwin.athena.appcore.auth.filter;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.DwAthenaAuthProperties;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.constant.ErrorTypeEnum;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.SecurityTokenCommonUtils;
import com.digiwin.athena.appcore.util.TokenCache;
import com.digiwin.athena.appcore.web.AntPathRequestMatcher;
import com.digiwin.athena.iam.sdk.manager.IamAuthManager;
import com.digiwin.athena.iam.sdk.manager.IamManager;
import com.digiwin.athena.iam.sdk.meta.dto.response.AuthoredUserDTO;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.io.serialize.json.JaJson;
import com.jugg.agile.spring.boot.util.JaI18nUtil;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.Ordered;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;

/**
 * 从http头的Authorization 项读取userToken数据，通过IAM解析租户用户信息。 如果解析成功，将用户信息添加到请求体上
 *
 * @author chenxsa
 * @date 2019/4/12
 */
@Slf4j
public class UserTokenAuthenticationFilter extends OncePerRequestFilter implements Ordered {

    private DwAthenaAuthProperties dwAthenaAuthProperties;

    public UserTokenAuthenticationFilter(DwAthenaAuthProperties dwAthenaAuthProperties) {
        this.dwAthenaAuthProperties = dwAthenaAuthProperties;
        AntPathRequestMatcher.initRequestMatchers(dwAthenaAuthProperties);
    }

    /**
     * 重载filter，处理useToken信息
     */
    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        AppAuthContextHolder.clearContext();
        if (request.getMethod().matches(HttpMethod.OPTIONS.name())) {
            chain.doFilter(request, response);
            return;
        }
        //请求头赋值
        setCustomRequestHeader(request);

        String userToken = request.getHeader(GlobalConstant.IAM_USER_TOKEN);
        if (StringUtils.isEmpty(userToken)) {
            userToken = request.getHeader(GlobalConstant.IAM_IDENTITY_TYPE_TOKEN);
        }

        AuthoredUser authentication = null;
        AuthoredUser proxyAuthentication = null;

        // 如果存在信任链路调用，先看是否可行，如果行，则
        String securityToken = request.getHeader("security-token");
        if (!StringUtils.isEmpty(securityToken)) {
            authentication = SecurityTokenCommonUtils.getAuthoredUserBySecurityToken(securityToken);
            // 使用新版本的
            if (null != authentication && StringUtils.isEmpty(authentication.getToken())) {
                // 已有securityToken的情况下，可以不用调用api，只需要从header中获取token即可
                authentication.setToken(userToken);
            }
        }
        // 如果信任链路断裂，重新去iam做授权验证
        if (authentication == null || JaStringUtil.isSafeEmpty(authentication.getVersion())) {
            securityToken = null;
            try {
                authentication = getUserInfo(userToken);
                verifyAndExchangeIamToken(authentication);
            } catch (BusinessException ex) {
                // 如果不是白名单的请求
                if (!AntPathRequestMatcher.inWriteList(request, dwAthenaAuthProperties) && AuthCheck.isCheckAuth(request)) {
                    writeUnAuth(request, response, ex);
                    return;
                }
            }
        } else {
            verifyAndExchangeIamToken(authentication);
        }

        String proxyToken = getRequestProxyToken(request);
        if (JaStringUtil.isNotEmpty(proxyToken)) {
            try {
                proxyAuthentication = getUserInfo(proxyToken);
                if (proxyAuthentication != null
                        && JaStringUtil.isNotEmpty(authentication.getTenantId())
                        && JaStringUtil.isNotEmpty(proxyAuthentication.getTenantId())) {
                    if (authentication.getTenantId().equals(proxyAuthentication.getTenantId())) {
                        proxyAuthentication = null;
                        proxyToken = null;
                    } else {
                        request.setAttribute(GlobalConstant.PROXY_AUTH_USER, proxyAuthentication);
                        AppAuthContextHolder.getContext().setProxyAuthoredUser(proxyAuthentication);
                    }
                }
            } catch (Exception e) {
                logger.error("代理token失效:" + proxyToken, e);
            }
        }

        if (authentication != null) {
            // 如果是第一次调用，或者信任链路断裂，重新发起信任链路
            if (StringUtils.isEmpty(securityToken)) {
                if (proxyAuthentication != null) {
                    securityToken = SecurityTokenCommonUtils.generateSecurityToken("", proxyToken, proxyAuthentication, authentication);
                } else {
                    securityToken = SecurityTokenCommonUtils.generateSecurityToken("", userToken, authentication, null);
                }
            }
            // 在请求体上添加数据，可以在controller的api上直接通过
            request.setAttribute(GlobalConstant.AUTH_USER, authentication);

            // 设置自定义线程上下文
            AppAuthContextHolder.getContext().setSecurityToken(securityToken);
            AppAuthContextHolder.getContext().setAuthoredUser(authentication);

            AppAuthContextHolder.getContext().setProxyToken(proxyToken);
        }

        // 添加routerKey
        String routerKey = request.getHeader(GlobalConstant.ROUTER_KEY);
        if (StringUtils.hasText(routerKey)) {
            MDC.put(GlobalConstant.ROUTER_KEY, routerKey);
        }
        chain.doFilter(request, response);
        AppAuthContextHolder.clearContext();
    }

    /**
     * 获取头信息中的代理token
     */
    private String getRequestProxyToken(HttpServletRequest request) {
        String proxyToken = request.getHeader(GlobalConstant.IAM_IDENTITY_TYPE_PROXY_TOKEN);
        if (StringUtils.isEmpty(proxyToken)) {
            proxyToken = request.getHeader(GlobalConstant.IAM_IDENTITY_TYPE_PROXY_TOKEN3);
        }
        if (StringUtils.isEmpty(proxyToken)) {
            proxyToken = request.getHeader(GlobalConstant.IAM_IDENTITY_TYPE_PROXY_TOKEN2);
        }
        return proxyToken;
    }

    private void writeUnAuth(HttpServletRequest request, HttpServletResponse response, BusinessException ex) throws IOException {
        if (JaStringUtil.isEmpty(ex.getBizErrorCode())) {
            throw ex;
        }
        response.setStatus(HttpStatus.UNAUTHORIZED.value());
        response.setCharacterEncoding("UTF-8");
        response.setContentType("application/json");
        BaseResultDTO<Object> result = getBaseResultG(request.getRequestURL().toString(), ex);
        response.getWriter().println(JaJson.toString(result));
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }

    private void setCustomRequestHeader(HttpServletRequest httpRequest) {
        try {
            Map<String, Object> headerMap = new HashMap<>(16);
            Enumeration<String> headerNames = httpRequest.getHeaderNames();

            while (headerNames.hasMoreElements()) {
                String key = headerNames.nextElement();
                String value = httpRequest.getHeader(key);
                headerMap.put(key, value);
            }

            AppAuthContextHolder.getContext().setRequestHeader(headerMap);
        } catch (Exception e) {
            logger.warn("请求头赋值失败");
        }
    }

    public AuthoredUser getUserInfo(String userToken) {
        AuthoredUserDTO userInfo = IamAuthManager.getUserInfo(userToken);
        AuthoredUser authoredUser = covert(userInfo);
        authoredUser.setToken(userToken);
        return authoredUser;
    }

    public void verifyAndExchangeIamToken(AuthoredUser authentication) {
        if (null != authentication.getTenantId() && !"integration".equals(authentication.getUserId())) {
            try {
                // 使用新的getOrLoad方法重构
                AuthoredUser authoredUser = TokenCache.getOrLoadAuthoredUser(
                        authentication.getTenantId(),
                        tenantId -> queryLoginInternal(authentication)
                );
                AppAuthContextHolder.getContext().setIntegrationAuthoredUser(authoredUser);
            } catch (Exception e) {
                log.warn("获取用户集成金钥失败");
                setIntegrationAuthoreUser(authentication);
            }
        } else {
            setIntegrationAuthoreUser(authentication);
        }
    }

    private AuthoredUser queryLoginInternal(AuthoredUser authentication) {
        AuthoredUser result = IamManager.queryApiVirtualToken(authentication.getTenantId());
        result.setUserName("集成账号");
        result.setUserId("integration");
        result.setVersion(authentication.getVersion());
        return result;
    }

    private void setIntegrationAuthoreUser(AuthoredUser authentication) {
        AuthoredUser copy = new AuthoredUser();
        copy.setToken(authentication.getToken());
        copy.setSid(authentication.getSid());
        copy.setTenantSid(authentication.getTenantSid());
        copy.setTenantId(authentication.getTenantId());
        copy.setTenantName(authentication.getTenantName());
        copy.setIdentityType(authentication.getIdentityType());
        copy.setUserId(authentication.getUserId());
        copy.setUserName(authentication.getUserName());
        copy.setVersion(authentication.getVersion());
        AppAuthContextHolder.getContext().setIntegrationAuthoredUser(copy);
    }

    public AuthoredUser covert(AuthoredUserDTO authoredUserDTO) {
        if (authoredUserDTO == null) {
            return null;
        }

        AuthoredUser authoredUser = new AuthoredUser();
        authoredUser.setSid(authoredUserDTO.getSid());
        authoredUser.setUserId(authoredUserDTO.getUserId());
        authoredUser.setUserName(authoredUserDTO.getUserName());
        authoredUser.setToken(authoredUserDTO.getToken());
        authoredUser.setTenantSid(authoredUserDTO.getTenantSid());
        authoredUser.setTenantId(authoredUserDTO.getTenantId());
        authoredUser.setTenantName(authoredUserDTO.getTenantName());
        authoredUser.setIdentityType(authoredUserDTO.getIdentityType());
        authoredUser.setVersion(authoredUserDTO.getVersion());
        return authoredUser;
    }

    private static final String[] PREFIXES = {"NoSuch", "ParseError"};

    public BaseResultDTO<Object> getBaseResultG(String path, BusinessException businessException) {
        JaLog.error("{} error : {}", path, businessException);
        BaseResultDTO<Object> result = new BaseResultDTO<>();
        result.setBizErrorCode(businessException.getBizErrorCode());
        result.setBizErrorMsg(getBizErrorMsg(businessException));
        result.setStatus(HttpStatus.UNAUTHORIZED.value());
        result.setStatusDescription(HttpStatus.UNAUTHORIZED.getReasonPhrase());
        result.setPath(path);
        result.setErrorType(ErrorTypeEnum.BUSINESS.getValue());
        result.setErrorCode(String.valueOf(HttpStatus.UNAUTHORIZED.value()));
        result.setErrorMessage(getErrorMessage(businessException));
        result.setServerTime(System.currentTimeMillis());
        return result;
    }

    private String getBizErrorMsg(BusinessException businessException) {
        if (StringUtils.isEmpty(businessException.getBizErrorMsg())) {
            String codeMsg = JaI18nUtil.getMessage(businessException.getBizErrorCode());
            if (Arrays.stream(PREFIXES).anyMatch(codeMsg::startsWith)) {
                codeMsg = genDefaultErrMsg();
            }
            return codeMsg;
        } else {
            return businessException.getBizErrorMsg();
        }
    }

    private Object getErrorMessage(BusinessException businessException) {
        if (StringUtils.isEmpty(businessException.getErrorMessage())) {
            String codeMsg = JaI18nUtil.getMessage(businessException.getErrorCode());
            if (Arrays.stream(PREFIXES).anyMatch(codeMsg::startsWith)) {
                codeMsg = genDefaultErrMsg();
            }
            return codeMsg;
        } else {
            return businessException.getErrorMessage();
        }
    }

    public String genDefaultErrMsg() {
        String local = LocaleContextHolder.getLocale().toString();
        String errorMessage = "页面出现异常，请联系客服！";
        if (Objects.equals(local, "zh_TW") || Objects.equals(local, "zh-TW")) {
            errorMessage = "頁面出現異常，請聯系客服！";
        }
        if (Objects.equals(local, "en_US") || Objects.equals(local, "en-US")) {
            errorMessage = "The page is abnormal, please contact customer service!";
        }
        return errorMessage;
    }
}
