package com.digiwin.athena.appcore.util;

import lombok.extern.slf4j.Slf4j;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * 基于JVM内存的带过期时间的Map工具类
 *
 * @author wzq
 */
@Slf4j
public class TimeCache<K, V> {
    private Map<K, ValueWrapper<V>> map;
    private long timeout;
    private final ScheduledExecutorService schedule;

    public TimeCache(long timeout) {
        this.map = new ConcurrentHashMap<>();
        this.timeout = timeout;
        this.schedule = Executors.newSingleThreadScheduledExecutor();
        this.schedule.scheduleAtFixedRate(this::cleanup, 0, 12, TimeUnit.HOURS);
    }

    /**
     * 存储键值对（带过期时间）
     *
     * @param key       键
     * @param value     值
     * @param ttlMillis 过期时间（毫秒），例如：30分钟 = 30 * 60 * 1000
     */
    public void put(K key, V value, long ttlMillis) {
        long expireTime = System.currentTimeMillis() + ttlMillis;
        map.put(key, new ValueWrapper<>(value, expireTime));
    }

    /**
     * 存储键值对（带过期时间）
     *
     * @param key   键
     * @param value 值
     */
    public void put(K key, V value) {
        put(key, value, timeout);
    }

    /**
     * 获取键对应的值（自动清理过期数据）
     *
     * @param key 键
     * @return 值（过期返回null）
     */
    public V get(K key) {
        ValueWrapper<V> wrapper = map.get(key);
        if (wrapper == null) {
            return null;
        }

        // 检查是否过期
        if (System.currentTimeMillis() > wrapper.expireTime) {
            map.remove(key);
            return null;
        }
        return wrapper.value;
    }

    /**
     * 获取或加载值（原子操作）
     * 1. 如果缓存中存在有效值，直接返回
     * 2. 如果不存在或已过期，使用加载函数获取值并存入缓存
     */
    public V getOrLoad(K key, Function<K, V> loader) {
        return getOrLoad(key, loader, timeout);
    }

    /**
     * 获取或加载值（可指定TTL）
     */
    @SuppressWarnings("all")
    public V getOrLoad(K key, Function<K, V> loader, long ttlMillis) {
        return map.compute(key, (k, existingWrapper) -> {
            // 如果已有有效值，直接返回
            if (existingWrapper != null && !isExpired(existingWrapper)) {
                return existingWrapper;
            }

            // 加载新值
            V newValue = loader.apply(k);
            // 如果loader返回null，则移除该键
            if (newValue == null) {
                return null;
            }
            return new ValueWrapper<>(newValue, System.currentTimeMillis() + ttlMillis);
        }).value;
    }

    /**
     * 移除指定键
     *
     * @param key 键
     * @return 被移除的值（不存在或过期返回null）
     */
    public V remove(K key) {
        ValueWrapper<V> wrapper = map.remove(key);
        return (wrapper != null && !isExpired(wrapper)) ? wrapper.value : null;
    }

    /**
     * 检查键是否存在（未过期）
     *
     * @param key 键
     * @return 是否存在有效值
     */
    public boolean containsKey(K key) {
        return get(key) != null;
    }

    /**
     * 清理所有过期条目（手动调用）
     */
    public void cleanup() {
        log.info("执行清理");
        map.entrySet().removeIf(entry -> entry.getValue() == null || isExpired(entry.getValue()));
    }

    /**
     * 内部值包装类（记录值+过期时间）
     */
    private static class ValueWrapper<V> {
        final V value;
        final long expireTime;  // 过期时间戳（毫秒）

        ValueWrapper(V value, long expireTime) {
            this.value = value;
            this.expireTime = expireTime;
        }
    }

    /**
     * 检查包装类是否过期
     */
    private boolean isExpired(ValueWrapper<V> wrapper) {
        return System.currentTimeMillis() > wrapper.expireTime;
    }
}