package com.digiwin.dap.middleware.lmc.appender;

import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.LoggingEvent;
import ch.qos.logback.classic.spi.StackTraceElementProxy;
import ch.qos.logback.classic.spi.ThrowableProxyUtil;
import ch.qos.logback.core.CoreConstants;
import ch.qos.logback.core.UnsynchronizedAppenderBase;
import ch.qos.logback.core.encoder.Encoder;
import com.digiwin.dap.middleware.lmc.common.TimeingSingletonEnum;
import com.digiwin.dap.middleware.lmc.http.client.HttpConfig;
import com.digiwin.dap.middleware.lmc.internal.LMCResourceUri;
import com.digiwin.dap.middleware.lmc.util.LogUtils;
import com.jugg.agile.framework.core.config.JaEnvProperty;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.json.JSONArray;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * LoghubAppender
 *
 * @author chenzhuang
 */
public class DwLogbackAppender<E> extends UnsynchronizedAppenderBase<E> {
    /**
     * 间隔时间 mill
     */
    public static final int PERIOD_MILL_SECONDS = 2 * 1000;
    private SimpleDateFormat _sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
    /**
     * 批量插入队列size
     */
    public static final int QUEUE_SIZE = 50;
    private static final Queue<Map<String, Object>> QUEUE = new LinkedBlockingQueue<>(QUEUE_SIZE);
    private CloseableHttpClient client;

    private String app;
    private String endpoint;
    protected Encoder<E> encoder;
    private String userAgent = "logback";


    public static boolean IsStop = false;

    @Override
    public void start() {
        try {
            doStart();
        } catch (Exception e) {
            addError("Failed to start LoghubAppender.", e);
        }
    }

    private void doStart() {
        super.start();
        // 请求配置
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(HttpConfig.CONNECT_TIMEOUT)
                .setConnectionRequestTimeout(HttpConfig.CONNECT_REQUEST_TIMEOUT)
                .setSocketTimeout(HttpConfig.SOCKET_TIMEOUT).build();

        // 连接池管理
        PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setMaxTotal(HttpConfig.MAX_TOTAL_CONNECTIONS);
        connectionManager.setDefaultMaxPerRoute(HttpConfig.MAXIMUM_CONNECTION_PER_ROUTE);
        connectionManager.setValidateAfterInactivity(HttpConfig.CONNECTION_VALIDATE_AFTER_INACTIVITY_MS);
        this.client = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionManager(connectionManager).build();

        initInsertQueueTask();
    }

    @Override
    public void stop() {
        try {
            doStop();
        } catch (Exception e) {
            addError("Failed to stop LoghubAppender.", e);
        }
    }

    private void doStop() {
        if (!isStarted()) {
            return;
        }
        super.stop();

        if (this.client != null) {
            try {
                this.client.close();
            } catch (IOException var2) {
                addError(var2.getMessage());
            }
        }
    }

    @Override
    public void append(E eventObject) {
        if (IsStop || JaEnvProperty.isLocal()) {
            return;
        }

        try {
            appendEvent(eventObject);
        } catch (Exception e) {
            addError("Failed to append event.", e);
        }
    }

    private void appendEvent(E eventObject) {
        //init Event Object
        if (!(eventObject instanceof LoggingEvent)) {
            return;
        }
        LoggingEvent event = (LoggingEvent) eventObject;
        Map<String, Object> logMap = new HashMap();

        Map<String, String> mdcPropertyMap = event.getMDCPropertyMap();
        String traceId;
        if (!mdcPropertyMap.isEmpty()) {
            String ptxId = mdcPropertyMap.get("PtxId");
            //String pspanId = mdcPropertyMap.get("PspanId ");
            if (ptxId != null && !"".equals(ptxId)) {
                traceId = ptxId;
            } else {
                traceId = LogUtils.getUUID();
            }
        } else {
            traceId = LogUtils.getUUID();
        }
        logMap.putIfAbsent("traceId", traceId);

        Map<String, Object> initMap = new HashMap<>();
        LogUtils.initLogMap(initMap);
        logMap.putAll(initMap);
        logMap.putIfAbsent("appId", this.getApp());
        logMap.putIfAbsent("time", this._sdf.format(new Date()));
        logMap.putIfAbsent("level", event.getLevel().toString());
        logMap.putIfAbsent("thread", event.getThreadName());

        logMap.putIfAbsent("loggerName", event.getLoggerName());
        String addr = LogUtils.getLocalHostIpName();
        logMap.putIfAbsent("source", addr);

        logMap.putIfAbsent("appender", "DwLogbackAppender");

        StackTraceElement[] caller = event.getCallerData();
        if (caller != null && caller.length > 0) {
            logMap.putIfAbsent("location", caller[0].toString());
        }

        String message = event.getFormattedMessage();
        logMap.putIfAbsent("message", message);

        IThrowableProxy iThrowableProxy = event.getThrowableProxy();
        if (iThrowableProxy != null) {
            String throwable = getExceptionInfo(iThrowableProxy);
            throwable += fullDump(event.getThrowableProxy().getStackTraceElementProxyArray());
            logMap.putIfAbsent("throwable", throwable);
        }

        if (this.encoder != null) {
            logMap.putIfAbsent("log", new String(this.encoder.encode(eventObject)));
        }

        if (QUEUE.offer(logMap)) {
            return;
        }
        insertQueue(logMap);
    }

    public void insertQueue(Map<String, Object> logMap) {
        if (IsStop) {
            return;
        }

        if (logMap == null) {
            boolean executable = System.currentTimeMillis() - TimeingSingletonEnum.getInstance().getExecutionTimeMs() > PERIOD_MILL_SECONDS && !QUEUE.isEmpty();
            if (!executable) {
                return;
            }
        }
        List<Map<String, Object>> insertList = new ArrayList<>(QUEUE);
        QUEUE.clear();
        TimeingSingletonEnum.getInstance().setExecutionTimeMs(System.currentTimeMillis());
        Optional.ofNullable(logMap).ifPresent(QUEUE::offer);
        if (insertList.isEmpty()) {
            return;
        }

        CloseableHttpResponse hResponse = null;
        HttpPost postMethod = new HttpPost(LMCResourceUri.getSaveDevLogBatchUrl(this.endpoint));
        try {
            JSONArray tmepObjects = new JSONArray(insertList);
            String formatValue = tmepObjects.toString();
            postMethod.setEntity(new StringEntity(formatValue, ContentType.APPLICATION_JSON));

            hResponse = this.client.execute(postMethod);
            HttpEntity repEntity = hResponse.getEntity();
            int statusCode = hResponse.getStatusLine().getStatusCode();
            if (statusCode != 200) {
                postMethod.abort();
            }
        } catch (Exception e) {
            addError(e.getMessage());
        } finally {
            if (hResponse != null) {
                try {
                    hResponse.close();
                } catch (IOException e) {
                    addError(e.getMessage());
                }
            }
        }
    }

    private void initInsertQueueTask() {
        ScheduledExecutorService service = Executors.newSingleThreadScheduledExecutor();
        // 任务体，首次执行的延时时间，任务执行的间隔，间隔时间单位
        service.scheduleAtFixedRate(() -> insertQueue(null), 0, PERIOD_MILL_SECONDS, TimeUnit.MILLISECONDS);
    }

    private String getExceptionInfo(IThrowableProxy iThrowableProxy) {
        String s = iThrowableProxy.getClassName();
        String message = iThrowableProxy.getMessage();
        return (message != null) ? (s + ": " + message) : s;
    }

    private String fullDump(StackTraceElementProxy[] stackTraceElementProxyArray) {
        StringBuilder builder = new StringBuilder();
        for (StackTraceElementProxy step : stackTraceElementProxyArray) {
            builder.append(CoreConstants.LINE_SEPARATOR);
            String string = step.toString();
            builder.append(CoreConstants.TAB).append(string);
            ThrowableProxyUtil.subjoinPackagingData(builder, step);
        }
        return builder.toString();
    }

    public String getApp() {
        return app;
    }

    public void setApp(String app) {
        this.app = app;
    }

    public String getEndpoint() {
        return endpoint;
    }

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public String getUserAgent() {
        return userAgent;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }

    public Encoder<E> getEncoder() {
        return encoder;
    }

    public void setEncoder(Encoder<E> encoder) {
        this.encoder = encoder;
    }
}
