package com.digiwin.athena.framework.mongodb.configurate;

import com.digiwin.athena.framework.mongodb.aop.MongoTemplateBeanPostProcessor;
import com.digiwin.athena.framework.mongodb.factory.DynamicMongoDatabaseFactory;
import com.digiwin.athena.framework.mongodb.properties.DynamicMongodbProperties;
import com.digiwin.athena.framework.mongodb.support.*;
import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import jdk.internal.joptsimple.internal.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.mongo.MongoClientFactory;
import org.springframework.boot.autoconfigure.mongo.MongoProperties;
import org.springframework.boot.autoconfigure.mongo.MongoPropertiesClientSettingsBuilderCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.env.Environment;
import org.springframework.data.mongodb.MongoDatabaseFactory;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.Map;

@Configuration(proxyBeanMethods = false)
public class DynamicMongoClientConfiguration {
    private final Logger logger = LoggerFactory.getLogger(DynamicMongoClientConfiguration.class);

    @Bean
    @Primary
    public MongoClient primaryMongoClient(DynamicMongodbProperties properties) {
        Map<String, MongoProperties> config = properties.getConfig();
        if (config == null || config.isEmpty()) {
            throw new IllegalStateException("请至少在 spring.data.mongodb.dynamic.config 下定义一个数据源");
        }
        // 取第一个
        Map.Entry<String, MongoProperties> firstEntry = config.entrySet().iterator().next();
        MongoProperties firstProps = firstEntry.getValue();
        String mongoUrl = Strings.EMPTY;
        if (!StringUtils.isEmpty(properties.getUri())) {
            mongoUrl = properties.getUri();
        } else {
            mongoUrl = firstProps.determineUri();
        }
        ConnectionString connectionString = new ConnectionString(mongoUrl);
        MongoClientSettings settings = MongoClientSettings.builder().applyConnectionString(connectionString).build();
        return MongoClients.create(settings);
    }

    @Bean
    @ConditionalOnMissingBean
    public MongoTemplateBeanPostProcessor mongoTemplateBeanPostProcessor(MongoDatabaseFactory mongoDatabaseFactory, MappingMongoConverter mappingMongoConverter) {
        return new MongoTemplateBeanPostProcessor(mongoDatabaseFactory, mappingMongoConverter);
    }


    @Bean
    @ConditionalOnMissingBean
    public MongoDatabaseFactory mongoClientFactory(MongoClient mongoClient, MongoProperties properties, DynamicMongoClientFactory dynamicMongoClientFactory) {
        logger.info("register dynamic mongo database factory");
        return new DynamicMongoDatabaseFactory(mongoClient, properties.getMongoClientDatabase(), dynamicMongoClientFactory);
    }

    @Bean
    @ConditionalOnMissingBean
    public DynamicMongoClientBuilder dynamicMongoClientBuilder(MongoProperties primaryProperties, ObjectProvider<DynamicMongoClientSettingsCustomizer> settingsCustomizers, ObjectProvider<DynamicMongoClientPropertiesCustomizer> propertiesCustomizers) {
        logger.info("register dynamic mongo client builder");
        return ((instanceId, properties) -> {
            propertiesCustomizers.orderedStream().forEach(customizers -> customizers.customize(primaryProperties, instanceId, properties));
            return new MongoClientFactory(Collections.singletonList(clientSettingsBuilder -> settingsCustomizers.orderedStream().forEach(customizers -> customizers.customize(instanceId, clientSettingsBuilder, properties)))).createMongoClient(MongoClientSettings.builder().build());
        });
    }

    @Bean("dynamicMongoClientSettingsCustomizer")
    @ConditionalOnMissingBean(name = "dynamicMongoClientSettingsCustomizer")
    public DynamicMongoClientSettingsCustomizer dynamicMongoClientSettingsCustomizer(Environment environment) {
        logger.info("register dynamic mongo client setting customizer");
        return (instanceId, dynamic, properties) -> new MongoPropertiesClientSettingsBuilderCustomizer(properties, environment).customize(dynamic);
    }
}
