package com.digiwin.athena.framework.mongodb.factory;

import com.digiwin.athena.framework.mongodb.properties.DynamicMongodbProperties;
import com.digiwin.athena.framework.mongodb.support.DynamicMongoClientBuilder;
import com.digiwin.athena.framework.mongodb.support.DynamicMongoClientFactory;
import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.mongo.MongoProperties;
import org.springframework.util.StringUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class SimpleDynamicMongoClientFactory implements DynamicMongoClientFactory, InitializingBean, DisposableBean {
    private final Logger logger = LoggerFactory.getLogger(SimpleDynamicMongoClientFactory.class);
    private final DynamicMongodbProperties properties;
    private final DynamicMongoClientBuilder mongoClientBuilder;
    private Map<String, MongoClient> dynamicMongoClients;

    public SimpleDynamicMongoClientFactory(DynamicMongodbProperties properties, DynamicMongoClientBuilder mongoClientBuilder) {
        this.properties = properties;
        this.mongoClientBuilder = mongoClientBuilder;
    }

    @Override
    public MongoClient getDynamicMongoClient(String key) {
        return dynamicMongoClients != null ? dynamicMongoClients.get(key) : null;
    }

    /**
     * 初始化动态Mongo客户端<br></>
     * todo 针对不同instanceId相同的url的情况，存在重复创建client的情况，后期根据实际情况，看是否需要优化<br></>
     * 如果相同instanceId的url，只创建一个client，销毁需要注意，防止误销毁<br></>
     * 动态刷新，同步更新动态Mongo客户端问题，本次不做考虑
     *
     * @throws Exception
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        Map<String, MongoProperties> config = properties.getPool();
        if (config == null || config.isEmpty()) {
            throw new IllegalStateException("请至少在 spring.data.mongodb.dynamic.config 下定义一个数据源");
        }

        dynamicMongoClients = new HashMap<>(config.size());
        // 用于复用相同URI的MongoClient
        Map<String, MongoClient> uriClientCache = new ConcurrentHashMap<>();

        if (!StringUtils.isEmpty(properties.getUri())) {
            // 全局统一URI，直接创建一个MongoClient
            String mongoUrl = properties.getUri();
            ConnectionString connectionString = new ConnectionString(mongoUrl);
            MongoClientSettings settings = MongoClientSettings.builder().applyConnectionString(connectionString).build();
            MongoClient mongoClient = MongoClients.create(settings);

            for (String key : config.keySet()) {
                logger.info("init dynamic mongo database for {} -> {}", key, mongoUrl);
                dynamicMongoClients.put(key, mongoClient);
            }
        } else {
            // 针对每个数据源的URI，复用相同URI的MongoClient
            for (Map.Entry<String, MongoProperties> entry : config.entrySet()) {
                String mongoUrl = entry.getValue().determineUri();
                MongoClient mongoClient = uriClientCache.computeIfAbsent(mongoUrl, url -> {
                    ConnectionString connectionString = new ConnectionString(url);
                    MongoClientSettings settings = MongoClientSettings.builder().applyConnectionString(connectionString).build();
                    logger.info("create new MongoClient for URI: {}", url);
                    return MongoClients.create(settings);
                });
                logger.info("init dynamic mongo database for {} -> {}", entry.getKey(), mongoUrl);
                dynamicMongoClients.put(entry.getKey(), mongoClient);
            }
        }
    }


    @Override
    public void destroy() throws Exception {
        for (Map.Entry<String, MongoClient> entry : dynamicMongoClients.entrySet()) {
            logger.info("closing dynamic mongo client for name:{} ...", entry.getKey());
            try {
                MongoClient client;
                if ((client = entry.getValue()) != null) {
                    client.close();
                }
            } catch (Exception e) {
                logger.error("close dynamic mongo client failed for name:{},msg:{}", entry.getKey(), e.getMessage());
            }
        }
    }
}
