
package com.digiwin.athena.appcore.util;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.serializer.*;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;

import java.io.IOException;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 该工具类指定给调用esp接口服务时使用
 */
@SuppressWarnings("rawtypes")
public class EspJacksonUtil {

    private static class Instance {
        private static final ObjectMapper objectMapper = createObjectMapper();
    }

    private EspJacksonUtil() {
    }

    public static ObjectMapper createObjectMapper() {
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(Constants.DATE_FORMATTER));
        javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
        javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(Constants.DATETIME_FORMATTER));
        javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
        javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(Constants.TIME_FORMATTER));
        javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
        javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer(Constants.DATETIME_FORMATTER));
        javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());
        ObjectMapper objectMapper = Jackson2ObjectMapperBuilder.json().simpleDateFormat("yyyy-MM-dd HH:mm:ss").failOnUnknownProperties(false).modules(new Module[]{javaTimeModule}).build();
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        //自定义序列化的规则，默认过滤"activity__backLog__data", "activity__approval__state", "activity__data__status"
        SimpleModule module = new SimpleModule();

        String ignoreKeys = JaProperty.get("esp.http.ignoreKeys", "activity__backLog__data,activity__approval__state,activity__data__status");
        module.addSerializer(Map.class, new MapSerializer(Arrays.stream(ignoreKeys.split(",")).collect(Collectors.toSet())));
        objectMapper.registerModule(module);
        return objectMapper;
    }

    /**
     * 该方法指定给调用esp接口服务时使用，序列化时默认过滤三个字段，且是自上而下全面过滤
     * "activity__backLog__data","activity__approval__state","activity__data__status"
     *
     * @param map 要序列化的Map对象
     * @return 序列化后的JSON字符串
     */
    public static String toStringFromMapIgnoreKeys(Map map) {
        if (map == null) {
            return null;
        }
        try {
            return Instance.objectMapper.writeValueAsString(map);
        } catch (JsonProcessingException e) {
            throw BusinessException.create(e);
        }
    }
}

/**
 * 自定义序列化的规则
 */
class MapSerializer extends JsonSerializer<Map> {
    private final Set<String> privacyFields;

    public MapSerializer(Set<String> privacyFields) {
        this.privacyFields = privacyFields;
    }

    @Override
    public void serialize(Map value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
        gen.writeStartObject();
        //自定义对象的序列化行为
        for (Object o : value.keySet()) {
            if (JaCollectionUtil.isNotEmpty(this.privacyFields) && this.privacyFields.contains(o.toString())) {
                continue;
            }
            Object o1 = value.get(o);
            if (null != o1) {
                gen.writeObjectField(o.toString(), value.get(o));
            }
        }
        gen.writeEndObject();
    }

}

