package com.digiwin.athena.appcore.util;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.serializer.*;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.NullSerializer;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import net.sf.json.JSONArray;
import net.sf.json.JSONNull;
import net.sf.json.JSONObject;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;

import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

public final class JsonUtils {
    private JsonUtils() {
    }

    public static ObjectMapper getInstance() {
        return Instance.objectMapper;
    }

    public static ObjectMapper getMVCInstance() {
        return ObjectMapperInstance.objectMapper;
    }

    public static ObjectMapper createNullAlwaysMapper() {
        return NullAlwaysObjectMapperInstance.objectMapper;
    }

    public static ObjectMapper createObjectMapper() {
        return Instance.objectMapper;
    }

    public static ObjectMapper createMVCObjectMapper() {
        return ObjectMapperInstance.objectMapper;
    }

    @Deprecated
    public static void setZeroJsonData(JSONObject jsonObject, Set<String> ignoreFields) {
        if (jsonObject == null) {
            return;
        }
        for (Object key : jsonObject.keySet().toArray()) {
            if (ignoreFields != null && ignoreFields.contains(key)) {
                continue;
            }
            Object value = jsonObject.get(key);

            if (value != null) {
                if (value instanceof JSONObject) {
                    setZeroJsonData((JSONObject) value, ignoreFields);
                } else if (value instanceof JSONArray) {
                    JSONArray jsonArray = (JSONArray) value;
                    boolean isSimpleArray = false;
                    for (Object item : jsonArray) {
                        if (item instanceof JSONObject) {
                            setZeroJsonData((JSONObject) item, ignoreFields);
                        } else {
                            isSimpleArray = true;
                        }
                    }
                    if (isSimpleArray) {
                        jsonObject.put(key, null);
                    }
                } else {
                    jsonObject.put(key, "0");
                }
            }
        }
    }

    /**
     * 返回JSON 对象
     */
    public static String objectToString(Object obj) {
        if (obj == null) {
            return null;
        }
        if (obj instanceof String) {
            return obj.toString();
        }
        String json;
        try {
            json = Instance.objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw BusinessException.create(e);
        }
        return json;
    }


    /**
     * 返回JSON 对象
     */
    public static <T> T jsonToObject(String json, Class<T> valueType) {
        if (null == json) {
            return null;
        }
        try {
            return Instance.objectMapper.readValue(json, valueType);
        } catch (JsonProcessingException e) {
            throw BusinessException.create(e);
        }
    }


    /**
     * 返回JSON 对象集合
     */
    public static <T> List<T> jsonToListObject(String json, Class<T> valueType) {
        if (null == json) {
            return null;
        }
        JavaType javaType = Instance.objectMapper.getTypeFactory().constructCollectionType(List.class, valueType);
        try {
            return Instance.objectMapper.readValue(json, javaType);
        } catch (JsonProcessingException e) {
            throw BusinessException.create(e);
        }
    }

    /**
     * 返回JSON 对象
     */
    public static <T> T jsonToObject(String json, TypeReference<T> typeReference) {
        if (null == json) {
            return null;
        }
        try {
            return Instance.objectMapper.readValue(json, typeReference);
        } catch (JsonProcessingException e) {
            throw BusinessException.create(e);
        }
    }

    private static class Instance {
        private static final ObjectMapper objectMapper = createObjectMapper();

        private static ObjectMapper createObjectMapper() {
            JavaTimeModule javaTimeModule = new JavaTimeModule();
            javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(Constants.DATE_FORMATTER));
            javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
            javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(Constants.DATETIME_FORMATTER));
            javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
            javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(Constants.TIME_FORMATTER));
            javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
            javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer(Constants.DATETIME_FORMATTER));
            javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());

            ObjectMapper objectMapper = Jackson2ObjectMapperBuilder.json()
                    .simpleDateFormat("yyyy-MM-dd HH:mm:ss")
                    .failOnUnknownProperties(false)
                    .modules(javaTimeModule)
                    .build();

            objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);

            SimpleModule netSfJsonModule = new SimpleModule("net.sf.json");
            netSfJsonModule.addSerializer(JSONNull.class, NullSerializer.instance);
            objectMapper.registerModule(netSfJsonModule);

            return objectMapper;
        }
    }

    private static class ObjectMapperInstance {
        private static final ObjectMapper objectMapper = Instance.createObjectMapper();
    }

    private static class NullAlwaysObjectMapperInstance {
        private static final ObjectMapper objectMapper = NullAlwaysObjectMapperInstance.createObjectMapper();
        private static ObjectMapper createObjectMapper() {
            JavaTimeModule javaTimeModule = new JavaTimeModule();
            javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(Constants.DATE_FORMATTER));
            javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
            javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(Constants.DATETIME_FORMATTER));
            javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
            javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(Constants.TIME_FORMATTER));
            javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
            javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer(Constants.DATETIME_FORMATTER));
            javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());

            ObjectMapper objectMapper = Jackson2ObjectMapperBuilder.json()
                    .simpleDateFormat("yyyy-MM-dd HH:mm:ss")
                    .failOnUnknownProperties(false)
                    .modules(javaTimeModule)
                    .build();

//            objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);

            SimpleModule netSfJsonModule = new SimpleModule("net.sf.json");
            netSfJsonModule.addSerializer(JSONNull.class, NullSerializer.instance);
            objectMapper.registerModule(netSfJsonModule);

            return objectMapper;
        }
    }


    /**
     * 适配dap框架的序列化方式，
     * null也会序列化
     * 日期格式序列化为yyyy/mm/dd hh:mm:ss
     */
    public static ObjectMapper getDapObjectMapperInstance() {
        return DapObjectMapperInstance.objectMapper;
    }

    private static class DapObjectMapperInstance {
        private static final ObjectMapper objectMapper = createObjectMapper();

        private static ObjectMapper createObjectMapper() {
            JavaTimeModule javaTimeModule = new JavaTimeModule();
            javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer(Constants.DATE_FORMATTER_DAP));
            javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
            javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer(Constants.DATETIME_FORMATTER_DAP));
            javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
            javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer(Constants.TIME_FORMATTER_DAP));
            javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
            javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer(Constants.DATETIME_FORMATTER_DAP));
            javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());

            ObjectMapper objectMapper = Jackson2ObjectMapperBuilder.json()
                    .simpleDateFormat("yyyy/MM/dd HH:mm:ss")
                    .failOnUnknownProperties(false)
                    .modules(javaTimeModule)
                    .build();

            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);

            SimpleModule netSfJsonModule = new SimpleModule("net.sf.json");
            netSfJsonModule.addSerializer(JSONNull.class, NullSerializer.instance);
            objectMapper.registerModule(netSfJsonModule);
            objectMapper.addMixIn(BaseResultDTO.class, BaseResultDTOMixin.class);

            return objectMapper;
        }
    }

    interface BaseResultDTOMixin {

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getCode();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getErrorCode();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        Object getErrorMessage();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getErrorType();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        Long getServerTime();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getPath();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        Object getDebugInfo();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getDescription();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getBizErrorCode();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        String getBizErrorMsg();

        @JsonInclude(JsonInclude.Include.NON_NULL)
        Map<String, Object> getEspError();

    }
}

