package com.digiwin.athena.appcore.util;

import com.digiwin.athena.appcore.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;

/**
 * @author hanhao
 * 时间工具类
 */
public final class TimeUtils {

    private static final Logger logger = LoggerFactory.getLogger(TimeUtils.class);
    public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";// 时间格式
    public static final String DEFAULT_DATE_FORMAT1 = "yyyy/MM/dd";// 时间格式1
    public static final String DEFAULT_FORMAT = "yyyy-MM-dd HH:mm:ss";// 时间格式
    public static final String DEFAULT_FORMAT1 = "yyyy/MM/dd HH:mm:ss";// 时间格式1
    public static final String DEFAULT_FORMATS = "yyyy-MM-dd";
    public static final String DATE_FOMATE_YYYYMMDDHHMMSS = "yyyyMMddHHmmss";
    public static final String Message_TIME = "yyyy年MM月dd日HH点mm分";

    /**
     * 计算剩余时间（结束时间与当前时间的差值）
     * 时间规则：年/月/周/天/小时/分钟，保留前两位
     *
     * @return String
     * @author hanhao
     */
    public static String remainTime(Instant startInstant, Instant endInstant) {
        // 已完成返回--
        if (startInstant == null || endInstant == null || endInstant.isBefore(startInstant)) {
            return "";
        }

//        // 使用joda工具算
////        Interval interval = new Interval(startTime.getTime(), endTime.getTime());
//        Period period = Period.between(startTime,endTime);
//
//        logger.info("remainTime, year:{}, month:{}, week:{}, day:{}, hour:{}, minute:{}", period.getYears(),
//                period.getMonths(), period.getWeeks(), period.getDays(), period.getHours(), period.getMinutes());

        // 存入有序map
        Map<String, Long> data = new LinkedHashMap();
        data.put("年", ChronoUnit.YEARS.between(startInstant,endInstant));
        data.put("月",  ChronoUnit.MONTHS.between(startInstant,endInstant));
        data.put("周",  ChronoUnit.WEEKS.between(startInstant,endInstant));
        data.put("天", ChronoUnit.DAYS.between(startInstant,endInstant));
        data.put("小时",ChronoUnit.HOURS.between(startInstant,endInstant));
        data.put("分钟", ChronoUnit.MINUTES.between(startInstant,endInstant));

        // 找出第一个和第二个不为0的数据
        String firstDate = "", secondDate = "";
        Iterator it = data.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, Integer> entry = (Map.Entry) it.next();
            if (entry.getValue() != 0) {
                if (StringUtils.isEmpty(firstDate)) {
                    firstDate = entry.getValue() + entry.getKey();
                    continue;
                }
                if (StringUtils.isEmpty(secondDate)) {
                    secondDate = entry.getValue() + entry.getKey();
                    break;
                }
            }
        }

        // 组装返回
        return firstDate + secondDate;
    }
    /**
     * 计算两个日期的时间差，如果两个日期相等 或 startDate 大于 endDate，返回null
     * @param startDate 小日期
     * @param endDate 大日期
     * @return 如果时间差大于等于1天的，返回天数；如果大于等于1小时的，返回小时数；如果大于等于1分钟的，返回分钟数
     *      <br>不足1分钟的，返回null
     */
    public static String dateInterval(LocalDateTime startDate, LocalDateTime endDate){
        if (startDate == null || endDate == null || endDate.isBefore(startDate)) {
            return null;
        }
        Duration duration = Duration.between(startDate,endDate);
        long diff =duration.toMillis(); //endDate.getTime() - startDate.getTime();
        long interval = diff / (24 * 60 * 60 * 1000);
        if(interval>0){
            return interval +"天";
        }

        interval = diff / (60 * 60 * 1000);
        if(interval>0){
            return interval +"小时";
        }

        interval = diff / (60 * 1000);
        if(interval>0){
            return interval +"分钟";
        }

        return null;

    }

    /**
     * 计算两个日期的时间差，如果两个日期相等 或 startDate 大于 endDate，返回null
     * @param startDate 小日期
     * @param endDate 大日期
     * @return 如果时间差大于等于1天的，返回天数；如果大于等于1小时的，返回小时数；如果大于等于1分钟的，返回分钟数
     *      <br>不足1分钟的，返回null
     */
    public static String dateInterval(Date startDate, Date endDate){
        if (startDate == null || endDate == null || (endDate.getTime() - startDate.getTime()) <= 0) {
            return null;
        }

        long diff = endDate.getTime() - startDate.getTime();
        long interval = diff / (24 * 60 * 60 * 1000);
        if(interval>0){
            return interval +"天";
        }

        interval = diff / (60 * 60 * 1000);
        if(interval>0){
            return interval +"小时";
        }

        interval = diff / (60 * 1000);
        if(interval>0){
            return interval +"分钟";
        }

        return null;

    }

    /**
     * String转Date
     *
     * @param date String
     * @return Date
     * @author hanhao
     */
    public static Date string2Date(String date) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        try {
            return simpleDateFormat.parse(date);
        } catch (ParseException e) {
            logger.error(e.getMessage());
            return null;
        }
    }

    /**
     * 日期格式化
     *
     * @param date
     * @param format
     * @return
     */
    public static String format(Date date, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return sdf.format(date);
    }

    /**
     * 字符串格式化为时间
     *
     * @param dateStr 时间字符串
     * @param format  时间格式 如：DEFAULT_FORMAT1 = "yyyy/MM/dd HH:mm:ss";// 时间格式1
     * @return
     */
    public static Date parseDate(String dateStr, String format) {
        Date date = null;
        if (!StringUtils.isEmpty(dateStr)) {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            try {
                date = sdf.parse(dateStr);
            } catch (ParseException e) {
                logger.error(e.getMessage(), e);
            }
        }
        return date;
    }


    /**
     * 获取当前周
     *
     * @param dateStr
     * @param format
     * @return
     */
    public static int parseWeek(String dateStr, String format) {
        Calendar calendar = Calendar.getInstance();
        //美国是以周日为每周的第一天 现把周一设成第一天
        calendar.setFirstDayOfWeek(Calendar.MONDAY);
        calendar.setTime(parseDate(dateStr, format));
        System.out.println("当前周为" + calendar.get(Calendar.WEEK_OF_YEAR));
        return calendar.get(Calendar.WEEK_OF_YEAR);
    }


    /**
     * 根据当前日期获得上周的日期区间（本周周一和周日日期）
     *
     * @param format
     * @return
     */
    public static String getThisWeekTimeInterval(String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar calendarBegin = Calendar.getInstance(Locale.CHINA);
        calendarBegin.setFirstDayOfWeek(Calendar.MONDAY);
        calendarBegin.setTimeInMillis(System.currentTimeMillis());
        //周一
        calendarBegin.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);

        Calendar calendarEnd = Calendar.getInstance(Locale.CHINA);
        calendarEnd.setFirstDayOfWeek(Calendar.MONDAY);
        calendarEnd.setTimeInMillis(System.currentTimeMillis());
        //周日
        calendarEnd.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
        return sdf.format(TimeUtils.parseDate( TimeUtils.format(calendarBegin.getTime(),DEFAULT_FORMATS)+" 00:00:00",TimeUtils.DEFAULT_FORMAT)  ) + "," +
                sdf.format(TimeUtils.parseDate( TimeUtils.format(calendarEnd.getTime(),DEFAULT_FORMATS)+" 23:59:59",TimeUtils.DEFAULT_FORMAT)  ) ;
    }

    /**
     * 根据日期获得上周的日期区间（本周周一和周日日期）
     *
     * @param format
     * @return
     */
    public static String getThisWeekTimeInterval(String format,Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar calendarBegin = Calendar.getInstance(Locale.CHINA);
        calendarBegin.setFirstDayOfWeek(Calendar.MONDAY);
        calendarBegin.setTimeInMillis(date.getTime());
        //周一
        calendarBegin.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);

        Calendar calendarEnd = Calendar.getInstance(Locale.CHINA);
        calendarEnd.setFirstDayOfWeek(Calendar.MONDAY);
        calendarEnd.setTimeInMillis(date.getTime());
        //周日
        calendarEnd.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
        return sdf.format(TimeUtils.parseDate( TimeUtils.format(calendarBegin.getTime(),DEFAULT_FORMATS)+" 00:00:00",TimeUtils.DEFAULT_FORMAT)  ) + "," +
                sdf.format(TimeUtils.parseDate( TimeUtils.format(calendarEnd.getTime(),DEFAULT_FORMATS)+" 23:59:59",TimeUtils.DEFAULT_FORMAT)  ) ;
    }


    /**
     * 获取当前日期季度开始时间和结束时间
     *
     * @param format
     * @return
     */
    public static String getThisQuarterInterval(String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        int month = calendar.get(Calendar.MONTH) + 1;
        Integer year =calendar.get(Calendar.YEAR);
        Calendar calendarStart = Calendar.getInstance();
        Calendar calendarEnd = Calendar.getInstance();
        if (month >= 1 && month <= 3) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-01-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-03-31 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        if (month >= 4 && month <= 6) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-04-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-06-30 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        if (month >= 7 && month <= 9) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-07-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-09-30 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        if (month >= 10 && month <= 12) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-10-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-12-31 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        return null;
    }

    /**
     * 获取日期季度开始时间和结束时间
     *
     * @param format
     * @return
     */
    public static String getThisQuarterInterval(String format,Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(date.getTime());
        int month = calendar.get(Calendar.MONTH) + 1;
        Integer year =calendar.get(Calendar.YEAR);
        Calendar calendarStart = Calendar.getInstance();
        Calendar calendarEnd = Calendar.getInstance();
        if (month >= 1 && month <= 3) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-01-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-03-31 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        if (month >= 4 && month <= 6) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-04-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-06-30 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        if (month >= 7 && month <= 9) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-07-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-09-30 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        if (month >= 10 && month <= 12) {
            calendarStart.setTime(TimeUtils.parseDate(year.toString()+"-10-01 00:00:00",TimeUtils.DEFAULT_FORMAT));
            calendarEnd.setTime(TimeUtils.parseDate(year.toString()+"-12-31 23:59:59",TimeUtils.DEFAULT_FORMAT));
            return sdf.format(calendarStart.getTime()) + "," + sdf.format(calendarEnd.getTime());
        }
        return null;
    }



    /**
     * 获取当前日期月开始时间和结束时间
     *
     * @return
     */
    public static String getThisMonthInterval() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(System.currentTimeMillis());
        int month = calendar.get(Calendar.MONTH) + 1;
        return TimeUtils.getFirstDayOfMonth(month) + "," + TimeUtils.getLastDayOfMonth(month);
    }

    /**
     * 获取日期月开始时间和结束时间
     *
     * @return
     */
    public static String getThisMonthInterval(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        int month = calendar.get(Calendar.MONTH) + 1;
        return TimeUtils.getFirstDayOfMonth(month) + "," + TimeUtils.getLastDayOfMonth(month);
    }



    public static String getFirstDayOfMonth(int month) {
        Calendar cal = Calendar.getInstance();
        // 设置月份
        cal.set(Calendar.MONTH, month - 1);
        // 获取某月最小天数
        int firstDay = cal.getActualMinimum(Calendar.DAY_OF_MONTH);
        // 设置日历中月份的最小天数
        cal.set(Calendar.DAY_OF_MONTH, firstDay);
        // 格式化日期
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String firstDayOfMonth = sdf.format(cal.getTime())+" 00:00:00";
        return firstDayOfMonth;
    }
    /**
     * 获得该月最后一天
     * @param month
     * @return
     */
    public static String getLastDayOfMonth(int month) {
        Calendar cal = Calendar.getInstance();
        // 设置月份
        cal.set(Calendar.MONTH, month - 1);
        // 获取某月最大天数
        int lastDay=0;
        //2月的平年瑞年天数
        if(month==2) {
            lastDay = cal.getLeastMaximum(Calendar.DAY_OF_MONTH);
        }else {
            lastDay = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
        }
        // 设置日历中月份的最大天数
        cal.set(Calendar.DAY_OF_MONTH, lastDay);
        // 格式化日期
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String lastDayOfMonth = sdf.format(cal.getTime())+" 23:59:59";
        return lastDayOfMonth;
    }

    /**
     * 根据当前日期获得上周的日期区间（上周周一和周日日期
     *
     * @param format
     * @return
     */
    public static String getLastWeekTimeInterval(SimpleDateFormat format) {

        Date date = new Date();
        Calendar calendarBegin = Calendar.getInstance();
        calendarBegin.setTime(date);
        //判断当前日期是否为周末，因为周末是本周第一天，如果不向后推迟一天的到的将是下周一的零点，而不是本周周一零点
        if (1 == calendarBegin.get(Calendar.DAY_OF_WEEK)) {
            calendarBegin.add(Calendar.DATE, -1);
        }
        //时间减去7天
        calendarBegin.add(Calendar.DAY_OF_MONTH, -7);
        calendarBegin.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);

        Calendar calendarEnd = Calendar.getInstance();
        calendarEnd.setTime(date);
        if (1 == calendarEnd.get(Calendar.DAY_OF_WEEK)) {
            calendarEnd.add(Calendar.DATE, -1);
        }
        calendarEnd.set(Calendar.DAY_OF_WEEK, Calendar.SUNDAY);
        return format.format(calendarBegin.getTime()) + "," + format.format(calendarEnd.getTime());
    }

    public static  Date addHour(Date date,int offset){
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.HOUR, offset);
        return cal.getTime();
    }
    public static  Date addMinute(Date date,int offset){
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.MINUTE, offset);

        return cal.getTime();
    }

    public static void main(String[] args) {

        LocalDate d=   LocalDate.parse("2020-10-04T16:00:00.000Z".substring(0,10),DateTimeFormatter.ISO_LOCAL_DATE);
        String startTime = TimeUtils.format(TimeUtils.parse("2010/12/12 12:12:00")
                , TimeUtils.DEFAULT_FORMAT).substring(5, 10).replace("-", ".");
        String endTime = TimeUtils.format(TimeUtils.parse("2020-10-04T16:00:00.000Z")
                , TimeUtils.DEFAULT_FORMAT).substring(5, 10).replace("-", ".");

        String aaa = TimeUtils.getThisWeekTimeInterval("yyyy-MM-dd HH:mm:ss",new Date());
        String bb = TimeUtils.getThisQuarterInterval("yyyy-MM-dd HH:mm:ss",new Date());
        String cc = getThisMonthInterval(new Date());
        System.out.println(aaa);
        System.out.println(bb);
        System.out.println(cc);

    }

    public static LocalTime parseTime(String dateCharSequence) {
        return parse(dateCharSequence).toLocalTime();
    }
    public static LocalDate parseDate(String dateCharSequence) {
        if(org.springframework.util.StringUtils.isEmpty(dateCharSequence)){
            return null;
        }
        if (dateCharSequence.contains("T")){
            return  LocalDate.parse(dateCharSequence.substring(0,10),DateTimeFormatter.ISO_LOCAL_DATE);
        }
        else if (dateCharSequence.contains("-")){
            DateTimeFormatter df = DateTimeFormatter.ofPattern(TimeUtils.DEFAULT_DATE_FORMAT);
            return  LocalDate.parse(dateCharSequence,df);
        }
        else  if (dateCharSequence.contains("/")){
            DateTimeFormatter df = DateTimeFormatter.ofPattern(TimeUtils.DEFAULT_DATE_FORMAT1);
            return  LocalDate.parse(dateCharSequence,df);
        }

        throw BusinessException.create("不支持的日期格式");
    }
    public static LocalDateTime parse(String dateCharSequence) {
        if(org.springframework.util.StringUtils.isEmpty(dateCharSequence)){
            return null;
        }
        if (!dateCharSequence.contains(":")){
            dateCharSequence += " 00:00:00";
        }
        if (dateCharSequence.contains("T")){
             return  LocalDateTime.parse(dateCharSequence.substring(0,19),DateTimeFormatter.ISO_LOCAL_DATE_TIME);
        }
        else if (dateCharSequence.contains("-")){
            DateTimeFormatter df = DateTimeFormatter.ofPattern(TimeUtils.DEFAULT_FORMAT);
            return  LocalDateTime.parse(dateCharSequence,df);
        }
        else  if (dateCharSequence.contains("/")){
            DateTimeFormatter df = DateTimeFormatter.ofPattern(TimeUtils.DEFAULT_FORMAT1);
            return  LocalDateTime.parse(dateCharSequence,df);
        }

        throw  BusinessException.create("不支持的日期格式");
    }

    public static  String format(LocalDateTime dateTime,String format){
        DateTimeFormatter df = DateTimeFormatter.ofPattern(format);
        return  df.format(dateTime);
    }

}
