package com.digiwin.athena.framework.jsonview.advice;

import com.digiwin.athena.framework.jsonview.annotation.DynamicJsonView;
import com.digiwin.athena.framework.jsonview.filter.DynamicBeanPropertyFilterBean;
import com.digiwin.athena.framework.jsonview.filter.DynamicFilterProvider;
import com.digiwin.athena.framework.jsonview.resolver.DynamicFilterResolver;
import com.digiwin.athena.framework.jsonview.resolver.FilterIgnoreResolver;
import com.digiwin.athena.framework.jsonview.resolver.FilterIncloudResolver;
import org.springframework.core.MethodParameter;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJacksonValue;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.servlet.mvc.method.annotation.AbstractMappingJacksonResponseBodyAdvice;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.stream.Stream;

//@ControllerAdvice
public class DynamicFilterResponseBodyAdvice extends AbstractMappingJacksonResponseBodyAdvice {

    private static final String JSON_VIEW_HEADER = "digi-json-view";

    @SuppressWarnings("rawtypes")
    protected final Map<Class, DynamicFilterResolver> resolvers = new HashMap<>();

    public DynamicFilterResponseBodyAdvice() {
        addResolvers(new FilterIgnoreResolver(), new FilterIncloudResolver());
    }

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        //根据header头去匹配
        return super.supports(returnType, converterType) && isSupport(returnType);
    }

    @Override
    protected void beforeBodyWriteInternal(MappingJacksonValue bodyContainer, MediaType contentType, MethodParameter returnType, ServerHttpRequest request, ServerHttpResponse response) {
        resolveFilter(returnType).map(DynamicFilterProvider::new).ifPresent(bodyContainer::setFilters);
    }

    public void addResolvers(DynamicFilterResolver resolver, DynamicFilterResolver... more) {
        resolvers.put(resolver.getClass(), resolver);
        Stream.of(more).filter(Objects::nonNull).forEach(r -> resolvers.put(resolver.getClass(), r));
    }

    protected Optional<DynamicBeanPropertyFilterBean> resolveFilter(MethodParameter returnType) {
        DynamicJsonView dynamicView = getSupportDynamicJsonView(returnType);
        return resolvers.values().stream().map(resolver -> resolver.resolve(dynamicView)).filter(Objects::nonNull).findFirst();
    }

    private boolean isSupport(MethodParameter returnType) {
        String jsonView = getCurrentViewFromHeader();
        if (StringUtils.isEmpty(jsonView)) {
            return false;
        }
        DynamicJsonView[] annotations = returnType.getMethod().getAnnotationsByType(DynamicJsonView.class);
        if (annotations == null || annotations.length == 0) {
            return false;
        }
        //校验动态视图
        Arrays.stream(annotations).forEach(this::validator);
        return Arrays.stream(annotations).anyMatch(annotation -> annotation.viewName().equals(jsonView));
    }

    /**
     * 获取适配的DynamicJsonView
     *
     * @param returnType
     * @return
     */
    private DynamicJsonView getSupportDynamicJsonView(MethodParameter returnType) {
        String jsonView = getCurrentViewFromHeader();
        DynamicJsonView[] annotations = returnType.getMethod().getAnnotationsByType(DynamicJsonView.class);
        //同样jsonView只有一个生效
        return Arrays.stream(annotations).filter(annotation -> annotation.viewName().equals(jsonView)).findFirst().orElse(null);
    }


    /**
     * 从当前请求中获取视图名称
     *
     * @return
     */
    private String getCurrentViewFromHeader() {
        ServletRequestAttributes attrs = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (attrs != null) {
            HttpServletRequest request = attrs.getRequest();
            return request.getHeader(JSON_VIEW_HEADER);
        }
        return null;
    }

    private void validator(DynamicJsonView dynamicView) {
        if (dynamicView != null) {
            String[] ignoreFields = dynamicView.ignoreFields();
            String[] includeFields = dynamicView.includeFields();
            if (ignoreFields.length > 0 && includeFields.length > 0) {
                throw new IllegalArgumentException("DynamicJsonView cannot have both ignoreFields and includeFields set at the same time.");
            }
        }
    }

}
