package com.digiwin.athena.framework.mq.retry.interceptor;

import com.digiwin.athena.framework.mq.retry.RabbitMqRetryProperties;
import com.digiwin.athena.framework.mq.retry.annotation.RabbitRetry;
import com.digiwin.athena.framework.mq.retry.context.MQRetryContextHolder;
import com.digiwin.athena.framework.mq.retry.exception.OverLimitException;
import com.digiwin.athena.framework.mq.retry.exception.RejectMQException;
import com.digiwin.athena.framework.mq.retry.exception.RetryMQException;
import com.digiwin.athena.framework.mq.retry.handler.*;
import com.digiwin.athena.framework.mq.retry.support.RabbitMqHandlerMethodArgumentResolver;
import com.digiwin.athena.framework.mq.retry.support.RabbitRetryMethod;
import com.digiwin.athena.framework.mq.retry.support.RetrySingleton;
import com.jugg.agile.framework.core.util.concurrent.JaExecutors;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInvocation;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


@Slf4j
@Aspect
@ConditionalOnProperty(prefix = "athena.mq.retry", name = "enable", havingValue = "true")
public class RabbitRetryInterceptor extends AbstractInterceptor {

    public RabbitRetryInterceptor(RabbitTemplate rabbitTemplate, RabbitMqRetryProperties retryProperties) {
        super(rabbitTemplate, retryProperties);
    }

    @Override
    protected Object processMessage(ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding, RabbitRetry rabbitRetry, Channel channel, Long tag) throws Throwable {
        try {
            Object result = joinPoint.proceed();
            executeSuccessHandler(retryMethod);
            return result;
        } catch (Throwable e) {
            handleProcessingException(e, queueBinding, rabbitRetry, joinPoint, retryMethod, channel, tag);
        }
        return null;
    }

    @Override
    protected void validateQueueBinding(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding) {
        if (StringUtils.isEmpty(queueBinding.getExchangeName()) || StringUtils.isEmpty(queueBinding.getRoutingkey())) {
            throw new RejectMQException("Exchange or Routing Key not found");
        }
    }

    @Override
    protected void validateRetryCount(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding) {
        Integer retryCount = Optional.ofNullable(queueBinding.getRetryCount()).orElse(0);
        if (retryCount > retryProperties.getMaxMQRetryCount()) {
            throw new OverLimitException("Retry count exceeds maximum limit");
        }
    }

    private void handleProcessingException(Throwable e, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding, RabbitRetry rabbitRetry, ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod, Channel channel, Long tag) throws Throwable {
        if (e instanceof RejectMQException) {
//            acKnowledgeMessage(channel, tag);
            // 更新最终失败状态和信息
            executeFailureMethod(retryMethod, e);
        } else if (e instanceof RetryMQException) {
            //校验重试次数
            validateRetryCount(queueBinding);
            // 重新放回队列并添加消息头信息
            retryWithBackToQueue(queueBinding);
        } else {
            //本地重试
            handleRetry(joinPoint, retryMethod, rabbitRetry, queueBinding);
        }
    }

    private void retryWithBackToQueue(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBindingBean) {
        MQRetryContextHolder.MQRetryContext mqRetryContext = MQRetryContextHolder.getContext();
        mqRetryContext.setMqRetrycount(mqRetryContext.getMqRetrycount() + 1);

        rabbitTemplate.convertAndSend(queueBindingBean.getExchangeName(), queueBindingBean.getRoutingkey(), queueBindingBean.getMessage(), message -> {
            MessageProperties messageProperties = message.getMessageProperties();
            int retryCount = Optional.ofNullable(messageProperties.getHeader("retry-count")).map(Object::toString).map(Integer::parseInt).orElse(0);
            messageProperties.setHeader("retry-count", retryCount + 1);
            return message;
        });
    }

    private void handleRetry(ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod, RabbitRetry rabbitRetry, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBindingBean) throws Throwable {
        RetryTemplate retryTemplate = createRetryTemplate(rabbitRetry);
        retryTemplate.execute(context -> {
            try {
                Object result = joinPoint.proceed();
                executeSuccessHandler(retryMethod);
                return result;
            } catch (Throwable e) {
                throw e;
            }
        }, context -> {
            //添加失败次数
            MQRetryContextHolder.getContext().setLocalRetrycount(context.getRetryCount() + 1);
            executeFailureMethod(retryMethod, context.getLastThrowable());
            return null;
        });
    }


}

