package com.digiwin.athena.framework.mq.retry.support;

import com.rabbitmq.client.Channel;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.ExchangeBuilder;
import org.springframework.amqp.rabbit.annotation.Exchange;
import org.springframework.amqp.rabbit.annotation.Queue;
import org.springframework.amqp.rabbit.annotation.QueueBinding;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.support.AmqpHeaders;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.messaging.Message;
import org.springframework.messaging.handler.annotation.Header;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.*;

@Slf4j
public class RabbitMqHandlerMethodArgumentResolver {

    public static Channel resolveArgumentChannel(Object[] args) {
        for (Object arg : args) {
            if (arg instanceof Channel) {
                Channel channel = (Channel) arg;
                // 处理 Channel
                return channel;
            }
        }
        return null;
    }

    public static Long resolveArgumentTag(Method method, Object[] args) {
        // 获取方法的所有参数类型和注解
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
//        Class<?>[] parameterTypes = method.getParameterTypes();
        // 遍历参数注解
        for (int i = 0; i < parameterAnnotations.length; i++) {
            Annotation[] annotations = parameterAnnotations[i];
            for (Annotation annotation : annotations) {
                if (annotation instanceof Header) {
                    Header header = (Header) annotation;
                    // 检查注解的值是否为 AmqpHeaders.DELIVERY_TAG
                    if (AmqpHeaders.DELIVERY_TAG.equals(header.value())) {
                        // 返回对应参数的值
                        return (Long) args[i];
                    }
                }
            }
        }
        // 未找到符合条件的参数
        return null;
    }


    public static QueueBindingBean resolveExchangeRoutingkeyMsg(RabbitListener rabbitListener, Method method, Object[] args) {
        QueueBinding[] queueBindings = rabbitListener.bindings();
        QueueBindingBean queueBindingBean = new QueueBindingBean();
        if (queueBindings != null) {
            QueueBinding queueBinding = queueBindings[0];
            Exchange exchange = queueBinding.exchange();
            String exchangeName = exchange.value();
            queueBindingBean.setExchangeName(exchangeName);
            Queue queue = queueBinding.value();
            String queueName = queue.value();
            queueBindingBean.setQueueName(queueName);
            String[] keys = queueBinding.key();
            if (keys != null) {
                String routingkey = keys[0];
                queueBindingBean.setRoutingkey(routingkey);
            }
            Object message = getMessage(method, args);
            queueBindingBean.setMessage(message);
            Integer retryCount = getRetryCountFromHeader(method, args);
            queueBindingBean.setRetryCount(retryCount);
        }

        return queueBindingBean;
    }

    private static boolean isMessageBody(Class<?> paramType, Annotation[] annotations) {
        return !isHeaderParam(annotations) && (paramType == String.class || paramType == byte[].class || paramType.isAssignableFrom(Message.class) || paramType.isAssignableFrom(Object.class));
    }

    private static boolean isHeaderParam(Annotation[] annotations) {
        return Arrays.stream(annotations).anyMatch(ann -> ann.annotationType() == Header.class);
    }

    public static Object getMessage(Method method, Object[] args) {
        Class<?>[] parameterTypes = method.getParameterTypes();
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
        for (int i = 0; i < parameterTypes.length; i++) {
            Class<?> paramType = parameterTypes[i];
            Annotation[] annotations = parameterAnnotations[i];
            if (isMessageBody(paramType, annotations)) {
                Object msg = args[i];
                return msg;
            }
        }
        return null;
    }

    private static Integer getRetryCountFromHeader(Method method, Object[] args) {
        Class<?>[] parameterTypes = method.getParameterTypes();
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
        for (int i = 0; i < parameterTypes.length; i++) {
            Annotation[] annotations = parameterAnnotations[i];
            if (isHeaderParam(annotations)) {
                Object msg = args[i];
                if (msg instanceof Map) {
                    Object retryValue = ((Map<?, ?>) msg).get("retry-count");
                    return parseRetryCount(retryValue);
                }
                return parseRetryCount(msg);
            }
        }
        return 0;
    }

    private static Integer parseRetryCount(Object value) {
        if (value == null) {
            return 0;
        }
        try {
            if (value instanceof Integer) {
                return (Integer) value;
            } else if (value instanceof String) {
                return Integer.parseInt((String) value);
            } else if (value instanceof Number) {
                return ((Number) value).intValue();
            } else {
                return Integer.parseInt(value.toString());
            }
        } catch (NumberFormatException ex) {
            log.error("Invalid retry-count value: " + value);
            return 0;
        }
    }

    @Data
    public static class QueueBindingBean {
        private String exchangeName;
        private String queueName;
        private String routingkey;
        private Object message;
        //重试次数，值针对mq的重试
        private Integer retryCount;
    }
}
