package com.digiwin.athena.framework.mq.retry.support;

import com.digiwin.athena.framework.mq.retry.annotation.RabbitRetry;
import com.digiwin.athena.framework.mq.retry.handler.*;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Method;


@Slf4j
public class RabbitRetryAnnotationBeanPostProcessor implements BeanPostProcessor, BeanFactoryAware {

    private BeanFactory beanFactory;

    @Override
    public Object postProcessAfterInitialization(final Object bean, final String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);
        buildRabbitRetryMethod(targetClass);
        return bean;
    }

    private void buildRabbitRetryMethod(Class<?> targetClass) {
        ReflectionUtils.doWithMethods(targetClass, method -> {
            RabbitRetry retryAnnotation = findRabbitRetryAnnotations(method);
            if (retryAnnotation != null) {
                RabbitRetryMethod retryMethod = new RabbitRetryMethod(method, retryAnnotation);
                HandlerAdapter handlerAdapter = new HandlerAdapter();

                String initHandlerBeanName = retryAnnotation.initHandler();
                if (!StringUtils.isEmpty(initHandlerBeanName)) {
                    InitHandler initHandler = (InitHandler) beanFactory.getBean(initHandlerBeanName);
                    if (initHandler == null) {
                        throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry initHandler not found");
                    }
                    handlerAdapter.setInitHandler(initHandler);
                }

//                String beforeHandlerBeanName = retryAnnotation.beforeHandler();
//                if (!StringUtils.isEmpty(beforeHandlerBeanName)) {
//                    BeforeHandler beforeHandler = (BeforeHandler) beanFactory.getBean(beforeHandlerBeanName);
//                    if (beforeHandler != null) {
//                        throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry beforeHandler not found");
//                    }
//                    handlerAdapter.setBeforeHandler(beforeHandler);
//                }

                String idempotentHandlerBeanName = retryAnnotation.idempotentHandler();
                if (!StringUtils.isEmpty(idempotentHandlerBeanName)) {
                    IdempotentHandler idempotentHandler = (IdempotentHandler) beanFactory.getBean(idempotentHandlerBeanName);
                    if (idempotentHandler == null) {
                        throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry beforeHandler not found");
                    }
                    handlerAdapter.setIdempotentHandler(idempotentHandler);
                }

                String successHanderBeanName = retryAnnotation.successHander();
                if (!StringUtils.isEmpty(successHanderBeanName)) {
                    SuccessHandler successHandler = (SuccessHandler) beanFactory.getBean(successHanderBeanName);
                    if (successHandler == null) {
                        throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry successHandler not found");
                    }
                    handlerAdapter.setSuccessHandler(successHandler);
                }

                String failureHandlerName = retryAnnotation.failureHandler();
                if (!StringUtils.isEmpty(failureHandlerName)) {
                    FailureHandler failureHandler = (FailureHandler) beanFactory.getBean(failureHandlerName);
                    if (failureHandler == null) {
                        throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry failureHandler not found");
                    }
                    handlerAdapter.setFailureHandler(failureHandler);
                }
                retryMethod.setHandlerAdapter(handlerAdapter);
                RetrySingleton.getInstance().put(method, retryMethod);
            }
        }, ReflectionUtils.USER_DECLARED_METHODS);
    }

    private RabbitRetry findRabbitRetryAnnotations(Method element) {
        if (element.getDeclaringClass() != Object.class) {
            RabbitRetry retryAnnotation = element.getAnnotation(RabbitRetry.class);
            return retryAnnotation;
        }
        return null;
    }

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

}
