package com.digiwin.athena.framework.mq.retry.interceptor;

import com.digiwin.athena.framework.mq.retry.RabbitMqRetryProperties;
import com.digiwin.athena.framework.mq.retry.annotation.RabbitRetry;
import com.digiwin.athena.framework.mq.retry.context.MQRetryContextHolder;
import com.digiwin.athena.framework.mq.retry.handler.FailureHandler;
import com.digiwin.athena.framework.mq.retry.support.RabbitMqHandlerMethodArgumentResolver;
import com.digiwin.athena.framework.mq.retry.support.RabbitRetryMethod;
import com.digiwin.athena.framework.mq.retry.support.RetrySingleton;
import com.jugg.agile.framework.core.util.concurrent.JaExecutors;
import com.jugg.agile.middleware.rabbitmq.spring.JaNodeSpanRabbitListenerConsumer;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Slf4j
public abstract class AbstractInterceptor {

    protected final RabbitTemplate rabbitTemplate;
    protected final RabbitMqRetryProperties retryProperties;
    private final ExecutorService threadPoolTaskExecutor;

    protected AbstractInterceptor(RabbitTemplate rabbitTemplate, RabbitMqRetryProperties retryProperties) {
        this.rabbitTemplate = rabbitTemplate;
        this.retryProperties = retryProperties;
        this.threadPoolTaskExecutor = createThreadPool(retryProperties);
    }

    private ExecutorService createThreadPool(RabbitMqRetryProperties rabbitMqRetryProperties) {
        RabbitMqRetryProperties.ThreadPoolConfig config = rabbitMqRetryProperties.getPoolConfig();
        return JaExecutors.createExecutorService(JaExecutors.Config.builder().prefixYaml("RabbitRetryExecutor").defaultPrefixName("RabbitRetryExecutor").defaultCorePoolSize(config.getCoreSize()).defaultMaximumPoolSize(config.getMaxSize()).defaultKeepAliveTime(config.getKeepAliveTime()).defaultQueueSize(config.getQueueSize()).handler(new ThreadPoolExecutor.CallerRunsPolicy()).build());
    }

    @Pointcut("@annotation(com.digiwin.athena.framework.mq.retry.annotation.RabbitRetry)")
    public void rabbitListenerMethods() {
    }

    @Around("rabbitListenerMethods()")
    public Object intercept(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();
        RabbitRetry rabbitRetry = method.getAnnotation(RabbitRetry.class);
        RabbitListener rabbitListener = method.getAnnotation(RabbitListener.class);

        if (rabbitRetry == null || rabbitListener == null) {
            log.warn("Missing RabbitRetry or RabbitListener annotation on method: {}", method.getName());
            return joinPoint.proceed();
        }
        log.info("RabbitRetryInterceptor start");
        MQRetryContextHolder.MQRetryContext mqRetryContext = MQRetryContextHolder.getContext();
        Object[] args = joinPoint.getArgs();
        //解析交换机和路由
        RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding = resolveQueueBinding(rabbitListener, method, args, rabbitTemplate);
        mqRetryContext.setQueueBinding(queueBinding);
        mqRetryContext.setMqRetrycount(queueBinding.getRetryCount());

        Channel channel = JaNodeSpanRabbitListenerConsumer.ChannelThreadLocal.get();
//        Long tag = resolveTag(method, args);
        Message message = JaNodeSpanRabbitListenerConsumer.MessageThreadLocal.get();
        MessageProperties properties = message.getMessageProperties();
        long tag = properties.getDeliveryTag();

        RabbitRetryMethod retryMethod = getRetryMethod(method);
        try {
            //校验信息
            validateQueueBinding(queueBinding);
            //校验重试次数
            validateRetryCount(queueBinding);
//            executeIdempotentHandler(retryMethod);
//            executeInitHandler(retryMethod);
            executeBeforeHandler(retryMethod);

            Future<?> future = threadPoolTaskExecutor.submit(() -> {
                try {
                    processMessage(joinPoint, retryMethod, queueBinding, rabbitRetry);
                } catch (Throwable e) {
                    log.error("Error processing message", e);
                } finally {
                    MQRetryContextHolder.clearContext();
                }
            });
            return retryProperties.isAsync() ? null : future.get(retryProperties.getAsyncTimeout(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
//            MQRetryContextHolder.getContext().setLastException(e);
            executeFailureMethod(retryMethod, e);
        } finally {
            MQRetryContextHolder.clearContext();
            acKnowledgeMessage(channel, tag);
        }
        return null;
    }

    protected abstract Object processMessage(ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding, RabbitRetry rabbitRetry) throws Throwable;

    protected abstract void validateQueueBinding(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding);

    protected abstract void validateRetryCount(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding);

    protected RabbitRetryMethod getRetryMethod(Method method) {
        return RetrySingleton.getInstance().get(method);
    }

//    protected void executeIdempotentHandler(RabbitRetryMethod retryMethod) {
//        retryMethod.getHandlerAdapter().getIdempotentHandler().invokeHandler();
//    }

//    protected void executeInitHandler(RabbitRetryMethod retryMethod) {
//        retryMethod.getHandlerAdapter().getInitHandler().invokeHandler();
//    }

    protected void executeBeforeHandler(RabbitRetryMethod retryMethod) {
        retryMethod.getHandlerAdapter().getBeforeHandler().invokeHandler();
    }

    protected void executeSuccessHandler(RabbitRetryMethod retryMethod) {
        retryMethod.getHandlerAdapter().getSuccessHandler().invokeHandler();
    }

    protected void executeFailureMethod(RabbitRetryMethod retryMethod, Throwable throwable) {
        MQRetryContextHolder.getContext().setLastException(throwable);
        FailureHandler failureHandler = retryMethod.getHandlerAdapter().getFailureHandler();
        failureHandler.invokeHandler();
    }

    protected Channel resolveChannel(Object[] args) {
        return RabbitMqHandlerMethodArgumentResolver.resolveArgumentChannel(args);
    }

    protected Long resolveTag(Method method, Object[] args) {
        return RabbitMqHandlerMethodArgumentResolver.resolveArgumentTag(method, args);
    }

    protected RabbitMqHandlerMethodArgumentResolver.QueueBindingBean resolveQueueBinding(RabbitListener rabbitListener, Method method, Object[] args, RabbitTemplate rabbitTemplate) {
        return RabbitMqHandlerMethodArgumentResolver.resolveExchangeRoutingkeyMsg(rabbitListener, method, args, rabbitTemplate);
    }

    protected RetryTemplate createRetryTemplate(RabbitRetry rabbitRetry) {
        RetryTemplate template = new RetryTemplate();

        Map<Class<? extends Throwable>, Boolean> retryableExceptions = new HashMap<>();
        retryableExceptions.put(Exception.class, true);

        Class<? extends Throwable>[] classes = rabbitRetry.rejectForException();
        Arrays.stream(classes).forEach(e -> {
            retryableExceptions.put(e, false);
        });
        SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy(rabbitRetry.maxAttempts(), retryableExceptions);
        template.setRetryPolicy(retryPolicy);

        ExponentialBackOffPolicy backOffPolicy = new ExponentialBackOffPolicy();
        backOffPolicy.setInitialInterval(rabbitRetry.initialInterval); // 初始间隔为1秒
        backOffPolicy.setMultiplier(rabbitRetry.multiplier); // 每次重试间隔倍增
        backOffPolicy.setMaxInterval(rabbitRetry.maxInterval); // 最大间隔为10秒
        template.setBackOffPolicy(backOffPolicy);

        return template;
    }

    /**
     * 消息确认
     *
     * @param channel
     * @param tag
     * @throws IOException
     */
    private void acKnowledgeMessage(Channel channel, Long tag) throws IOException {
        log.info("RabbitRetryInterceptor auto ack!!!");
        if (channel != null && channel.isOpen() && tag != null) {
            log.info("RabbitRetryInterceptor msg ack");
            channel.basicAck(tag, false);
        } else {
            log.error("RabbitRetryInterceptor cannot found channel or tag");
//            throw new IllegalArgumentException("RabbitRetryInterceptor cannot found channel or tag");
        }
    }


}
