package com.digiwin.athena.framework.mq.retry.support;

import com.digiwin.athena.framework.mq.retry.annotation.RabbitRetry;
import com.digiwin.athena.framework.mq.retry.handler.*;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.support.AmqpHeaders;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.messaging.handler.annotation.Header;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;


@Slf4j
public class RabbitRetryAnnotationBeanPostProcessor implements BeanPostProcessor, BeanFactoryAware {

    private BeanFactory beanFactory;

    @Override
    public Object postProcessAfterInitialization(final Object bean, final String beanName) throws BeansException {
        Class<?> targetClass = AopUtils.getTargetClass(bean);
        buildRabbitRetryMethod(targetClass);
        return bean;
    }

    private void buildRabbitRetryMethod(Class<?> targetClass) {
        ReflectionUtils.doWithMethods(targetClass, method -> {
            RabbitRetry retryAnnotation = findRabbitRetryAnnotations(method);
            if (retryAnnotation != null) {
                RabbitListener rabbitListener = findRabbitListerAnnotations(method);
                if (rabbitListener == null) {
                    throw new IllegalArgumentException("RabbitRetry annotation must be used with RabbitListener annotation, but found on method " + method.getName() + " in class " + targetClass.getName());
                }

//                validateRabbitListenerParameters(method);

                RabbitRetryMethod retryMethod = new RabbitRetryMethod(method, retryAnnotation);
                HandlerAdapter handlerAdapter = new HandlerAdapter();

                extracted(targetClass, method, retryAnnotation, handlerAdapter, retryMethod);
            }
        }, ReflectionUtils.USER_DECLARED_METHODS);
    }

    private void extracted(Class<?> targetClass, Method method, RabbitRetry retryAnnotation, HandlerAdapter handlerAdapter, RabbitRetryMethod retryMethod) {
        String beforeHandlerBeanName = retryAnnotation.beforeHandler();
        if (!StringUtils.isEmpty(beforeHandlerBeanName)) {
            BeforeHandler beforeHandler = (BeforeHandler) beanFactory.getBean(beforeHandlerBeanName);
            if (beforeHandler == null) {
                throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry beforeHandler not found");
            }
            handlerAdapter.setBeforeHandler(beforeHandler);
        }

        String successHanderBeanName = retryAnnotation.successHander();
        if (!StringUtils.isEmpty(successHanderBeanName)) {
            SuccessHandler successHandler = (SuccessHandler) beanFactory.getBean(successHanderBeanName);
            if (successHandler == null) {
                throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry successHandler not found");
            }
            handlerAdapter.setSuccessHandler(successHandler);
        }

        String failureHandlerName = retryAnnotation.failureHandler();
        if (!StringUtils.isEmpty(failureHandlerName)) {
            FailureHandler failureHandler = (FailureHandler) beanFactory.getBean(failureHandlerName);
            if (failureHandler == null) {
                throw new NoSuchBeanDefinitionException(targetClass.getName() + "RabbitRetry failureHandler not found");
            }
            handlerAdapter.setFailureHandler(failureHandler);
        }
        retryMethod.setHandlerAdapter(handlerAdapter);
        RetrySingleton.getInstance().put(method, retryMethod);
    }

    private RabbitRetry findRabbitRetryAnnotations(Method element) {
        if (element.getDeclaringClass() != Object.class) {
            RabbitRetry retryAnnotation = element.getAnnotation(RabbitRetry.class);
            return retryAnnotation;
        }
        return null;
    }

    private RabbitListener findRabbitListerAnnotations(Method element) {
        if (element.getDeclaringClass() != Object.class) {
            RabbitListener rabbitListener = element.getAnnotation(RabbitListener.class);
            return rabbitListener;
        }
        return null;
    }

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    /**
     * 校验 @RabbitListener 方法参数是否包含必须的 @Header 注解
     */
    private void validateRabbitListenerParameters(Method method) {
        Parameter[] parameters = method.getParameters();
        boolean hasDeliveryTag = false;
        boolean hasRetryCount = false;

        for (Parameter parameter : parameters) {
            Annotation headerAnnotation = AnnotationUtils.getAnnotation(parameter, Header.class);
            if (headerAnnotation != null) {
                String headerName = ((Header) headerAnnotation).value();
                if (AmqpHeaders.DELIVERY_TAG.equals(headerName)) {
                    hasDeliveryTag = true;
                }
                if ("retry-count".equals(headerName)) {
                    hasRetryCount = true;
                }
            }
        }

        if (!hasDeliveryTag) {
            throw new IllegalArgumentException("@RabbitListener method " + method.getName() + " in class " + method.getDeclaringClass().getName() + " must include parameter annotated with @Header(AmqpHeaders.DELIVERY_TAG).");
        }
        if (!hasRetryCount) {
            throw new IllegalArgumentException("@RabbitListener method " + method.getName() + " in class " + method.getDeclaringClass().getName() + " must include parameter annotated with @Header(\"retry-count\").");
        }
    }

}
