package com.digiwin.athena.framework.mq.retry.interceptor;

import com.digiwin.athena.framework.mq.retry.RabbitMqRetryProperties;
import com.digiwin.athena.framework.mq.retry.annotation.RabbitRetry;
import com.digiwin.athena.framework.mq.retry.context.MQRetryContextHolder;
import com.digiwin.athena.framework.mq.retry.exception.OverLimitException;
import com.digiwin.athena.framework.mq.retry.exception.RejectMQException;
import com.digiwin.athena.framework.mq.retry.exception.RetryMQException;
import com.digiwin.athena.framework.mq.retry.support.ExecutorServiceProvider;
import com.digiwin.athena.framework.mq.retry.support.RabbitMqHandlerMethodArgumentResolver;
import com.digiwin.athena.framework.mq.retry.support.RabbitRetryMethod;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.Nullable;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.util.StringUtils;

import java.util.Optional;


@Slf4j
@Aspect
@ConditionalOnProperty(prefix = "athena.mq.retry", name = "enable", havingValue = "true")
public class RabbitRetryInterceptor extends AbstractInterceptor {

    public RabbitRetryInterceptor(RabbitTemplate rabbitTemplate, RabbitMqRetryProperties retryProperties, @Nullable ExecutorServiceProvider provider) {
        super(rabbitTemplate, retryProperties, provider);
    }

    @Override
    protected Object processMessage(ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding, RabbitRetry rabbitRetry) throws Throwable {
        try {
            Object result = joinPoint.proceed();
            executeSuccessHandler(retryMethod);
            return result;
        } catch (Throwable e) {
            handleProcessingException(e, queueBinding, rabbitRetry, joinPoint, retryMethod);
        }
        return null;
    }

    @Override
    protected void validateQueueBinding(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding) {
        if (StringUtils.isEmpty(queueBinding.getExchangeName()) || StringUtils.isEmpty(queueBinding.getRoutingkey())) {
            throw new RejectMQException("RabbitRetryInterceptor Exchange or Routing Key not found");
        }
    }

    @Override
    protected void validateRetryCount(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding) {
        Integer retryCount = Optional.ofNullable(queueBinding.getRetryCount()).orElse(0);
        if (retryCount >= retryProperties.getMaxMQRetryCount()) {
            throw new OverLimitException("RabbitRetryInterceptor Retry count exceeds maximum limit");
        }
    }

    private void handleProcessingException(Throwable e, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBinding, RabbitRetry rabbitRetry, ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod) throws Throwable {

        Class<? extends Throwable>[] rejectClasses = rabbitRetry.rejectForException();
        Class<? extends Throwable>[] retryClasses = rabbitRetry.retryForMQExceptions();

//        executeFailureMethod(retryMethod, e);

        //默认快速失败
        if (e instanceof RejectMQException || isExceptionInList(e, rejectClasses)) {
//            executeFailureMethod(retryMethod, e);
            executeFailureMethod(retryMethod, e);
            return;
        }
        //mq重试
        if (e instanceof RetryMQException || isExceptionInList(e, retryClasses)) {
            //校验重试次数
//            validateRetryCount(queueBinding);
            executeFailureMethod(retryMethod, e);
            // 重新放回队列并添加消息头信息
            retryWithBackToQueue(queueBinding);
            return;
        }
        //本地重试
        handleRetry(joinPoint, retryMethod, rabbitRetry, queueBinding);
    }

    private void retryWithBackToQueue(RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBindingBean) {
//        MQRetryContextHolder.MQRetryContext mqRetryContext = MQRetryContextHolder.getContext();
//        mqRetryContext.setMqRetrycount(queueBindingBean.getRetryCount() + 1);

        rabbitTemplate.convertAndSend(queueBindingBean.getExchangeName(), queueBindingBean.getRoutingkey(), queueBindingBean.getMessage(), message -> {
            MessageProperties messageProperties = message.getMessageProperties();
            int retryCount = Optional.ofNullable(queueBindingBean.getRetryCount()).orElse(0);
            messageProperties.setHeader("retry-count", retryCount + 1);
            return message;
        });
    }

    private void handleRetry(ProceedingJoinPoint joinPoint, RabbitRetryMethod retryMethod, RabbitRetry rabbitRetry, RabbitMqHandlerMethodArgumentResolver.QueueBindingBean queueBindingBean) throws Throwable {
        RetryTemplate retryTemplate = createRetryTemplate(rabbitRetry);
        retryTemplate.execute(context -> {
            try {
                Object result = joinPoint.proceed();
                //添加失败次数
                MQRetryContextHolder.getContext().setLocalRetrycount(context.getRetryCount() + 1);
                executeSuccessHandler(retryMethod);
                return result;
            } catch (Throwable e) {
                throw e;
            }
        }, context -> {
            MQRetryContextHolder.getContext().setLocalRetrycount(context.getRetryCount() + 1);
//            MQRetryContextHolder.getContext().setLastException(context.getLastThrowable());
            executeFailureMethod(retryMethod, context.getLastThrowable());
            return null;
        });
    }

    private boolean isExceptionInList(Throwable e, Class<? extends Throwable>[] throwableClasses) {
        for (Class<? extends Throwable> clazz : throwableClasses) {
            if (clazz.isAssignableFrom(e.getClass())) {
                return true;
            }
        }
        return false;
    }

}

