package com.digiwin.athena.framework.rw;

import com.alibaba.druid.DbType;
import com.alibaba.druid.util.JdbcConstants;
import com.digiwin.athena.framework.rw.dto.ReadWriterDto;
import com.digiwin.athena.framework.rw.router.DataSourRouter;
import com.digiwin.athena.framework.rw.router.DbSwitchConfig;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.executor.statement.StatementHandler;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.reflection.SystemMetaObject;

import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Arrays;

@Data
@Slf4j
@Intercepts(@Signature(type = StatementHandler.class, method = "prepare", args = {Connection.class, Integer.class}))
public class ShardPlugin implements Interceptor {

    private DataSourRouter dataSourRouter;

    private DbSwitchConfig dbSwitchConfig;

    public ShardPlugin(DataSourRouter dataSourRouter, DbSwitchConfig dbSwitchConfig) {
        this.dataSourRouter = dataSourRouter;
        this.dbSwitchConfig = dbSwitchConfig;
    }

    public static final DbType DB_TYPE = JdbcConstants.MYSQL;

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        Object target = realTarget(invocation.getTarget());
        MetaObject metaObject = SystemMetaObject.forObject(target);
        try {
            MappedStatement mappedStatement = (MappedStatement) metaObject.getValue("delegate.mappedStatement");
            SqlCommandType sqlCommandType = mappedStatement.getSqlCommandType();

            if (sqlCommandType == SqlCommandType.SELECT) {
                processRead(metaObject);
            } else if (Arrays.asList(SqlCommandType.INSERT, SqlCommandType.UPDATE, SqlCommandType.DELETE).contains(sqlCommandType)) {
                processWrite(metaObject, invocation);
            }
        } catch (Exception e) {
            log.error("[shard-plugin] ERROR", e);
        }

        return invocation.proceed();
    }

    private void processRead(MetaObject metaObject) {
        ShardProcessor shardProcessor = new ShardProcessor(metaObject, dbSwitchConfig);
        shardProcessor.route();
        ReadWriterDto readWriterDto = new ReadWriterDto(true, dbSwitchConfig.getReadMode());
        shardProcessor.processLocal(readWriterDto);
    }

    private void processWrite(MetaObject metaObject, Invocation invocation) throws SQLException {
        ShardProcessor shardProcessor = new ShardProcessor(metaObject, dbSwitchConfig);
        shardProcessor.processParams();
        ReadWriterDto readWriterDto = new ReadWriterDto(false, dbSwitchConfig.getWriteMode());
        shardProcessor.processLocal(readWriterDto);
        shardProcessor.processWrite(dataSourRouter.getNewDs().getConnection());
    }

    private Object handleDualWrite(Invocation invocation, MetaObject metaObject) throws Throwable {
        Object result = null;
        result = invocation.proceed();

        try {
            invocation.proceed();
        } catch (Exception e) {
            log.warn("[shard-plugin] 双写 second 数据源失败：{}", e.getMessage());
        }
        return result;
    }

    /**
     * 获得真正的处理对象,可能多层代理
     */
    @SuppressWarnings("unchecked")
    private <T> T realTarget(Object target) {
        if (Proxy.isProxyClass(target.getClass())) {
            MetaObject metaObject = SystemMetaObject.forObject(target);
            return realTarget(metaObject.getValue("h.target"));
        }
        return (T) target;
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }
}

