package com.digiwin.athena.framework.rw;

import com.alibaba.druid.sql.SQLUtils;
import com.alibaba.druid.sql.ast.SQLStatement;
import com.alibaba.druid.sql.visitor.SchemaStatVisitor;
import com.alibaba.druid.stat.TableStat;
import com.alibaba.fastjson.JSON;
import com.digiwin.athena.framework.rw.contants.ReadType;
import com.digiwin.athena.framework.rw.contants.WriteType;
import com.digiwin.athena.framework.rw.dto.ReadWriterDto;
import com.digiwin.athena.framework.rw.exception.MyBatisShardException;
import com.digiwin.athena.framework.rw.router.DataSourRouter;
import com.digiwin.athena.framework.rw.router.DbSwitchConfig;
import com.digiwin.athena.framework.rw.router.MySqlReplaceTableNameVisitor;
import com.digiwin.athena.framework.rw.strategy.AbstractShardStrategy;
import com.digiwin.athena.framework.rw.strategy.ShardStrategyContext;
import com.digiwin.athena.framework.rw.utils.CommonUtils;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.executor.parameter.ParameterHandler;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.scripting.defaults.DefaultParameterHandler;
import org.springframework.util.Assert;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

@Slf4j
public class ShardProcessor {

    private final MetaObject metaObject;

    private DbSwitchConfig dbSwitchConfig;
    private final BoundSql boundSql;
    private final MappedStatement mappedStatement;
    private final String originalSql;
    private final String shardSql;
//    private String tableName;
//    private ReadType readType = ReadType.OLD;
//    private WriteType writeType = WriteType.OLD;

    public ShardProcessor(@NonNull MetaObject metaObject, DbSwitchConfig dbSwitchConfig) {
        this.dbSwitchConfig = dbSwitchConfig;
        this.metaObject = metaObject;
        this.boundSql = (BoundSql) metaObject.getValue("delegate.boundSql");
        this.mappedStatement = (MappedStatement) metaObject.getValue("delegate.mappedStatement");
        this.originalSql = boundSql.getSql();
        this.shardSql = calculateShardSql();
    }

    private String calculateShardSql() {
        List<SQLStatement> stmtList = getSqlStatementList();
        for (SQLStatement stmt : stmtList) {
            stmt.accept(new MySqlReplaceTableNameVisitor(boundSql, dbSwitchConfig));
        }
        return SQLUtils.toMySqlString(stmtList.get(0));
    }

    public void route() {
        ReadType readType = ReadType.valueOfKey(dbSwitchConfig.getReadMode());
        switch (readType) {
            case OLD:
                return;
            case NEW:
                setShardSql();
                return;
            default:
                throw new MyBatisShardException("未知ReadType：" + dbSwitchConfig.getReadMode());
        }
    }

    private void setShardSql() {
        metaObject.setValue("delegate.boundSql.sql", shardSql);
    }

    public void processParams() {
        SQLStatement stmt = getSqlStatement();
        // 统计SQL中使用的表、字段、过滤条件、排序表达式、分组表达式等
        SchemaStatVisitor schemaStatVisitor = SQLUtils.createSchemaStatVisitor(ShardPlugin.DB_TYPE);
        stmt.accept(schemaStatVisitor);
        // 表名
        Map<TableStat.Name, TableStat> tables = schemaStatVisitor.getTables();

        for (TableStat.Name name : tables.keySet()) {
            AbstractShardStrategy shardStrategy = ShardStrategyContext.getStrategyByTableName(name.getName());
            shardStrategy.processParams(metaObject, boundSql, schemaStatVisitor);
        }
    }

//    public void processBefore(ReadWriterDto readWriterDto) {
//        AbstractShardStrategy shardStrategy = ShardStrategyContext.getStrategyByTableName(tableName);
//        shardStrategy.processBefore(readWriterDto);
//    }

    private List<SQLStatement> getSqlStatementList() {
        List<SQLStatement> stmtList = SQLUtils.parseStatements(originalSql, ShardPlugin.DB_TYPE);
        Assert.notEmpty(stmtList, "stmtList is empty, sql: " + originalSql);
        return stmtList;
    }

    private SQLStatement getSqlStatement() {
        List<SQLStatement> stmtList = getSqlStatementList();
        Assert.notEmpty(stmtList, "stmtList is empty, sql: " + originalSql);
        return stmtList.get(0);
    }

    public void processWrite(DataSourRouter dataSourRouter) throws SQLException {
        WriteType writeType = WriteType.valueOfKey(dbSwitchConfig.getWriteMode());
        switch (writeType) {
            case OLD:
//                setShardSql();
                return;
            case NEW:
                setShardSql();
                return;
            case NEWOLD:
                setShardSql();
                Connection oldConnection = dataSourRouter.getOldDs().getConnection();
                writeShard(oldConnection, originalSql);
                return;
            case OLDNEW:
                Connection newConnection = dataSourRouter.getNewDs().getConnection();
                writeShard(newConnection, shardSql);
                return;
            default:
                throw new MyBatisShardException("未知WriteType：" + dbSwitchConfig.getWriteMode());
        }
    }

    private void writeShard(Connection connection, String sql) {
        Object parameterObject = boundSql.getParameterObject();
        log.info("[rw-plugin] double write sql: {} \n  parameterObject({}): {}", CommonUtils.removeBreakingWhitespace(sql), parameterObject.getClass().getSimpleName(), JSON.toJSONString(parameterObject));
        try (PreparedStatement statement = connection.prepareStatement(sql)) {
            ParameterHandler parameterHandler = new DefaultParameterHandler(mappedStatement, parameterObject, boundSql);
            parameterHandler.setParameters(statement);
            statement.executeUpdate();
        } catch (Exception e) {
            throw new MyBatisShardException(String.format("Error: Method ShardPlugin.write execution error of sql : \n %s \n", CommonUtils.removeBreakingWhitespace(shardSql)), e);
        }
    }
}
