package com.digiwin.athena.framework.rw.router;

import javax.sql.DataSource;

import com.digiwin.athena.framework.rw.RWSetUpConfigUpdater;
import com.digiwin.athena.framework.rw.RWTypeInterceptor;
import com.digiwin.athena.framework.rw.ShardPlugin;
import lombok.Data;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.List;


@Data
@Configuration
@EnableConfigurationProperties(DbSwitchConfig.class)
@ConditionalOnProperty(prefix = "athena.dbswitch", name = "enable", havingValue = "true", matchIfMissing = false)
public class DataSourRouterConfig {

    private ApplicationContext applicationContext;
    private DbSwitchConfig dbSwitchConfig;

    public DataSourRouterConfig(ApplicationContext applicationContext,
                                DbSwitchConfig dbSwitchConfig) {
        this.applicationContext = applicationContext;
        this.dbSwitchConfig = dbSwitchConfig;
    }


    @Bean
    public RWTypeInterceptor rwTypeInterceptor(DataSourRouter dataSourRouter, DbSwitchConfig dbSwitchConfig) {
        return new RWTypeInterceptor(dataSourRouter, dbSwitchConfig);
    }

    @Bean
    public ShardPlugin shardPlugin(DataSourRouter dataSourRouter, DbSwitchConfig dbSwitchConfig) {
        return new ShardPlugin(dataSourRouter, dbSwitchConfig);
    }


    @Bean
    public MybatisInterceptorInit mybatisInterceptorInit(List<SqlSessionFactory> sqlSessionFactoryList, ShardPlugin shardPlugin, RWTypeInterceptor rwTypeInterceptor) {
        return new MybatisInterceptorInit(sqlSessionFactoryList, shardPlugin, rwTypeInterceptor);
    }

    @Bean(name = "dataSourRouter")
    public DataSourRouter dataSourRouter() {
        String oldName = dbSwitchConfig.getOldDatasourceName();
        String newName = dbSwitchConfig.getNewDatasourceName();

        DataSource oldDs = getDataSourceByBeanName(oldName);
        DataSource newDs = getDataSourceByBeanName(newName);

        DataSourRouter dataSourRouter = new DataSourRouter(oldDs, newDs);
        return dataSourRouter;
    }

    @Bean
    public RWSetUpConfigUpdater rwSetUpConfigUpdater() {
        return new RWSetUpConfigUpdater();
    }

    private DataSource getDataSourceByBeanName(String beanName) {
        if (!applicationContext.containsBean(beanName)) {
            throw new IllegalArgumentException("数据源Bean不存在: " + beanName);
        }
        Object bean = applicationContext.getBean(beanName);
        if (!(bean instanceof DataSource)) {
            throw new IllegalArgumentException("Bean不是DataSource类型: " + beanName);
        }
        return (DataSource) bean;
    }

}
