package com.digiwin.athena.framework.snowflake.listener;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.athena.framework.snowflake.handler.AthenaWorkerIdHandler;
import com.digiwin.athena.framework.snowflake.exception.InstanceNotFoundException;
import com.digiwin.athena.framework.snowflake.exception.SnowflakeInitException;
import com.digiwin.athena.framework.snowflake.exception.WorkerIdExhaustedException;
import com.jugg.agile.framework.core.config.JaEnvProperty;
import com.jugg.agile.middleware.redis.JaRedisLock;
import com.jugg.agile.middleware.redis.meta.JaRedisLockEntity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.EventListener;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Nacos服务注册事件监听器
 * 在应用启动时自动分配雪花算法的工作节点ID
 * 关键功能：
 * 1. 使用分布式锁保证并发安全
 * 2. 基于Nacos元数据分配唯一workerId
 * 3. 支持0-63共64个工作节点
 *
 * @author wzq
 */
@Slf4j
public class AthenaRegistrationEventListener {
    // 元数据键名（存储workerId）
    private static final String WORKER_ID_KEY = "snowflake.worker.id";
    // 分布式锁键名
    private static final String LOCK_KEY_PREFIX = "athena:snowflake:init:workId:";
    private static final String LOCK_KEY_FORMAT = "%s:%s";

    // 工作节点ID范围常量
    private static final int MIN_WORKER_ID = 0;
    private static final int MAX_WORKER_ID = 63;
    private static final int INVALID_WORKER_ID = -1;

    private final NacosDiscoveryProperties discoveryProperties;
    private final NacosServiceManager nacosServiceManager;

    public AthenaRegistrationEventListener(NacosDiscoveryProperties discoveryProperties, NacosServiceManager nacosServiceManager) {
        this.discoveryProperties = discoveryProperties;
        this.nacosServiceManager = nacosServiceManager;
    }

    /**
     * 应用启动完成事件处理
     * 在Spring Boot应用就绪后触发工作节点分配
     */
    @EventListener(ApplicationReadyEvent.class)
    public void onApplicationReady() {
        try {
            processWorkerIdAssignment();
        } catch (Exception e) {
            log.error("雪花算法工作节点分配失败", e);
            throw new SnowflakeInitException("雪花算法初始化失败", e);
        }
    }

    /**
     * 工作节点分配主流程
     */
    private void processWorkerIdAssignment() {
        NamingService namingService = getNamingService();
        Instance currentInstance = getCurrentInstance(namingService);

        if (currentInstance == null) {
            throw new InstanceNotFoundException("当前服务实例不存在");
        }

        // 使用分布式锁保证原子操作
        String lockKey = String.format(LOCK_KEY_FORMAT, LOCK_KEY_PREFIX, JaEnvProperty.getApplicationName());
        JaRedisLock.lock(JaRedisLockEntity.builder()
                        .lockKey(lockKey)
                        .build(),
                () -> {
                    int workerId = assignWorkerId(namingService, currentInstance);
                    AthenaWorkerIdHandler.INSTANCE.setWorkerId(workerId);
                    log.info("雪花算法工作节点设置完成: {}", workerId);
                    return null;
                });
    }

    /**
     * 获取NamingService实例
     */
    private NamingService getNamingService() {
        return nacosServiceManager.getNamingService(discoveryProperties.getNacosProperties());
    }

    /**
     * 获取当前服务实例
     */
    private Instance getCurrentInstance(NamingService namingService) {
        String currentIp = discoveryProperties.getIp();
        int currentPort = discoveryProperties.getPort();

        return getAllInstances(namingService).stream()
                .filter(instance ->
                        currentIp.equals(instance.getIp()) &&
                                currentPort == instance.getPort()
                )
                .findFirst()
                .orElse(null);
    }

    /**
     * 分配工作节点ID
     */
    private int assignWorkerId(NamingService namingService, Instance instance)
            throws NacosException {
        int workerId = findAvailableWorkerId(namingService);
        updateInstanceMetadata(instance, workerId);
        registerInstance(namingService, instance);
        return workerId;
    }

    /**
     * 查找可用工作节点ID
     */
    private int findAvailableWorkerId(NamingService namingService) {
        Set<Integer> usedIds = getUsedWorkerIds(namingService);
        return findNextAvailableId(usedIds)
                .orElseThrow(() -> new WorkerIdExhaustedException("无可用工作节点ID"));
    }

    /**
     * 获取已使用的WorkerID集合
     */
    private Set<Integer> getUsedWorkerIds(NamingService namingService) {
        return getAllInstances(namingService).stream()
                .map(Instance::getMetadata)
                .map(metadata -> metadata.get(WORKER_ID_KEY))
                .map(this::safeParseInt)
                .filter(id -> id >= MIN_WORKER_ID)
                .collect(Collectors.toSet());
    }

    /**
     * 安全的整数转换
     */
    private int safeParseInt(String value) {
        try {
            return (value != null) ? Integer.parseInt(value) : INVALID_WORKER_ID;
        } catch (NumberFormatException e) {
            log.warn("无效的WorkerID格式: {}", value);
            return INVALID_WORKER_ID;
        }
    }

    /**
     * 查找下一个可用ID
     */
//    private Optional<Integer> findNextAvailableId(Set<Integer> usedIds) {
//        // 从0到最大ID遍历寻找可用ID
//        for (int id = MIN_WORKER_ID; id <= MAX_WORKER_ID; id++) {
//            if (!usedIds.contains(id)) {
//                log.info("分配工作节点ID: {}", id);
//                return Optional.of(id);
//            }
//        }
//        return Optional.empty();
//    }

    /**
     * 查找下一个可用ID
     */
    private Optional<Integer> findNextAvailableId(Set<Integer> usedIds) {
        for (int id = MAX_WORKER_ID; id >= MIN_WORKER_ID; id--) {
            if (!usedIds.contains(id)) {
                log.info("分配工作节点ID: {}", id);
                return Optional.of(id);
            }
        }
        return Optional.empty();
    }

    /**
     * 更新实例元数据
     */
    private void updateInstanceMetadata(Instance instance, int workerId) {
        instance.getMetadata().put(WORKER_ID_KEY, String.valueOf(workerId));
    }

    /**
     * 注册服务实例
     */
    private void registerInstance(NamingService namingService, Instance instance)
            throws NacosException {
        namingService.registerInstance(
                discoveryProperties.getService(),
                discoveryProperties.getGroup(),
                instance
        );
    }

    /**
     * 获取所有服务实例
     */
    private List<Instance> getAllInstances(NamingService namingService) {
        try {
            return namingService.getAllInstances(
                    discoveryProperties.getService(),
                    discoveryProperties.getGroup()
            );
        } catch (NacosException e) {
            log.error("获取Nacos服务实例失败", e);
            return Collections.emptyList();
        }
    }
}






