package com.digiwin.athena.framework.snowflake.utils;

import com.digiwin.athena.framework.snowflake.enums.DefaultBizKey;
import com.digiwin.athena.framework.snowflake.enums.SnowflakeBizKey;
import com.digiwin.athena.framework.snowflake.handler.AthenaWorkerIdHandler;
import com.jugg.agile.framework.core.util.algorithm.id.snowflake.JaSnowflakeId;
import com.jugg.agile.framework.core.util.algorithm.id.snowflake.JaSnowflakeIdConfig;

/**
 * 雪花算法入口类
 * 提供全局访问点获取不同业务的ID生成器
 *
 * @author wzq
 */
public class AthenaSnowflake {

    /**
     * 获取指定业务键的雪花算法实例
     *
     * @param bizKey 业务键（枚举值）
     * @return 配置好的雪花算法实例
     * @throws IllegalArgumentException 如果业务键未注册
     */
    public static JaSnowflakeId get(SnowflakeBizKey bizKey) {
        JaSnowflakeId generator = bizKey.getSnowflakeId();
        if (generator == null) {
            throw new IllegalArgumentException("未注册的业务键: " + bizKey);
        }
        return generator;
    }


    /**
     * 快速生成分布式ID
     *
     * @param bizKey 业务键（枚举值）
     * @return 生成的64位唯一ID
     */
    public static long nextId(SnowflakeBizKey bizKey) {
        return get(bizKey).get();
    }


    /**
     * 创建自定义起始时间的配置
     *
     * @param beginTimeStr 起始时间字符串（格式：yyyy-MM-dd HH:mm:ss）
     */
    public static JaSnowflakeIdConfig create(String beginTimeStr) {
        JaSnowflakeIdConfig config = new JaSnowflakeIdConfig();
        config.setBeginTimeStr(beginTimeStr);
        config.setSequenceNumberBit(6);
        config.setTimeUnit(1);
        return config;
    }

    /**
     * 创建模块A的默认配置（适合ptm系统）
     * 起始时间：2020-12-20 20:20:20
     */
    public static JaSnowflakeIdConfig createDefaultA() {
        JaSnowflakeIdConfig config = new JaSnowflakeIdConfig();
        config.setBeginTimeStr("2020-12-20 20:20:20");
        config.setSequenceNumberBit(6);
        config.setTimeUnit(1);
        return config;
    }

    /**
     * 创建模块B的默认配置（适合小ai atmc等系统）
     * 起始时间：2018-12-16 21:08:16
     */
    public static JaSnowflakeIdConfig createDefaultB() {
        JaSnowflakeIdConfig config = new JaSnowflakeIdConfig();
        config.setBeginTimeStr("2018-12-16 21:08:16");
        config.setSequenceNumberBit(6);
        config.setTimeUnit(1);
        return config;
    }

    public static JaSnowflakeId newInstance(JaSnowflakeIdConfig config) {
        return JaSnowflakeId.newInstance(config, AthenaWorkerIdHandler.INSTANCE);
    }

    public static JaSnowflakeId newInstanceA() {
        return JaSnowflakeId.newInstance(createDefaultA(), AthenaWorkerIdHandler.INSTANCE);
    }

    public static JaSnowflakeId newInstanceB() {
        return JaSnowflakeId.newInstance(createDefaultB(), AthenaWorkerIdHandler.INSTANCE);
    }

    // ===================== 使用示例 =====================
    public static void main(String[] args) {
        // 设置工作节点ID（实际由监听器自动设置）
        AthenaWorkerIdHandler.INSTANCE.setWorkerId(0);

        // 生成模块A的ID
        System.out.println(AthenaSnowflake.nextId(DefaultBizKey.DEFAULT_MODULE_A));
        // 生成模块B的ID
        System.out.println(AthenaSnowflake.nextId(DefaultBizKey.DEFAULT_MODULE_B));
    }
}