package com.digiwin.athena.framework.snowflake.example;

import com.jugg.agile.framework.core.util.algorithm.id.snowflake.JaSnowflakeId;

import java.util.Arrays;
import java.util.Optional;

import static com.digiwin.athena.framework.snowflake.example.AthenaSnowflake.createDefaultA;
import static com.digiwin.athena.framework.snowflake.example.AthenaSnowflake.createDefaultB;

/**
 * 默认业务键枚举，实现SnowflakeBizKey接口
 * 为不同业务模块提供预配置的雪花算法实例
 *
 * @author wzq
 */
public enum DefaultSnowflakeId implements SnowflakeBizKey {
    DEFAULT_MODULE_A("default_module_a", AthenaSnowflake.newInstance(createDefaultA())),
    DEFAULT_MODULE_B("default_module_b", AthenaSnowflake.newInstance(createDefaultB()));

    private final String keyName;
    private final JaSnowflakeId snowflakeId;

    /**
     * 枚举构造函数
     *
     * @param keyName     业务键名称
     * @param snowflakeId 对应的雪花算法实例
     */
    DefaultSnowflakeId(String keyName, JaSnowflakeId snowflakeId) {
        this.keyName = keyName;
        this.snowflakeId = snowflakeId;
    }

    @Override
    public String getKeyName() {
        return keyName;
    }

    @Override
    public JaSnowflakeId getSnowflakeId() {
        return snowflakeId;
    }

    /**
     * 根据keyName查找对应的SnowflakeId
     *
     * @param keyName 业务键名称
     * @return 对应的SnowflakeId，找不到返回Optional.empty()
     */
    public static Optional<JaSnowflakeId> findByKeyName(String keyName) {
        return Arrays.stream(values())
                .filter(e -> e.getKeyName().equals(keyName))
                .findFirst()
                .map(DefaultSnowflakeId::getSnowflakeId);
    }

    /**
     * 根据keyName查找SnowflakeId，找不到时返回预定义的默认实例A
     *
     * @param keyName 业务键名称
     * @return 找到的SnowflakeId或InstanceA
     */
    public static JaSnowflakeId findByKeyNameOrDefault(String keyName) {
        return findByKeyName(keyName).orElse(AthenaSnowflake.newInstanceA());
    }

    /**
     * 根据keyName查找SnowflakeId，并获取id
     *
     * @param keyName 业务键名称
     * @return 找到id
     */
    public static Long nextId(String keyName) {
        return findByKeyNameOrDefault(keyName).get();
    }

    /**
     * 根据bizKey查找SnowflakeId，并获取id
     *
     * @param bizKey 业务键名称
     * @return 找到id
     */
    public static Long nextId(SnowflakeBizKey bizKey) {
        return findByEnumDefault(bizKey).get();
    }

    /**
     * 根据bizKey查找SnowflakeId，找不到时返回预定义的默认实例A
     *
     * @param bizKey 业务键名称
     * @return 找到的SnowflakeId或InstanceA
     */
    public static JaSnowflakeId findByEnumDefault(SnowflakeBizKey bizKey) {
        return findEnum(bizKey).orElse(AthenaSnowflake.newInstanceA());
    }

    /**
     * 根据bizKey查找对应的SnowflakeId
     *
     * @param bizKey 业务键名称
     * @return 对应的SnowflakeId，找不到返回Optional.empty()
     */
    public static Optional<JaSnowflakeId> findEnum(SnowflakeBizKey bizKey) {
        return Arrays.stream(values())
                .filter(e -> e.getKeyName().equals(bizKey.getKeyName()))
                .findFirst()
                .map(DefaultSnowflakeId::getSnowflakeId);
    }
}