package com.digiwin.athena.framework.snowflake.example;

import com.digiwin.athena.framework.snowflake.AthenaWorkerIdAdapter;
import com.jugg.agile.framework.core.util.algorithm.id.snowflake.JaSnowflakeId;
import lombok.Getter;

import java.util.Arrays;
import java.util.Optional;

import static com.digiwin.athena.framework.snowflake.AthenaSnowflakeIdConfigUtil.createDefaultA;
import static com.digiwin.athena.framework.snowflake.AthenaSnowflakeIdConfigUtil.createDefaultB;


/**
 * 将所有业务放一个枚举中方便管理
 *
 * @author wzq
 */
@Getter
public enum DefaultEnumSnowflakeId {
    DEFAULT_MODULE_A("default_module_a", JaSnowflakeId.newInstance(createDefaultA(), AthenaWorkerIdAdapter.INSTANCE)),
    DEFAULT_MODULE_B("default_module_b", JaSnowflakeId.newInstance(createDefaultB(), AthenaWorkerIdAdapter.INSTANCE));

    private final String keyName;
    private final JaSnowflakeId snowflakeId;

    /**
     * 枚举构造函数
     *
     * @param keyName     业务键名称
     * @param snowflakeId 对应的雪花算法实例
     */
    DefaultEnumSnowflakeId(String keyName, JaSnowflakeId snowflakeId) {
        this.keyName = keyName;
        this.snowflakeId = snowflakeId;
    }

    /**
     * 根据keyName查找对应的SnowflakeId
     *
     * @param keyName 业务键名称
     * @return 对应的SnowflakeId，找不到返回Optional.empty()
     */
    public static Optional<JaSnowflakeId> findByKeyName(String keyName) {
        return Arrays.stream(values())
                .filter(e -> e.getKeyName().equals(keyName))
                .findFirst()
                .map(DefaultEnumSnowflakeId::getSnowflakeId);
    }

    /**
     * 根据keyName查找SnowflakeId，找不到时返回预定义的默认实例A
     *
     * @param keyName 业务键名称
     * @return 找到的SnowflakeId或InstanceA
     */
    public static JaSnowflakeId findByKeyNameOrDefault(String keyName) {
        return findByKeyName(keyName).orElse(JaSnowflakeId.newInstance(createDefaultA(), AthenaWorkerIdAdapter.INSTANCE));
    }

    /**
     * 根据keyName查找SnowflakeId，并获取id
     *
     * @param keyName 业务键名称
     * @return 找到id
     */
    public static Long nextId(String keyName) {
        return findByKeyNameOrDefault(keyName).get();
    }

    /**
     * 根据bizKey查找SnowflakeId，并获取id
     *
     * @param bizKey 业务键名称
     * @return 找到id
     */
    public static Long nextId(DefaultEnumSnowflakeId bizKey) {
        return findByEnumDefault(bizKey).get();
    }

    /**
     * 根据bizKey查找SnowflakeId，找不到时返回预定义的默认实例A
     *
     * @param bizKey 业务键名称
     * @return 找到的SnowflakeId或InstanceA
     */
    public static JaSnowflakeId findByEnumDefault(DefaultEnumSnowflakeId bizKey) {
        return findEnum(bizKey).orElse(JaSnowflakeId.newInstance(createDefaultA(), AthenaWorkerIdAdapter.INSTANCE));
    }

    /**
     * 根据bizKey查找对应的SnowflakeId
     *
     * @param bizKey 业务键名称
     * @return 对应的SnowflakeId，找不到返回Optional.empty()
     */
    public static Optional<JaSnowflakeId> findEnum(DefaultEnumSnowflakeId bizKey) {
        return Arrays.stream(values())
                .filter(e -> e.getKeyName().equals(bizKey.getKeyName()))
                .findFirst()
                .map(DefaultEnumSnowflakeId::getSnowflakeId);
    }
}