package com.digiwin.athena.framework.snowflake;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.fastjson.JSON;
import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
public class DwSnowflakeNacosProcessor {
    private static NacosDiscoveryProperties discoveryProperties;
    private static NacosServiceManager nacosServiceManager;

    public static NacosDiscoveryProperties getDiscoveryProperties() {
        return discoveryProperties;
    }

    /**
     * 确保依赖可用
     */
    public static void checkRegistered() {
        if (discoveryProperties == null) {
            discoveryProperties = JaSpringBeanUtil.getBean(NacosDiscoveryProperties.class);
        }
        if (nacosServiceManager == null) {
            nacosServiceManager = JaSpringBeanUtil.getBean(NacosServiceManager.class);
        }

        if (discoveryProperties == null || nacosServiceManager == null) {
            throw new BusinessException("Nacos依赖未找到，请检查配置");
        }
    }

    /**
     * 获取NamingService实例
     */
    public static NamingService getNamingService() {
        return nacosServiceManager.getNamingService(discoveryProperties.getNacosProperties());
    }

    /**
     * 获取服务名称
     */
    public static String getServiceName() {
        return discoveryProperties.getService();
    }

    /**
     * 获取分组名称
     */
    public static String getGroupName() {
        return discoveryProperties.getGroup();
    }

    // ===================== 实例操作 =====================

    /**
     * 获取当前服务实例
     */
    public static Instance getCurrentInstance(NamingService namingService) {
        String currentIp = discoveryProperties.getIp();
        int currentPort = discoveryProperties.getPort();
        log.info("currentIp:{},currentPort:{}", currentIp, currentPort);
        List<Instance> allInstances = getAllInstances(namingService);
        log.info("allInstances:{}", JSON.toJSONString(allInstances));

        Map<String, Instance> instanceMap = allInstances.stream()
                .collect(Collectors.toMap(
                        DwSnowflakeNacosProcessor::getInstanceKey,
                        Function.identity(),
                        (existing, replacement) -> existing
                ));

        String currentKey = createInstanceKey(currentIp, currentPort);
        return instanceMap.get(currentKey);
    }

    /**
     * 获取所有服务实例
     */
    public static List<Instance> getAllInstances(NamingService namingService) {
        try {
            return namingService.getAllInstances(getServiceName(), getGroupName(), new ArrayList<>(), false);
        } catch (NacosException e) {
            JaLog.error("获取Nacos服务实例失败", e);
            return Collections.emptyList();
        }
    }

    /**
     * 创建实例唯一键
     */
    public static String createInstanceKey(String ip, int port) {
        return ip + ":" + port;
    }

    /**
     * 获取实例唯一键
     */
    public static String getInstanceKey(Instance instance) {
        return instance.getIp() + ":" + instance.getPort();
    }
}
