package com.digiwin.athena.framework.snowflake;

import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.jugg.agile.framework.core.config.JaEnvProperty;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.util.algorithm.id.snowflake.WorkerIdConfig;
import com.jugg.agile.middleware.redis.JaRedisLock;
import com.jugg.agile.middleware.redis.meta.JaRedisLockEntity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.event.EventListener;

import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Nacos服务注册事件监听器
 * <p>
 * 核心功能：
 * 1. 在应用启动时自动分配雪花算法的工作节点ID
 * 2. 使用分布式锁保证并发安全
 * 3. 基于Nacos元数据分配唯一workerId
 * 4. 支持0-63共64个工作节点
 * 5. 提供本地开发模式支持
 *
 * @author wzq
 */
@Slf4j
public class DwWorkerIdInitializer {
    // ===================== 常量定义 =====================
    private static final String WORKER_ID_KEY = "snowflake.worker.id";
    private static final String LOCK_KEY_PATTERN = "athena:snowflake:init:workId:%s";
    private static final int MIN_WORKER_ID = 0;
    private static final int MAX_WORKER_ID = 63;
    private static final int INVALID_WORKER_ID = -1;

    // ===================== 配置键名 =====================
    private static final String WORKER_ID_BIT_KEY = "athena.snowflake.workerIdBit";
    private static final String LOCAL_WORKER_ID_KEY = "athena.snowflake.local.worker-id";

    // ===================== 核心方法 =====================

    /**
     * 应用启动完成事件处理
     * 在Spring Boot应用就绪后触发工作节点分配
     */
    @EventListener(ApplicationReadyEvent.class)
    public void onApplicationReady() {
        if (Boolean.FALSE.equals(JaProperty.getBoolean("athena.snowflake.enable", true))) {
            log.info("雪花算法未启用，跳过初始化");
            return;
        }

        try {
            WorkerIdConfig workerIdConfig = resolveWorkerIdConfig();
            DwWorkerIdAdapter.INSTANCE = () -> workerIdConfig;
            log.info("雪花算法工作节点配置完成");
        } catch (Exception e) {
            log.error("雪花算法初始化失败", e);
            throw new BusinessException("雪花算法初始化失败", e);
        }
    }

    // ===================== 配置解析逻辑 =====================

    /**
     * 解析工作节点配置
     */
    private WorkerIdConfig resolveWorkerIdConfig() {
        if (JaEnvProperty.isLocal()) {
            return createLocalWorkerConfig();
        }
        return createCloudWorkerConfig();
    }

    /**
     * 创建本地环境配置
     */
    private WorkerIdConfig createLocalWorkerConfig() {
        int workerId = JaProperty.getInteger(LOCAL_WORKER_ID_KEY, 0);
        long workerIdBit = JaProperty.getLong(WORKER_ID_BIT_KEY, 6L);
        log.info("本地环境使用固定工作节点ID: {}", workerId);
        return WorkerIdConfig.builder()
                .workerId(workerId)
                .workerIdBit(workerIdBit)
                .build();
    }

    /**
     * 创建云环境配置
     */
    private WorkerIdConfig createCloudWorkerConfig() {
        DwSnowflakeNacosProcessor.checkRegistered();
        NamingService namingService = DwSnowflakeNacosProcessor.getNamingService();
        Instance currentInstance = DwSnowflakeNacosProcessor.getCurrentInstanceWithRetry(namingService);

        if (currentInstance == null) {
            throw new BusinessException("当前服务实例不存在");
        }

        int workerId = assignWorkerIdWithLock(namingService, currentInstance);
        return WorkerIdConfig.builder()
                .workerId(workerId)
                .workerIdBit(JaProperty.getLong(WORKER_ID_BIT_KEY, 6L))
                .build();
    }

    // ===================== 云环境工作逻辑 =====================

    /**
     * 加锁分配工作节点ID
     */
    private int assignWorkerIdWithLock(NamingService namingService, Instance instance) {
        String lockKey = createDistributedLockKey();
        JaRedisLockEntity lockEntity = createLockEntity(lockKey);

        return JaRedisLock.lock(lockEntity, () -> {
            int workerId = findAvailableWorkerId(namingService);
            updateAndRegisterInstance(namingService, instance, workerId);
            return workerId;
        });
    }

    /**
     * 查找可用工作节点ID
     */
    private int findAvailableWorkerId(NamingService namingService) {
        Set<Integer> usedIds = getUsedWorkerIds(namingService);
        return findAvailableId(usedIds)
                .orElseThrow(() -> new BusinessException("无可用工作节点ID (0-63)"));
    }

    /**
     * 更新实例并注册
     */
    private void updateAndRegisterInstance(NamingService namingService, Instance instance, int workerId)
            throws NacosException {

        instance.getMetadata().put(WORKER_ID_KEY, String.valueOf(workerId));
        namingService.registerInstance(
                DwSnowflakeNacosProcessor.getServiceName(),
                DwSnowflakeNacosProcessor.getGroupName(),
                instance
        );
        log.info("注册实例完成，工作节点ID: {}", workerId);
    }

    // ===================== 工具方法 =====================


    /**
     * 创建分布式锁键
     */
    private String createDistributedLockKey() {
        return String.format(LOCK_KEY_PATTERN, JaEnvProperty.getApplicationName());
    }

    /**
     * 创建锁配置
     */
    private JaRedisLockEntity createLockEntity(String lockKey) {
        JaProperty.put("ja.redis.lock.maxSpinTime", 40000L);
        return JaRedisLockEntity.builder()
                .lockKey(lockKey)
                .spinTime(40000)
                .build();
    }


    // ===================== Worker ID 分配逻辑 =====================

    /**
     * 获取已使用的WorkerID集合
     */
    private Set<Integer> getUsedWorkerIds(NamingService namingService) {
        return DwSnowflakeNacosProcessor.getAllInstances(namingService).stream()
                .map(Instance::getMetadata)
                .map(metadata -> metadata.get(WORKER_ID_KEY))
                .map(this::safeParseWorkerId)
                .filter(id -> id >= MIN_WORKER_ID && id <= MAX_WORKER_ID)
                .collect(Collectors.toSet());
    }

    /**
     * 安全的WorkerID转换
     */
    private int safeParseWorkerId(String value) {
        if (value == null) {
            return INVALID_WORKER_ID;
        }

        try {
            return Integer.parseInt(value);
        } catch (NumberFormatException e) {
            log.warn("无效的WorkerID格式: '{}'", value);
            return INVALID_WORKER_ID;
        }
    }

    /**
     * 查找可用ID (从高到低分配)
     */
    private Optional<Integer> findAvailableId(Set<Integer> usedIds) {
        // 优先分配高ID（兼容旧系统）
        for (int id = MAX_WORKER_ID; id >= MIN_WORKER_ID; id--) {
            if (!usedIds.contains(id)) {
                log.info("分配工作节点ID: {}", id);
                return Optional.of(id);
            }
        }
        return Optional.empty();
    }
}