package com.digiwin.athena.framework.snowflake;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.fastjson.JSON;
import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.listener.EventListener;
import com.alibaba.nacos.api.naming.listener.NamingEvent;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
public class DwSnowflakeNacosProcessor {
    private static NacosDiscoveryProperties discoveryProperties;
    private static NacosServiceManager nacosServiceManager;

    /**
     * 确保依赖可用
     */
    public static void checkRegistered() {
        if (discoveryProperties == null) {
            discoveryProperties = JaSpringBeanUtil.getBean(NacosDiscoveryProperties.class);
        }
        if (nacosServiceManager == null) {
            nacosServiceManager = JaSpringBeanUtil.getBean(NacosServiceManager.class);
        }

        if (discoveryProperties == null || nacosServiceManager == null) {
            throw new BusinessException("Nacos依赖未找到，请检查配置");
        }
    }

    /**
     * 获取NamingService实例
     */
    public static NamingService getNamingService() {
        return nacosServiceManager.getNamingService(discoveryProperties.getNacosProperties());
    }

    /**
     * 获取服务名称
     */
    public static String getServiceName() {
        return discoveryProperties.getService();
    }

    /**
     * 获取分组名称
     */
    public static String getGroupName() {
        return discoveryProperties.getGroup();
    }

    // ===================== 实例操作 =====================

    /**
     * 获取当前服务实例
     */
    public static Instance getCurrentInstance(NamingService namingService) {
        String currentIp = discoveryProperties.getIp();
        int currentPort = discoveryProperties.getPort();
        log.info("currentIp:{},currentPort:{}", currentIp, currentPort);
        List<Instance> allInstances = getAllInstances(namingService);
        log.info("allInstances:{}", JSON.toJSONString(allInstances));

        Map<String, Instance> instanceMap = allInstances.stream()
                .collect(Collectors.toMap(
                        DwSnowflakeNacosProcessor::getInstanceKey,
                        Function.identity(),
                        (existing, replacement) -> existing
                ));

        String currentKey = createInstanceKey(currentIp, currentPort);
        return instanceMap.get(currentKey);
    }

    /**
     * 获取所有服务实例
     */
    public static List<Instance> getAllInstances(NamingService namingService) {
        try {
            log.info("serviceName:{}，groupName:{}", getServiceName(), getGroupName());
            return namingService.getAllInstances(getServiceName(), getGroupName());
        } catch (NacosException e) {
            JaLog.error("获取Nacos服务实例失败", e);
            return Collections.emptyList();
        }
    }

    /**
     * 创建实例唯一键
     */
    public static String createInstanceKey(String ip, int port) {
        return ip + ":" + port;
    }

    /**
     * 获取实例唯一键
     */
    public static String getInstanceKey(Instance instance) {
        return instance.getIp() + ":" + instance.getPort();
    }

    /**
     * 使用订阅机制获取服务实例列表，带有超时和重试机制
     *
     * @param namingService Nacos命名服务实例
     * @return 服务实例列表，如果获取失败则返回空列表
     * @throws BusinessException 当发生严重错误时抛出
     */
    public static List<Instance> getAllInstancesWithSubscribe(NamingService namingService) {
        final AtomicReference<List<Instance>> resultRef = new AtomicReference<>();
        final CountDownLatch latch = new CountDownLatch(1);
        final long timeout = 30; // 超时时间(秒)

        // 1. 定义事件监听器
        EventListener listener = event -> {
            if (event instanceof NamingEvent) {
                NamingEvent namingEvent = (NamingEvent) event;
                List<Instance> instances = namingEvent.getInstances();
                if (!instances.isEmpty()) {
                    resultRef.set(instances);
                    latch.countDown();
                }
            }
        };

        // 2. 执行订阅
        try {
            namingService.subscribe(getServiceName(), getGroupName(), listener);
        } catch (NacosException e) {
            log.error("Nacos订阅服务失败，服务名: {}, 分组: {}", getServiceName(), getGroupName(), e);
            throw new BusinessException("Nacos订阅服务失败", e);
        }

        // 3. 等待首次通知
        try {
            if (!latch.await(timeout, TimeUnit.SECONDS)) {
                log.warn("等待Nacos服务实例通知超时，将尝试直接获取实例");
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.warn("等待Nacos服务实例通知被中断");
        } finally {
            // 4. 确保取消订阅
            try {
                namingService.unsubscribe(getServiceName(), getGroupName(), listener);
            } catch (NacosException e) {
                log.warn("取消Nacos订阅失败", e);
            }
        }

        // 5. 返回结果或降级方案
        List<Instance> instances = resultRef.get();
        if (instances != null && !instances.isEmpty()) {
            return instances;
        }

        // 降级方案：直接获取实例
        try {
            return namingService.getAllInstances(getServiceName(), getGroupName());
        } catch (NacosException e) {
            log.error("获取Nacos服务实例失败", e);
            return Collections.emptyList();
        }
    }


    public static Instance getCurrentInstanceWithRetry(NamingService namingService) {
        int retryCount = 0;
        int maxCount = 5;
        int timeOut = 2000;

        while (retryCount < maxCount) {
            try {
                Instance instance = getCurrentInstance(namingService);
                if (instance != null) {
                    return instance;
                }
                Thread.sleep(timeOut);
            } catch (Exception e) {
                log.error("获取Nacos服务实例失败，重试次数: {}", retryCount, e);
                try {
                    Thread.sleep(timeOut);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            } finally {
                retryCount++;
            }
        }
        return null;
    }
}
