package com.digiwin.athena.framework.snowflake;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.fastjson.JSON;
import com.alibaba.nacos.api.exception.NacosException;
import com.alibaba.nacos.api.naming.NamingService;
import com.alibaba.nacos.api.naming.pojo.Instance;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.spring.util.JaSpringBeanUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
public class DwSnowflakeNacosProcessor {
    private static NacosDiscoveryProperties discoveryProperties;
    private static NacosServiceManager nacosServiceManager;

    /**
     * 确保依赖可用
     */
    public static void checkRegistered() {
        if (discoveryProperties == null) {
            discoveryProperties = JaSpringBeanUtil.getBean(NacosDiscoveryProperties.class);
        }
        if (nacosServiceManager == null) {
            nacosServiceManager = JaSpringBeanUtil.getBean(NacosServiceManager.class);
        }

        if (discoveryProperties == null || nacosServiceManager == null) {
            throw new BusinessException("Nacos依赖未找到，请检查配置");
        }
    }

    /**
     * 获取NamingService实例
     */
    public static NamingService getNamingService() {
        return nacosServiceManager.getNamingService(discoveryProperties.getNacosProperties());
    }

    /**
     * 获取服务名称
     */
    public static String getServiceName() {
        return discoveryProperties.getService();
    }

    /**
     * 获取分组名称
     */
    public static String getGroupName() {
        return discoveryProperties.getGroup();
    }

    // ===================== 实例操作 =====================

    /**
     * 获取当前服务实例
     */
    public static Instance getCurrentInstance(NamingService namingService) {
        String currentIp = discoveryProperties.getIp();
        int currentPort = discoveryProperties.getPort();
        log.info("currentIp:{},currentPort:{}", currentIp, currentPort);
        List<Instance> allInstances = getAllInstances(namingService);
        log.info("allInstances:{}", allInstances.size());

        Map<String, Instance> instanceMap = allInstances.stream()
                .collect(Collectors.toMap(
                        DwSnowflakeNacosProcessor::getInstanceKey,
                        Function.identity(),
                        (existing, replacement) -> existing
                ));

        String currentKey = createInstanceKey(currentIp, currentPort);
        return instanceMap.get(currentKey);
    }

    /**
     * 获取所有服务实例
     */
    public static List<Instance> getAllInstances(NamingService namingService) {
        try {
            log.info("serviceName:{}，groupName:{}", getServiceName(), getGroupName());
            return namingService.getAllInstances(getServiceName(), getGroupName());
        } catch (NacosException e) {
            JaLog.error("获取Nacos服务实例失败", e);
            return Collections.emptyList();
        }
    }

    /**
     * 创建实例唯一键
     */
    public static String createInstanceKey(String ip, int port) {
        return ip + ":" + port;
    }

    /**
     * 获取实例唯一键
     */
    public static String getInstanceKey(Instance instance) {
        return instance.getIp() + ":" + instance.getPort();
    }

    public static Instance getCurrentInstanceWithRetry(NamingService namingService) {
        int retryCount = 0;
        int maxCount = 5;
        // 初始等待时间为2秒（可根据需求调整）
        // 当前轮次的延迟时间
        long currentDelay = 2000L;

        while (retryCount < maxCount) {
            try {
                Instance instance = getCurrentInstance(namingService);
                if (instance != null) {
                    return instance;
                }
                log.info("获取Nacos服务实例重试, 重试次数: {}, 本次等待时间: {}ms", retryCount, currentDelay);
                Thread.sleep(currentDelay); // 使用动态计算的等待时间
            } catch (Exception e) {
                log.error("获取Nacos服务实例失败，重试次数: {}, 本次等待时间: {}ms", retryCount, currentDelay, e);
                try {
                    Thread.sleep(currentDelay); // 异常时同样等待相同时间
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("线程被中断", ie); // 显式抛出中断异常
                }
            } finally {
                retryCount++;
                // 更新下一次的等待时间（指数退避策略）
                if (retryCount < maxCount) {
                    currentDelay = Math.min(currentDelay * 2, 30000);
                }
            }
        }
        return null;
    }
}
