package com.digiwin.athena.iam.sdk.manager;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.auth.domain.IdentityUser;
import com.digiwin.athena.appcore.auth.domain.RoleDTO;
import com.digiwin.athena.appcore.auth.domain.UserRoleDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.common.sdk.manager.meta.constants.DwHttpHeaderConstants;
import com.digiwin.athena.common.sdk.manager.util.DwSpringHttpUtil;
import com.digiwin.athena.iam.sdk.meta.dto.IamResultDTO;
import com.digiwin.athena.iam.sdk.meta.dto.request.*;
import com.digiwin.athena.iam.sdk.meta.dto.response.*;
import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.util.JaStringUtil;
import org.springframework.core.ParameterizedTypeReference;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import static com.digiwin.athena.iam.sdk.meta.constants.IamErrorCode.*;

/**
 * iam api
 * <p>
 *
 * @author gengwei
 * @since 2024/8/15 11:48
 */
public class IamManager {
    private static final String CATALOG_ID = "basic";

    private static final String KEY = "doNotDisturb";

    private static String getIamUri() {
        return JaProperty.get("iam.uri");
    }

    /**
     * 查询租户虚拟tokenUrl
     */
    private static final ParameterizedTypeReference<AuthoredUser> authoredUserType = new ParameterizedTypeReference<AuthoredUser>() {
    };
    private static final ParameterizedTypeReference<List<JSONObject>> JSONObjectListType = new ParameterizedTypeReference<List<JSONObject>>() {
    };
    private static final ParameterizedTypeReference<IamResultDTO<Map<Object, Object>>> MapType = new ParameterizedTypeReference<IamResultDTO<Map<Object, Object>>>() {
    };
    private static final ParameterizedTypeReference<UserDTO> UserDTOType = new ParameterizedTypeReference<UserDTO>() {
    };
    private static final ParameterizedTypeReference<IamResultDTO<UserDTO>> IamUserDTOType = new ParameterizedTypeReference<IamResultDTO<UserDTO>>() {
    };
    private static final ParameterizedTypeReference<IamResultDTO<BossIamTenantAuthDTO>> BossIamTenantAuthDTOType = new ParameterizedTypeReference<IamResultDTO<BossIamTenantAuthDTO>>() {
    };
    private static final ParameterizedTypeReference<InnerExistDTO> InnerExistDTOType = new ParameterizedTypeReference<InnerExistDTO>() {
    };
    private static final ParameterizedTypeReference<IamResultDTO<Object>> ObjectType = new ParameterizedTypeReference<IamResultDTO<Object>>() {
    };
    private static final ParameterizedTypeReference<UserRoleDTO> UserRoleDTOType = new ParameterizedTypeReference<UserRoleDTO>() {
    };
    private static final ParameterizedTypeReference<WithTenantRespDTO> WithTenantRespDTOType = new ParameterizedTypeReference<WithTenantRespDTO>() {
    };
    private static final ParameterizedTypeReference<IamResultDTO<PersonalizedDTO>> PersonalizedDTOType = new ParameterizedTypeReference<IamResultDTO<PersonalizedDTO>>() {
    };

    private static final ParameterizedTypeReference<IdentityUser> identityUserType = new ParameterizedTypeReference<IdentityUser>() {
    };

    private static final ParameterizedTypeReference<RoleDTO> roleDTOType = new ParameterizedTypeReference<RoleDTO>() {
    };

    private static final String VIRTUAL_TOKEN_URL = getIamUri() + "/api/iam/v2/identity/login/internal";

    public static AuthoredUser queryApiVirtualToken(String tenantId) {
        if (JaStringUtil.isEmpty(tenantId)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5003);
        }
        AuthoredUser result = null;
        try {
            VirtualTokenReqDTO virtualTokenDTO = new VirtualTokenReqDTO();
            virtualTokenDTO.setTenantId(tenantId);
            result = DwSpringHttpUtil.post(
                    VIRTUAL_TOKEN_URL,
                    virtualTokenDTO,
                    authoredUserType,
                    httpHeaders -> httpHeaders.add(DwHttpHeaderConstants.Header_App_Token, DwHttpHeaderConstants.App_Token_Athena));
        } catch (BusinessException e) {
            e.setBizErrorCode(e.getErrorCode());
            e.setBizErrorMsg(null == e.getErrorMessage() ? "" : e.getErrorMessage().toString());
            throw e;
        } catch (Throwable e) {
            throw BusinessException.buildErrorCode(P_IAM_500_5004);
        }
        if (null == result || JaStringUtil.isEmpty(result.getToken())) {
            throw BusinessException.buildErrorCode(P_IAM_500_5004);
        }
        return result;
    }

    /**
     * 获取第三方平台的映射用户信息
     */
    public static final String GMC_V_2_MAPPING_QUERY_URL = getIamUri() + "/api/iam/v2/mapping/query";

    public static List<JSONObject> queryAppUser(long sid, String token, String identityId) {
        QueryAppUserDTO body = QueryAppUserDTO.builder()
                .userSid(sid)
                .identityId(identityId)
                .build();
        return DwSpringHttpUtil.post(GMC_V_2_MAPPING_QUERY_URL, body, JSONObjectListType, httpHeaders -> {
            httpHeaders.set(DwHttpHeaderConstants.Header_User_Token, token);
            httpHeaders.set(DwHttpHeaderConstants.Header_App_Token, DwHttpHeaderConstants.App_Token_Athena);
        });
    }

    /**
     * 测试租户
     */
    public static final String TENANT_SIMPLE_URL = getIamUri() + "/api/iam/v2/tenant/simple";

    public static UserRoleDTO queryTenantSimple(String token, String id) {
        QueryByIdDTO body = QueryByIdDTO.builder()
                .id(id)
                .build();
        return DwSpringHttpUtil.post(TENANT_SIMPLE_URL, body, UserRoleDTOType, httpHeaders -> httpHeaders.set(DwHttpHeaderConstants.Header_User_Token, token));

    }

    /**
     * 根据邮箱获取任务的实际执行人
     */
    public static final String USER_EMAIL_URL = getIamUri() + "/api/iam/v2/user/email";

    /**
     * 根据用户email和租户Id获取用户相关的信息
     *
     * @param email 用户email
     * @return 用户基本信息
     */
    public static IamResultDTO<UserDTO> queryByEmail(String email, String tenantId) {
        QueryByEmailDTO body = QueryByEmailDTO.builder()
                .email(email)
                .tenantId(tenantId)
                .build();
        return DwSpringHttpUtil.post(USER_EMAIL_URL, body, IamUserDTOType);

    }

    /**
     * 根据邮箱获取个人租户
     * status的值
     * -1 调用接口失败
     * 0 没有个人租户
     * 1 没有授权
     * 2 有个人租户并且有授权
     */
    public static final String USER_EMAIL_TENANTS_AUTH_URL = getIamUri() + "/api/iam/v2/user/email/tenants/auth";

    /**
     * 根据邮箱获取个人租户
     * status的值
     * -1 调用接口失败
     * 0 没有个人租户
     * 1 没有授权
     * 2 有个人租户并且有授权
     */
    public static IamResultDTO<BossIamTenantAuthDTO> getTenantAuth(AuthoredUser authoredUser, UserEmailTenantsAuthReqDTO userEmailTenantsAuthReqDTO) {
        return DwSpringHttpUtil.post(USER_EMAIL_TENANTS_AUTH_URL, userEmailTenantsAuthReqDTO, BossIamTenantAuthDTOType, httpHeaders -> httpHeaders.set(DwHttpHeaderConstants.Header_User_Token, authoredUser.getToken()));

    }

    /**
     * 判断邮箱帐号是否已经注册,且是否为鼎捷用户，排除掉企业内部用户
     */
    public static final String USER_EMAIL_WITHOUT_INNER_EXIST_URL = getIamUri() + "/api/iam/v2/user/email/without/inner/exist";

    /**
     * 判断邮箱帐号是否已经注册,且是否为鼎捷用户，排除掉企业内部用户
     */
    public static InnerExistDTO emailInnerExist(String email) {
        QueryByEmailDTO body = QueryByEmailDTO.builder()
                .email(email)
                .build();
        return DwSpringHttpUtil.post(USER_EMAIL_WITHOUT_INNER_EXIST_URL, body, InnerExistDTOType);

    }

    /**
     * 根据用户id获取用户相关的信息
     */
    public static final String USER_URL = getIamUri() + "/api/iam/v2/user/full";


    public static UserDTO queryUser(String userId) {
        QueryByIdDTO body = QueryByIdDTO.builder()
                .id(userId)
                .build();
        return DwSpringHttpUtil.post(USER_URL, body, UserDTOType);

    }

    /**
     * 通过用户ID 获取是否静默
     */
    public static final String USER_METADATA_VALUE_URL = getIamUri() + "/api/iam/v2/user/metadata/value";


    /**
     * 通过用户ID 获取是否静默
     */
    public static IamResultDTO<Object> getMessageDoNotDisturbUserId(String userId, String tenantId, String token) {
        QueryDisturbUserDTO body = QueryDisturbUserDTO.builder()
                .tenantId(tenantId)
                .userId(userId)
                .catalogId(CATALOG_ID)
                .key(KEY)
                .build();
        return DwSpringHttpUtil.post(USER_METADATA_VALUE_URL, body, ObjectType, httpHeaders -> httpHeaders.set(DwHttpHeaderConstants.Header_User_Token, token));
    }

    /**
     * 根据用户id获取角色信息
     */
    public static final String USER_ROLE_URL = getIamUri() + "/api/iam/v2/user/role";


    public static UserRoleDTO queryUserRoles(String userId, String token) {
        if (JaStringUtil.isEmpty(userId)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5007);
        }
        if (JaStringUtil.isEmpty(token)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5001);
        }
        UserRoleDTO result = null;
        try {
            QueryByIdDTO body = QueryByIdDTO.builder()
                    .id(userId)
                    .build();
            result = DwSpringHttpUtil.post(
                    USER_ROLE_URL,
                    body,
                    UserRoleDTOType,
                    httpHeaders -> httpHeaders.add(DwHttpHeaderConstants.Header_User_Token, token),
                    Collections.emptyMap());
        } catch (BusinessException e) {
            e.setBizErrorCode(e.getErrorCode());
            e.setBizErrorMsg(null == e.getErrorMessage() ? "" : e.getErrorMessage().toString());
            throw e;
        } catch (Throwable e) {
            throw BusinessException.buildErrorCode(P_IAM_500_5006);
        }
        if (null == result) {
            return new UserRoleDTO();
        }
        return result;
    }


    /**
     * 获取用户的租户信息;4.11增加返回agreeAgreement
     * iam接口安全调整方案，替换/api/iam/v2/user/contacts接口;
     */
    public static final String TENANT_INFO_URL = getIamUri() + "/api/iam/v2/user/withtenant";

    /**
     * 根据用户id查询出用户的email
     * update: iam接口安全调整给了替换方案，替换为接口/api/iam/v2/user/withtenant
     *
     * @param userId 用户id
     */
    public static WithTenantRespDTO queryUserEmail(String userId) {
        QueryByIdDTO body = QueryByIdDTO.builder()
                .id(userId)
                .build();
        return DwSpringHttpUtil.post(TENANT_INFO_URL, body, WithTenantRespDTOType);
    }

    /**
     * 根据userSid获取租户语言别设置
     */
    public static final String USERMETADATA_USER_GET_URL = getIamUri() + "/api/iam/v2/usermetadata/user/get";

    /**
     * 查询用户的个性化设置数据
     * <p>
     * tenantSid 租户SID，注意非租户id
     * userId    用户id
     * key       需要查询的key
     * catalogId 类目id
     */
    public static IamResultDTO<PersonalizedDTO> queryPersonalizedInfo(QueryPersonalizedInfoReqDTO queryPersonalizedInfoReqDTO) {
        return DwSpringHttpUtil.post(USERMETADATA_USER_GET_URL, queryPersonalizedInfoReqDTO, PersonalizedDTOType);
    }

    public static final String GET_USER_TYPE_URL = getIamUri() + "/api/iam/v2/query/user";

    private static final ParameterizedTypeReference<UserTypeRespDTO> getUserTypeRespType = new ParameterizedTypeReference<UserTypeRespDTO>() {
    };

    public static UserTypeRespDTO queryPersonalizedInfo(String userId, String token) {
        QueryByUserIdDTO reqBody = QueryByUserIdDTO.builder()
                .userId(userId)
                .build();
        return DwSpringHttpUtil.post(GET_USER_TYPE_URL, reqBody, getUserTypeRespType, httpHeaders -> httpHeaders.add("token", token));
    }

    public static final String QUERY_IDENTITY_USER_URL = getIamUri() + "/api/iam/v2/identity/token/analyze/get/agent";

    public static IdentityUser queryIdentityUser(String userToken) {
        if (JaStringUtil.isEmpty(userToken)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5001);
        }
        IdentityUser result = null;
        try {
            result = DwSpringHttpUtil.post(
                    QUERY_IDENTITY_USER_URL,
                    null,
                    identityUserType,
                    httpHeaders -> httpHeaders.add(DwHttpHeaderConstants.Header_App_Token, DwHttpHeaderConstants.App_Token_Athena));
        } catch (BusinessException e) {
            e.setBizErrorCode(e.getErrorCode());
            e.setBizErrorMsg(null == e.getErrorMessage() ? "" : e.getErrorMessage().toString());
            throw e;
        } catch (Throwable e) {
            throw BusinessException.buildErrorCode(P_IAM_500_5005);
        }
        if (null == result) {
            return null;
        }
        return result;
    }

    public static final String QUERY_ROLE_URL = getIamUri() + "/api/iam/v2/role";

    public static RoleDTO queryRole(String userId, String token) {
        if (JaStringUtil.isEmpty(userId)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5007);
        }
        if (JaStringUtil.isEmpty(token)) {
            throw BusinessException.buildBizErrorCode(P_IAM_500_5001);
        }
        RoleDTO result = null;
        try {
            QueryByIdDTO body = QueryByIdDTO.builder()
                    .id(userId)
                    .build();
            result = DwSpringHttpUtil.post(
                    QUERY_ROLE_URL,
                    body,
                    roleDTOType,
                    httpHeaders -> httpHeaders.add(DwHttpHeaderConstants.Header_User_Token, token));
        } catch (BusinessException e) {
            e.setBizErrorCode(e.getErrorCode());
            e.setBizErrorMsg(null == e.getErrorMessage() ? "" : e.getErrorMessage().toString());
            throw e;
        } catch (Throwable e) {
            throw BusinessException.buildErrorCode(P_IAM_500_5006);
        }
        if (null == result) {
            return null;
        }
        return result;
    }
}
