package com.digiwin.athena.atmc.common.bk.service;

import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.common.bk.domain.TEProjectBKLackData;
import com.digiwin.athena.atmc.common.bk.domain.TEProjectBKLackData.TEProjectBKData;
import com.digiwin.athena.atmc.common.bk.domain.TEProjectBKLackData.TenantToken;
import com.digiwin.athena.atmc.common.constant.BkConstant;
import com.digiwin.athena.atmc.common.dao.BpmActivityWorkitemMapper;
import com.digiwin.athena.atmc.common.mongodb.MongoDbQueryService;
import com.digiwin.athena.atmc.common.mongodb.MongodbService;
import com.digiwin.athena.atmc.common.mongodb.data.IndexData;
import com.digiwin.athena.atmc.common.util.BKUtils;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.restful.ptm.PtmService;
import com.digiwin.athena.atmc.infrastructure.pojo.bo.migration.PtmProjectRecordBO;
import com.digiwin.athena.atmc.http.restful.thememap.ThemeMapService;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmPageName;
import com.github.pagehelper.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author：SYQ
 * @date：2021/12/6
 */
@Slf4j
@Service
public class BkTaskActivityService {

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    private MongodbService mongodbService;

    @Autowired
    private MongoDbQueryService mongoDbQueryService;

    @Autowired
    private BpmActivityWorkitemMapper bpmActivityWorkitemMapper;

    @Autowired
    private PtmService ptmService;

    //用于存储任务定义中的dataSources
    Map<String, Object> tmActivityDataSourceMap = new HashMap<>();

    /**
     * 新增原始bk和卡片的关系
     *
     * @Author：SYQ
     * @Date：2022/1/6 14:48
     */
    public void insertOriBk(List<JSONObject> oriBkList, List<Object> oriObjBkList, Long cardId, String tenantId, String type) {
        insertOriBk(oriBkList, oriObjBkList, cardId, tenantId, type, "ori");
    }

    /**
     * 新增原始bk和卡片的关系
     *
     * @Author：majfa
     * @Date：2022/11/24 14:48
     */
    public void insertOriBk(List<JSONObject> oriBkList, List<Object> oriObjBkList, Long cardId, String tenantId, String type, String source) {
        //集合名称
        String collectionName = tenantId;
        IndexData indexData = new IndexData();
        indexData.append(BkConstant.CARD_ID, BkConstant.ASC);
        indexData.append(BkConstant.SEARCH_INDEX_PREFIX, BkConstant.ASC);
        mongodbService.createCollectionLock(collectionName, indexData.getIndexList());

        //新增数据
        List<Map> dataList = new ArrayList<>();
        for (JSONObject bk : oriBkList) {
            Map<String, Object> dataMap = new HashMap<>();
            String bkStr = JsonUtils.objectToString(bk);
            String bkContent = bkStr.replaceAll("[{}\\\"]", "").replace(",", "&");
            TreeMap treeMap = JsonUtils.jsonToObject(bkStr, TreeMap.class);
            dataMap.put("bk", treeMap);

            TreeMap bkSortMap = BKUtils.recursionToTreeMap(bk);
            String searchJsonStr = JsonUtils.objectToString(bkSortMap);
            String searchContent = searchJsonStr.replaceAll("[{}\\\"]", "").replace(",", "&");
            dataMap.put(BkConstant.SEARCH_CONTENT, searchContent);
            dataMap.put(BkConstant.BK_CONTENT, bkContent);
            dataMap.put(BkConstant.CARD_ID, cardId);
            dataMap.put("type", type);
            dataMap.put(BkConstant.TENANT_ID, tenantId);
            dataMap.put(BkConstant.CREATE_DATE, LocalDateTime.now());
            dataMap.put(BkConstant.MODIFY_DATE, LocalDateTime.now());
            dataMap.put(BkConstant.SOURCE, source);
            //生成bkSearch字段信息，作为bk查询的索引字段
            List<Map<String, Object>> bkSearchList = BKUtils.convertBkIndex(bk);
            dataMap.put(BkConstant.SEARCH_PREFIX, bkSearchList);
            dataMap.put(BkConstant.BK_INDEX_STATE, BkConstant.INDEX_STATE_CREATE);
            dataList.add(dataMap);
        }
        for (Object bk : oriObjBkList) {
            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("bk", bk);
            dataMap.put(BkConstant.BK_CONTENT, bk);
            dataMap.put(BkConstant.CARD_ID, cardId);
            dataMap.put("type", type);
            dataMap.put(BkConstant.TENANT_ID, tenantId);
            dataMap.put(BkConstant.CREATE_DATE, LocalDateTime.now());
            dataMap.put(BkConstant.MODIFY_DATE, LocalDateTime.now());
            dataMap.put(BkConstant.SOURCE, source);
            //生成bkSearch字段信息，作为bk查询的索引字段
            List<Map<String, Object>> bkSearchList = BKUtils.convertBkIndexSimple(bk);
            dataMap.put(BkConstant.SEARCH_PREFIX, bkSearchList);
            dataMap.put(BkConstant.BK_INDEX_STATE, BkConstant.INDEX_STATE_CREATE);
            dataList.add(dataMap);
        }
        log.info("{}", cardId);
        mongodbService.insert(dataList, collectionName);
    }


    /**
     * 更新
     *
     * @Author：SYQ
     * @Date：2021/12/28 16:26
     */
    public void updateBkTaskActivity(String collectionName, String bk, Long cardId, Map<String, Object> updateDataMap) {
        if (StringUtil.isEmpty(collectionName) || StringUtil.isEmpty(bk) || cardId == null) {
            throw ErrorCodeEnum.TASK_BK_PARAM_EMPTY.getBusinessException();
        }
        if (updateDataMap == null || updateDataMap.size() == 0) {
            return;
        }
        //查询条件
        Query query = Query.query(Criteria.where("bk").is(bk).and(BkConstant.CARD_ID).is(cardId));
        //需要更新的字段及值
        Update update = new Update();
        for (Map.Entry<String, Object> entry : updateDataMap.entrySet()) {
            update.set(entry.getKey(), entry.getValue());
        }
        //更新数据
        mongodbService.updateFirst(collectionName, query, update);
    }

    /**
     * 【严格匹配】根据业务主键，查询任务ID
     * <p>这里指json格式的严格匹配，并不指内容的严格匹配</p>
     * <p>给定的json必须与存储的完全匹配，属性数量一致，属性顺序一致</p>
     *
     * @param bkList
     * @return
     */
    public List<Long> selectTaskIdByBusinessKey(String collectionName, List<Map> bkList) {
        Query query = Query.query(Criteria.where("type").is(BkConstant.WORKITEM).and("bk").in(bkList));
        query.fields().include("bk").include(BkConstant.CARD_ID).exclude("_id");
        List<Map> list = mongodbService.query(collectionName, query);
        return list.stream().map(y -> (Long) y.get(BkConstant.CARD_ID)).collect(Collectors.toList());
    }

    /**
     * 【模糊匹配】根据业务主键，查询任务ID
     * <p>这里指json格式的模糊匹配，并不指内容的模糊匹配</p>
     * <p>给定的json 可以与存储的不完全一致，属性数量可以不一致，属性顺序可以不一致</p>
     *
     * @param bkList
     * @return
     */
    public List<Long> selectTaskIdByBusinessKeyFuzzy(String collectionName, List<Map> bkList) {
        List<Map> list = this.selectTaskIdByBusinessKeyFuzzyV2(collectionName, bkList);
        return list.stream().map(y -> (Long) y.get(BkConstant.CARD_ID)).collect(Collectors.toList());
    }


    /**
     * 【模糊匹配】根据业务主键，查询任务ID
     * <p>这里指json格式的模糊匹配，并不指内容的模糊匹配</p>
     * <p>给定的json 可以与存储的不完全一致，属性数量可以不一致，属性顺序可以不一致</p>
     *
     * @param bkList
     * @return
     */
    public List<Map> selectTaskIdByBusinessKeyFuzzyV2(String collectionName, List<Map> bkList) {
        return selectTaskIdByBusinessKeyFuzzyV2(collectionName, BkConstant.WORKITEM, bkList);
    }

    /**
     * 【模糊匹配】根据业务主键，查询任务ID
     * <p>这里指json格式的模糊匹配，并不指内容的模糊匹配</p>
     * <p>给定的json 可以与存储的不完全一致，属性数量可以不一致，属性顺序可以不一致</p>
     *
     * @param collectionName 集合名称
     * @param type           任务/项目
     * @param bkList
     * @return
     */
    public List<Map> selectTaskIdByBusinessKeyFuzzyV2(String collectionName, String type, List<Map> bkList) {
        return selectWithMultipleBusinessKeys(collectionName, type, bkList);
//        bkList = bkList.stream().map(x -> mongoDbQueryService.tileMap("bk", x)).collect(Collectors.toList());
//
//        Map filter = new HashMap();
//        Map andMap = new HashMap();
//        andMap.put("type", type);
//        Map orMap = new HashMap();
//        orMap.put("$or", bkList);
//        List<Map> andList = new ArrayList<>();
//        andList.add(andMap);
//        andList.add(orMap);
//        filter.put("$and", andList);
//        Map command = mongoDbQueryService.getExecuteQueryCommand(collectionName, filter, null);
//        log.info("selectTaskIdByBusinessKeyFuzzyV2-command:{}", JsonUtils.objectToString(command));
//        List<Map> list = mongodbService.executeQueryCommand(command);
//        return list;
    }

    /**
     * 【模糊匹配】根据业务主键，查询任务ID
     * <p>这里指json格式的模糊匹配，并不指内容的模糊匹配</p>
     * <p>给定的json 可以与存储的不完全一致，属性数量可以不一致，属性顺序可以不一致</p>
     *
     * @param collectionName 集合名称
     * @param type           任务/项目
     * @param bkList
     * @return
     */
    public List<Map> selectWithMultipleBusinessKeys(String collectionName, String type, List<Map> bkList) {

        //获取bk列表信息
        List<List<String>> indexList = BKUtils.bkObjectToIndexList(bkList);


        List<Map<String, Object>> finalBkList = new ArrayList<>();
        for (List<String> tempList : indexList) {
            Map<String, Object> indexAndMap = new HashMap<>(1);
            List<Map<String, Object>> singleList = new ArrayList<>();
            for (String temp : tempList) {
                Map<String, Object> singleMap = new HashMap<>(1);
                singleMap.put(BkConstant.SEARCH_INDEX_PREFIX, temp);
                singleList.add(singleMap);
            }
            indexAndMap.put("$and", singleList);
            finalBkList.add(indexAndMap);
        }

        Map filter = new HashMap();
        Map andMap = new HashMap();
        andMap.put("type", type);
        Map orMap = new HashMap();
        orMap.put("$or", finalBkList);
        List<Map> andList = new ArrayList<>();
        andList.add(andMap);
        andList.add(orMap);
        filter.put("$and", andList);

        //设置对返回的结果的字段控制，减少数据的传输
        Map<String, Object> projectMap = new HashMap<>();
        projectMap.put(BkConstant.SEARCH_PREFIX, 0);
        projectMap.put(BkConstant.BK_INDEX_STATE, 0);
        Map command = mongoDbQueryService.getExecuteQueryCommand(collectionName, filter, projectMap);
        log.info("selectWithMultipleBusinessKeys-command:{}", JsonUtils.objectToString(command));
        List<Map> list = mongodbService.executeQueryCommand(command);
        return list;
    }

    /**
     * 【业务字段全量匹配】根据业务主键，查询任务ID
     *
     * @param bkList
     * @return
     */
    public List<Long> queryCardWithSearchInfos(String collectionName, List<Map> bkList, String type) {

        Query query = new Query();
        if (StringUtil.isNotEmpty(type)) {
            query.addCriteria(Criteria.where("type").is(type));
        }
        List<Criteria> criteriaList = new ArrayList<>();
        for (Map tempMap : bkList) {
            TreeMap sortMap = BKUtils.recursionToTreeMap(tempMap);
            String tempJson = JsonUtils.objectToString(sortMap);
            String searchContent = tempJson.replaceAll("\\{|}|\"", "").replace(",", "&");
            Criteria criteria = Criteria.where(BkConstant.SEARCH_CONTENT).is(searchContent);

            criteriaList.add(criteria);
        }
        Criteria[] orCriteriaArr = new Criteria[criteriaList.size()];
        criteriaList.toArray(orCriteriaArr);
        Criteria criteria = new Criteria().orOperator(orCriteriaArr);
        query.addCriteria(criteria);
        List<Map> retList = mongodbService.query(collectionName, query);
        return retList.stream().map(y -> (Long) y.get(BkConstant.CARD_ID)).collect(Collectors.toList());
    }


    /**
     * 根据指定的任务Id 获取业务主键
     *
     * @param workitemId
     * @return
     */
    public List<Map> selectBusinessKeyByTaskId(String collectionName, Long workitemId) {
        List<Long> list = new ArrayList<>();
        list.add(workitemId);
        return selectBusinessKeyByTaskId(collectionName, list);
    }

    /**
     * 根据指定的任务Id 获取业务主键
     *
     * @param workitemIdList
     * @return
     */
    public List<Map> selectBusinessKeyByTaskId(String collectionName, List<Long> workitemIdList) {
        Query query = Query.query(Criteria.where("type").is(BkConstant.WORKITEM).and(BkConstant.CARD_ID).in(workitemIdList));
        query.fields().include("bk").include("cardId").include(BkConstant.CARD_ID).exclude("_id");
        return mongodbService.query(collectionName, query);
    }

    /**
     * 根据指定的任务Id 获取业务主键
     *
     * @param projectIdList
     * @return
     */
    public List<Map> selectBusinessKeyByProjectId(String collectionName, List<Long> projectIdList) {
        Query query = Query.query(Criteria.where("type").is(BkConstant.TASK).and(BkConstant.CARD_ID).in(projectIdList));
        query.fields().include("bk").include("cardId").include(BkConstant.CARD_ID).exclude("_id");
        return mongodbService.query(collectionName, query);
    }

    /**
     * 根据指定的任务Id（忽略source条件） 获取业务主键
     *
     * @param workitemIdList
     * @return
     */
    public List<Map> selectBusinessKeyByTaskIdIgnoreSource(String collectionName, List<Long> workitemIdList) {
        Query query = Query.query(Criteria.where("type").is(BkConstant.WORKITEM).and(BkConstant.CARD_ID).in(workitemIdList));
        query.fields().include("bk").include("cardId").include(BkConstant.CARD_ID).exclude("_id");
        return mongodbService.query(collectionName, query);
    }

    /**
     * 根据条件查询数据
     *
     * @Author：SYQ
     * @Date：2022/1/6 17:19
     */
    public List<Map> selectByParam(String collectionName, String paramStr, String type) {
        return selectByParamWithIndex(collectionName, paramStr, type);
//        Query query = new Query();
//        if (StringUtil.isNotEmpty(type)) {
//            query.addCriteria(Criteria.where("type").is(type));
//        }
//        //是否是json
//        boolean isJson = false;
//        //转json报错不是json，否则为json
//        try {
//            JSONObject.fromObject(paramStr);
//            isJson = true;
//        } catch (Exception e) {
//            log.error("select bkParam has exception:", e);
//        }
//        //解析查询参数
//        if (isJson) {
//            List<String> paramList = dealParamJson(paramStr);
//            List<Criteria> criteriaList = new ArrayList<>();
//            for (String param : paramList) {
//                Criteria criteria = Criteria.where(BkConstant.BK_CONTENT).regex(".*?" + param + ".*?");
//                criteriaList.add(criteria);
//            }
//            Criteria[] criteriaArr = new Criteria[criteriaList.size()];
//            criteriaList.toArray(criteriaArr);
//            Criteria criteria = new Criteria().andOperator(criteriaArr);
//            query.addCriteria(criteria);
//        } else {
//            query.addCriteria(Criteria.where(BkConstant.BK_CONTENT).regex(".*?" + paramStr + ".*?"));
//        }
//        return mongodbService.query(collectionName, query);
    }


    /**
     * 该方法目前先保留，暂不替换为新的查询api
     *
     * @param collectionName 集合信息
     * @param paramStr       bk参数字符串
     * @param type           类型
     * @return list
     */
    public List<Map> selectByParamWithIndex(String collectionName, String paramStr, String type) {
        Query query = new Query();
        if (StringUtil.isNotEmpty(type)) {
            query.addCriteria(Criteria.where("type").is(type));
        }
        //判断字符串是否是json字符串
        boolean isJson = BKUtils.checkJsonStr(paramStr);
        //如果不是json,则是普通字符串，直接根据内容匹配即可
        if (!isJson) {
            query.addCriteria(Criteria.where(BkConstant.SEARCH_INDEX_PREFIX).is(paramStr));
        } else {
            Map<String, Object> bkParam = JsonUtils.jsonToObject(paramStr, Map.class);
            List<String> bkIndexList = BKUtils.bkObjectToIndex(bkParam);
            //拼接查询条件，多个index为and的关系
            Criteria singleCriteria = new Criteria();
            List<Criteria> criteriaList = new ArrayList<>();
            bkIndexList.forEach(x -> {
                Criteria criteria = Criteria.where(BkConstant.SEARCH_INDEX_PREFIX).is(x);
                criteriaList.add(criteria);
            });
            singleCriteria.andOperator(criteriaList.toArray(new Criteria[0]));
            query.addCriteria(singleCriteria);
        }
        //排除index以及indexState字段信息
        query.fields().exclude(BkConstant.SEARCH_PREFIX).exclude(BkConstant.BK_INDEX_STATE);
        log.debug("query:{}", query);
        return mongodbService.query(collectionName, query);
    }


    /**
     * 根据条件查询数据
     */
    public <T> List<Map> selectByParamV2(String collectionName, List<T> businessKeys, String type) {
        List<Map> bkList = JsonUtils.jsonToListObject(JsonUtils.objectToString(businessKeys), Map.class);
        return this.selectTaskIdByBusinessKeyFuzzyV2(collectionName, bkList);
    }

    /**
     * 处理json类型的查询参数
     *
     * @Author：SYQ
     * @Date：2022/1/6 17:19
     */
    private List<String> dealParamJson(String param) {
        List<String> resultList = new ArrayList<>();
        //解析查询参数
        Map<String, Object> paramMap = JsonUtils.jsonToObject(param, Map.class);
        for (Map.Entry<String, Object> entry : paramMap.entrySet()) {
            getChildrenParam(resultList, entry.getKey(), entry.getValue());
        }
        return resultList;
    }

    /**
     * 递归获取所有层级的查询参数
     *
     * @Author：SYQ
     * @Date：2022/1/6 16:05
     */
    private void getChildrenParam(List<String> paramList, String key, Object object) {
        String param = "";
        if (object instanceof Collection) {
            List<Object> list = JsonUtils.jsonToObject(JsonUtils.objectToString(object), List.class);
            for (Object obj : list) {
                getChildrenParam(paramList, key, obj);
            }
        } else if (object instanceof Map) {
            Map<String, Object> map = JsonUtils.jsonToObject(JsonUtils.objectToString(object), Map.class);
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                getChildrenParam(paramList, entry.getKey(), entry.getValue());
            }
        } else {
            param = key + ":" + object;
            paramList.add(param);
        }
    }


    /**
     * 删除索引
     *
     * @Author：SYQ
     * @Date：2022/1/26 13:51
     */
    public void deleteIndexFromCollections(String deleteIndexName) {
        mongodbService.deleteIdexFromCollections(deleteIndexName);
    }

    /**
     * 添加索引
     *
     * @Author：SYQ
     * @Date：2022/1/26 13:51
     */
    public void insertIndexFromCollections(Map<String, Object> indexMap) {
        mongodbService.insertIndex(indexMap);
    }

    /**
     * 获取数据源中的businessKey
     *
     * @Author：SYQ
     * @Date：2022/3/10 14:07
     */
    public Map<String, Object> getTmBusinessKeyNew(String tmTaskId, String tmActivityId, JSONObject bpmData) {
        Map<String, Object> resultMap = new HashMap<>();
        //bk值集合
        List<JSONObject> bkValueList = new ArrayList<>();
        List<Object> bkObjectValueList = new ArrayList<>();
        //获取数据源
        Map<String, Object> dataSourcesMap = null;
        if (tmActivityDataSourceMap.containsKey(tmTaskId + tmActivityId) && tmActivityDataSourceMap.get(tmTaskId + tmActivityId) != null) {
            dataSourcesMap = (Map) tmActivityDataSourceMap.get(tmTaskId + tmActivityId);
        } else {
            //获取任务定义
            Map tmActivityMap = themeMapService.getActivityAction2(tmTaskId, tmActivityId, TmPageName.TASK_DETAIL_NAME.getValue());
            if (tmActivityMap != null && tmActivityMap.size() > 0) {
                //获取数据源
                dataSourcesMap = tmActivityMap.containsKey("dataSources") ? (Map) tmActivityMap.get("dataSources") : null;
            }
        }
        if (dataSourcesMap == null || dataSourcesMap.size() == 0) {
            return resultMap;
        }
        //获取key
        Set<String> businessKeySet = new HashSet<>();
        for (Map.Entry<String, Object> entry : dataSourcesMap.entrySet()) {
            getTmActivityParams(businessKeySet, entry.getValue());
        }
        //从bpmData中解析出业务数据
        for (String key : businessKeySet) {
            if (!bpmData.containsKey(key) || bpmData.get(key) == null) {
                continue;
            }
            //集合
            if (bpmData.get(key) instanceof Collection) {
                JSONArray jsonArray = bpmData.getJSONArray(key);
                jsonArray.forEach(obj -> bkValueList.add(JSONObject.fromObject(JsonUtils.objectToString(obj))));
            } else if (bpmData.get(key) instanceof Map) {//对象
                bkValueList.add(bpmData.getJSONObject(key));
            } else {
                bkObjectValueList.add(bpmData.get(key));
            }
        }
        if (!CollectionUtils.isEmpty(bkValueList)) {
            resultMap.put(BkConstant.BK_COMPLEX, bkValueList);
        }
        if (!CollectionUtils.isEmpty(bkValueList)) {
            resultMap.put(BkConstant.BK_SIMPLE, bkObjectValueList);
        }

        return resultMap;
    }

    /**
     * 获取数据源中的流程变量
     *
     * @Author：lisheng
     * @Date：2024/4/2 14:07
     */
    public String getVariableName(String tmTaskId, String tmActivityId) {
        //获取数据源
        Map<String, Object> dataSourcesMap = null;
        if (tmActivityDataSourceMap.containsKey(tmTaskId + tmActivityId) && tmActivityDataSourceMap.get(tmTaskId + tmActivityId) != null) {
            dataSourcesMap = (Map) tmActivityDataSourceMap.get(tmTaskId + tmActivityId);
        } else {
            //获取任务定义
            Map tmActivityMap = themeMapService.getActivityAction2(tmTaskId, tmActivityId, TmPageName.TASK_DETAIL_NAME.getValue());
            if (tmActivityMap != null && tmActivityMap.size() > 0) {
                //获取数据源
                dataSourcesMap = tmActivityMap.containsKey("dataSources") ? (Map) tmActivityMap.get("dataSources") : null;
            }
        }
        if (dataSourcesMap == null || dataSourcesMap.size() == 0) {
            return null;
        }
        //获取key
        Set<String> businessKeySet = new HashSet<>();
        for (Map.Entry<String, Object> entry : dataSourcesMap.entrySet()) {
            getTmActivityParams(businessKeySet, entry.getValue());
        }
        if (CollectionUtils.isEmpty(businessKeySet)) {
            return null;
        }
        //bug:Call "Optional#isPresent()" before accessing the value.
        Optional<String> businessKeyOptional = businessKeySet.stream().findFirst();
        return businessKeyOptional.orElse(null);
    }

    /**
     * 获取数据源中的流程变量
     *
     * @Author：lisheng
     * @Date：2024/4/2 14:07
     */
    public Map<String, String> getVariableName(String tmTaskId, List<String> tmActivityIdList) {
        //获取数据源
        Map<String, String> taskVariableMap = new HashMap<>();
        //获取任务定义
        Map<String, Map> tmActivityMap = themeMapService.getActivityDefinitionListByPageCode(tmTaskId, tmActivityIdList, TmPageName.TASK_DETAIL_NAME.getValue());
        if (MapUtils.isEmpty(tmActivityMap)) {
            return taskVariableMap;
        }
        for (String tmActivityId : tmActivityIdList) {
            Map tmActivity = tmActivityMap.get(tmActivityId);
            if (MapUtils.isEmpty(tmActivity)) {
                continue;
            }
            //获取数据源
            Map<String, Object> dataSourcesMap = (Map) tmActivity.get("dataSources");
            if (MapUtils.isEmpty(dataSourcesMap)) {
                continue;
            }
            //获取key
            Set<String> businessKeySet = new HashSet<>();
            for (Map.Entry<String, Object> entry : dataSourcesMap.entrySet()) {
                getTmActivityParams(businessKeySet, entry.getValue());
            }
            if (CollectionUtils.isEmpty(businessKeySet)) {
                continue;
            }
            //soanr:Call "Optional#isPresent()" before accessing the value.
            Optional<String> businessKeyOptional = businessKeySet.stream().findFirst();
            if(businessKeyOptional.isPresent()){
                taskVariableMap.put(tmActivityId, businessKeyOptional.get());
            }
        }
        return taskVariableMap;
    }

    /**
     * 递归获取任务定义中的actionParams
     *
     * @Author：SYQ
     * @Date：2022/3/10 13:36
     */
    private void getTmActivityParams(Set<String> businessKeySet, Object dataObj) {
        if (dataObj instanceof Collection) {
            List<Object> list = JSONArray.fromObject(dataObj);
            for (Object object : list) {
                getTmActivityParams(businessKeySet, object);
            }
        } else if (dataObj instanceof Map) {
            Map<String, Object> map = JSONObject.fromObject(dataObj);
            if (!map.containsKey(BkConstant.ACTION_PARAMS) || map.get(BkConstant.ACTION_PARAMS) == null) {
                for (Map.Entry<String, Object> entry : map.entrySet()) {
                    getTmActivityParams(businessKeySet, entry.getValue());
                }
            } else {
                List<JSONObject> tmActionParameterDTOS = JSONArray.fromObject(map.get(BkConstant.ACTION_PARAMS));
                for (JSONObject parameterDTO : tmActionParameterDTOS) {
                    if ("PROCESS_VARIABLE".equals(parameterDTO.get("type"))) {
                        businessKeySet.add(parameterDTO.getString("value"));
                        break;
                    }
                }
            }
        }
    }

    /**
     * 新增缺失的数据
     *
     * @Author：SYQ
     * @Date：2022/4/6 16:40
     */
    public List<Long> insertLackDataByParam(Map<String, Object> paramMap) {
        if (paramMap == null || paramMap.size() == 0 || !paramMap.containsKey(BkConstant.TENANT_ID) || paramMap.get(BkConstant.TENANT_ID) == null) {
            return new ArrayList<>();
        }
        String tenantId = paramMap.get(BkConstant.TENANT_ID).toString();
        //根据租户获取任务卡数据
        List<Map<String, Object>> workitemList = bpmActivityWorkitemMapper.selectWorkitemByParam(paramMap);
        if (CollectionUtils.isEmpty(workitemList)) {
            return new ArrayList<>();
        }
        return insertWorkitemBk(tenantId, workitemList);
    }


    /**
     * 新增缺失的数据
     *
     * @Author：SYQ
     * @Date：2022/4/6 16:40
     */
    private List<Long> insertWorkitemBk(String tenantId, List<Map<String, Object>> workitemList) {
        //错误结果
        List<Long> errorWorkitem = new ArrayList<>();
        //任务卡id集合
        List<Long> workitemIdList = new ArrayList<>();
        Map<Long, Map<String, Object>> workitemMap = new HashMap<>();
        for (Map<String, Object> workitemData : workitemList) {
            Long id = Long.valueOf(workitemData.get("id").toString());
            workitemIdList.add(id);
            workitemMap.put(id, workitemData);
        }
        //查找已经存在的任务卡bk
        List<Map> dataList = selectBusinessKeyByTaskId(tenantId, workitemIdList);
        //获取哪些任务没有bk数据
        if (!CollectionUtils.isEmpty(dataList)) {
            for (Map bkData : dataList) {
                Long workitemId = Long.valueOf(bkData.get(BkConstant.CARD_ID).toString());
                //删除已经存在bk的任务卡
                workitemIdList.remove(workitemId);
            }
        }
        //解析没有存bk数据的任务卡的bk
        for (Long workitemId : workitemIdList) {
            try {
                Map<String, Object> workitemData = workitemMap.get(workitemId);
                String tmTaskId = workitemData.get("tmTaskId").toString();
                String tmActivityId = workitemData.get("tmActivityId").toString();
                Object data = workitemData.get("bpmData");
                if (data == null) {
                    continue;
                }
                //转成json
                JSONObject bpmData = JSONObject.fromObject(JsonUtils.objectToString(data));
                //解析bk
                Map<String, Object> bkMap = getTmBusinessKeyNew(tmTaskId, tmActivityId, bpmData);
                //复杂结构的bk
                List<JSONObject> bkValueList = new ArrayList<>();
                if (bkMap.containsKey(BkConstant.BK_COMPLEX)) {
                    bkValueList = JSONArray.fromObject(bkMap.get(BkConstant.BK_COMPLEX));

                }
                //简单结构的bk
                List<Object> bkObjectValueList = new ArrayList<>();
                if (bkMap.containsKey(BkConstant.BK_SIMPLE)) {
                    bkObjectValueList = JSONArray.fromObject(bkMap.get(BkConstant.BK_SIMPLE));
                }
                //记录BusinessKey和项目卡/任务卡的关联关系
                insertOriBk(bkValueList, bkObjectValueList, workitemId, tenantId, BkConstant.WORKITEM);
            } catch (Exception e) {
                errorWorkitem.add(workitemId);
            }
        }
        return errorWorkitem;
    }

    public Object insertLackDataForTaskEngine(TEProjectBKLackData teProjectBKLackData) {
        if (!StringUtils.equals(teProjectBKLackData.getPassword(), "majfa14537")) {
            throw BusinessException.create("密码错误");
        }

        // 未能获取到项目信息的项目id列表
        Map<String, List<Long>> notFindProjects = new HashMap<>();
        // 缺少bpmData数据的项目id列表
        Map<String, List<Long>> lackBpmDataProjects = new HashMap<>();

        // <租户, token>
        Map<String, String> tenantTokenMap = teProjectBKLackData.getTokens().stream()
                .collect(Collectors.toMap(TenantToken::getTenantId, TenantToken::getToken, (oldValue, newValue) -> oldValue));

        // <租户, 项目id列表>
        Map<String, List<Long>> tenantProjectIdMap = new HashMap<>();
        teProjectBKLackData.getProjects().stream().forEach(project ->
                tenantProjectIdMap.computeIfAbsent(project.getTenantId(), tenantId -> new ArrayList<>()).add(project.getId()));

        // 已存在BK记录的项目id列表
        Map<String, List<Long>> existBKProjects = new HashMap<>();
        for (Map.Entry<String, List<Long>> tenantProjectIds : tenantProjectIdMap.entrySet()) {
            String tenantId = tenantProjectIds.getKey();
            List<Map> maps = this.selectBusinessKeyByProjectId(tenantId, tenantProjectIds.getValue());

            if (!CollectionUtils.isEmpty(maps)) {
                Set<Long> cardIds = maps.stream().map(map -> MapUtil.get(map, BkConstant.CARD_ID, Long.class)).collect(Collectors.toSet());
                existBKProjects.computeIfAbsent(tenantId, key -> new ArrayList<>()).addAll(cardIds);
            }
        }

        for (TEProjectBKData project : teProjectBKLackData.getProjects()) {
            // 表中已经有该项目id的BK记录了
            if (null != existBKProjects.get(project.getTenantId()) && existBKProjects.get(project.getTenantId()).contains(project.getId())) {
                continue;
            }

            String token = tenantTokenMap.get(project.getTenantId());
            // 根据项目id从ptm获取项目信息
            PtmProjectRecordBO projectRecord = getProjectRecord(project.getId(), token);
            if (null != projectRecord) {
                if (null != projectRecord.getData() && null != projectRecord.getData().get("data")) {
                    List<JSONObject> bkDataList = parseBKData(projectRecord.getData().get("data"), teProjectBKLackData.getBkFields());
                    if (!CollectionUtils.isEmpty(bkDataList)) {
                        // id还是projectCardId
                        insertOriBk(bkDataList, Collections.emptyList(), project.getId(), project.getTenantId(), BkConstant.TASK);
                    }
                } else {
                    lackBpmDataProjects.computeIfAbsent(project.getTenantId(), tenantId -> new ArrayList<>()).add(project.getId());
                }
            } else {
                notFindProjects.computeIfAbsent(project.getTenantId(), tenantId -> new ArrayList<>()).add(project.getId());
            }
        }

        Map<String, Object> result = new HashMap<>();
        result.put("notFindProjects", notFindProjects);
        result.put("lackBpmDataProjects", lackBpmDataProjects);
        result.put("existBKProjects", existBKProjects);
        return result;
    }

    private List<JSONObject> parseBKData(Object bpmDataObj, List<String> bkFields) {
        List<JSONObject> bkDataList = new ArrayList<>();

        List<Map> bpmDataList = new ArrayList<>();
        if (bpmDataList instanceof Collection) {
            bpmDataList.addAll((List<Map>) bpmDataObj);
        } else {
            bpmDataList.add((Map) bpmDataObj);
        }

        for (Map bpmData : bpmDataList) {
            JSONObject bkData = new JSONObject();
            for (String bkField : bkFields) {
                bkData.put(bkField, MapUtil.get(bpmData, bkField, Object.class));
            }
            bkDataList.add(bkData);
        }

        return bkDataList;
    }

    private PtmProjectRecordBO getProjectRecord(Long projectCardId, String token) {
        resetAuthoredUser(token);
        try {
            return ptmService.getProjectRecord(projectCardId);
        } catch (Exception ex) {
            // 判断token失效的情况、其他异常场景
            return null;
        }
    }

    private void resetAuthoredUser(String token) {
        AppAuthContextHolder.getContext().setAuthoredUser(null);

        AuthoredUser user = new AuthoredUser();
        user.setToken(token);
        AppAuthContextHolder.getContext().setAuthoredUser(user);
    }

}
