package com.digiwin.athena.atmc.common.service.ptm;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.application.configuration.EnvProperties;
import com.digiwin.athena.atmc.common.domain.eventbus.SyncBackLogEventDTO;
import com.digiwin.athena.atmc.core.meta.enums.ActivityState;
import com.digiwin.athena.atmc.core.meta.enums.ptm.PtmBacklogTypeEnum;
import com.digiwin.athena.atmc.infrastructure.mapper.biz.migration.PtmBacklogMapper;
import com.digiwin.athena.atmc.common.dao.ptm.PtmProjectCardMapper;
import com.digiwin.athena.atmc.application.dto.response.backlog.PtmWorkItemDetailResp;
import com.digiwin.athena.atmc.application.dto.response.eoc.ProxyUserResp;
import com.digiwin.athena.atmc.application.utils.SummaryMessagesUtils;
import com.digiwin.athena.atmc.common.dao.ptm.PtmProjectCardMapper;
import com.digiwin.athena.atmc.common.domain.BpmProcess;
import com.digiwin.athena.atmc.common.enums.WorkitemCreateType;
import com.digiwin.athena.atmc.common.service.tenantToken.TenantTokenService;
import com.digiwin.athena.atmc.common.util.StringUtil;
import com.digiwin.athena.atmc.core.meta.constants.GlobalConstant;
import com.digiwin.athena.atmc.core.meta.enums.ActivityState;
import com.digiwin.athena.atmc.core.meta.enums.km.TmTaskCategoryEnum;
import com.digiwin.athena.atmc.core.meta.enums.ptm.PtmBacklogTypeEnum;
import com.digiwin.athena.atmc.http.constant.BpmConstant;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.domain.Task;
import com.digiwin.athena.atmc.http.restful.eoc.EocService;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocEmployeeDTO;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocTenantAllUserDTO;
import com.digiwin.athena.atmc.http.restful.ptm.PtmService;
import com.digiwin.athena.atmc.http.restful.ptm.model.*;
import com.digiwin.athena.atmc.http.restful.thememap.ThemeMapService;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmPageName;
import com.digiwin.athena.atmc.core.meta.enums.km.TmTaskCategoryEnum;
import com.google.common.eventbus.AsyncEventBus;
import com.digiwin.athena.atmc.infrastructure.mapper.biz.migration.PtmBacklogMapper;
import com.digiwin.athena.atmc.infrastructure.pojo.bo.migration.*;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.Backlog;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.BpmActivityWorkitem;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmBacklog;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmProjectCard;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StopWatch;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 数据转换服务(新)
 */
@Slf4j
@Service
public class CommonPtmTransformService {
    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private PtmService ptmService;

    @Autowired
    private TenantTokenService tenantTokenService;

    @Autowired
    private PtmBacklogMapper ptmBacklogMapper;

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private EocService eocService;

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    private PtmProjectCardMapper ptmProjectCardMapper;

    @Autowired
    private CommonPtmTransformService commonPtmTransformService;

    @Autowired
    private AsyncEventBus localEventBus;

    public boolean enablePTM() {
        return StringUtils.isNotBlank(envProperties.getPtmUri());
    }

    /**
     * 整理合并任务数据
     *
     * @Author：SYQ
     * @Date：2022/6/16 18:15
     */
    public Map<Long, PtmMergeBacklogBO> dealBacklogData(List<Long> backlogIdList, Boolean history) {
        if (history == null) {
            history = false;
        }
        if (CollectionUtils.isEmpty(backlogIdList)) {
            return new HashMap<>();
        }
        //获取任务详情
        List<PtmBacklogItemBO> ptmBacklogItemBOList = ptmService.getBacklogByBacklogIds(backlogIdList);
        if (CollectionUtils.isEmpty(ptmBacklogItemBOList)) {
            return null;
        }
        //按backlogId分组
        Map<Long, List<PtmBacklogItemBO>> ptmBacklogItemMap = ptmBacklogItemBOList.stream().collect(Collectors.groupingBy(PtmBacklogItemBO::getBacklogId));
        //整理任务合并数据
        Map<Long, PtmMergeBacklogBO> resultMap = new HashMap<>();
        for (Map.Entry<Long, List<PtmBacklogItemBO>> entry : ptmBacklogItemMap.entrySet()) {
            Long backlogId = entry.getKey();
            List<PtmBacklogItemBO> itemDTOList = entry.getValue();
            //如果是已完成数据，则按照预计完成日升序排序；若是未完成数据，去除已完成的任务并且按照预计完成日升序排序
            List<LocalDateTime> planEndTimeList = null;
            if (history) {
                planEndTimeList = itemDTOList.stream().sorted(Comparator.comparing(PtmBacklogItemBO::getPlanEndTime))
                        .map(PtmBacklogItemBO::getPlanEndTime)
                        .collect(Collectors.toList());
            } else {
                planEndTimeList = itemDTOList.stream().filter(item -> !isWorkItemClosed(item.getState()))
                        .sorted(Comparator.comparing(PtmBacklogItemBO::getPlanEndTime))
                        .map(PtmBacklogItemBO::getPlanEndTime)
                        .collect(Collectors.toList());
            }

            //子任务是否存在未完成的
            List<PtmBacklogItemBO> pendingList = itemDTOList.stream().filter(item -> !isWorkItemClosed(item.getState()))
                    .collect(Collectors.toList());

            PtmMergeBacklogBO ptmMergeBacklogBO = new PtmMergeBacklogBO();
            ptmMergeBacklogBO.setBacklogId(backlogId);
            ptmMergeBacklogBO.setTaskName(itemDTOList.get(0).getTaskName());
            if (CollectionUtils.isNotEmpty(planEndTimeList)) {
                ptmMergeBacklogBO.setPlanEndTimeMin(planEndTimeList.get(0));
                ptmMergeBacklogBO.setPlanEndTimeMax(planEndTimeList.get(planEndTimeList.size() - 1));
            }
            ptmMergeBacklogBO.setClosed(CollectionUtils.isEmpty(pendingList));
            resultMap.put(backlogId, ptmMergeBacklogBO);
        }
        return resultMap;
    }

    private boolean isWorkItemClosed(Integer state) {
        return Objects.nonNull(state) && state >= ActivityState.COMPLETED.getCode();
    }

    public List<ActivityDataBO> getActivityDataByBacklogId(Long backlogId, Boolean isHistory) {
        return getActivityDataByBacklogId(backlogId, isHistory, null);
    }

    /**
     * 获取待办的显示数据
     * 针对任务提交链路优化后的方法
     *
     * @param workitemId 待办id
     * @param backlogId  任务卡id
     */
    public List<ActivityDataBO> getActivityDataByBacklogIdV2(Long workitemId, Long backlogId) {
        List<ActivityDataBO> result = new ArrayList<>();
        //获取ptm待办详情信息
        List<PtmWorkItemDetailResp> workItemDetailList = ptmService.getWorkItemDetailList(workitemId, backlogId, false);
        if (CollectionUtils.isNotEmpty(workItemDetailList)) {
            //获取atmc待办详情信息
            PtmBacklog ptmBacklogSelf = commonPtmTransformService.getBacklogByIdFromDb(workItemDetailList.get(0).getBacklogId());
            //获取任务关联的bpmdata数据
            List<PtmTaskCardDataDTO> taskCardDataList = ptmService.getTaskDataByBacklogId(backlogId, "");

            workItemDetailList.forEach(bl -> {
                PtmBacklog ptmBacklog = null == ptmBacklogSelf ? buildPtmBacklogV2(bl) : ptmBacklogSelf;

                //活动数据赋值
                ActivityDataBO activityDataBO = getActivityDataBO(bl, ptmBacklog, taskCardDataList);
                //事项数据赋值
                BpmActivityWorkitem bpmWorkitem = getBpmActivityWorkitem(bl, ptmBacklog);

                activityDataBO.setWorkitemList(JsonUtils.objectToString(Arrays.asList(bpmWorkitem)));
                result.add(activityDataBO);
            });
        }
        return result;
    }

    /**
     * 整合任务基本信息
     *
     * @param bl
     * @param ptmBacklog
     * @param taskCardDataList
     * @return
     */
    private ActivityDataBO getActivityDataBO(PtmWorkItemDetailResp bl, PtmBacklog ptmBacklog, List<PtmTaskCardDataDTO> taskCardDataList) {
        ActivityDataBO activityDataBO = ActivityDataBO.builder()
                .bpmActivitySqlId(bl.getTaskId())
                .bpmData(null)
                .bpmStateData(null)
                .tmActivityId(bl.getTaskDefCode())
                .tmPattern(bl.getTaskDefPattern())
                .tmCategory(bl.getTaskDefCategory())
                .bpmActivityType(bl.getBpmnType())
                .bpmActivityId(bl.getBpmActivityId())
                .performerIds(null)
                .processSerialNumber(bl.getProcessSerialNumber())
                .backlogId(bl.getBacklogId())
                .backlogName(bl.getTaskName())
                .targetTenantId(ptmBacklog.getTargetTenantId())
                .performerId(bl.getPerformerId())
                .performerName(bl.getPerformerName())
                .startTime(bl.getTaskCreateTime())
                .planEndTime(bl.getTaskPlanEndTime())
                .endTime(bl.getClosedTime())
                .taskId(bl.getProjectId())
                .tmTaskId(bl.getProjectDefCode())
                .chargeId(bl.getPersonInCharge())
                .chargeName(bl.getPersonInChargeName())
                .tenantId(bl.getProjectTenantId())
                .businessUnit(JsonUtils.objectToString(bl.getBusinessUnit()))
                .taskStartTime(bl.getProjectStartTime())
                .taskEndTime(bl.getProjectEndTime())
                .taskName(bl.getProjectName())
                .tmActivityName(bl.getTaskDefName())
                .finishedActionId(ptmBacklog.getClosed() ? ptmBacklog.getActionId() : null)
                .tmShowFlow(false)
                .stepId(bl.getActivityId())
                .taskSourceIds(bl.getSourceIds())
                .stepSignReason(bl.getSignReason())
                .approvalState(ptmBacklog.getApprovalState())
                .backlogClosed(ptmBacklog.getClosed() || bl.getClosed())
                .merge(bl.getMerge())
                .proxyToken(StringUtils.isNotBlank(bl.getTaskProxyToken()) ? bl.getTaskProxyToken() : bl.getProjectProxyToken())
                .processCreateTime(bl.getProjectCreatedTime())
                .compositionId("")
                .overdueWorkitemId(0L)
                .workItemId(bl.getWorkItemId())
                .taskUid(bl.getBpmTaskUid())
                .preTaskUid(null)
                .stepState(bl.getActivityState())
                .stepSubState(bl.getActivitySubState())
                .traceId(bl.getTraceId())
                .eocName(bl.getEocName())
                .dataFrom(BpmConstant.DATA_FROM_PTM)
                .ptmBacklogId(bl.getBacklogId())
                .ptmWorkItemId(bl.getWorkItemId())
                .type(SummaryMessagesUtils.translateBacklogType(ptmBacklog.getType()))
                .sourceWorkitemId(ptmBacklog.getSourceBacklogId())
                .merge(ptmBacklog.getMerge())
                .backlogPerformId(ptmBacklog.getPerformerId())
                .build();

        // 任务/项目上有代理token
        if ((StringUtils.isNotBlank(bl.getProjectProxyToken()) || StringUtils.isNotBlank(bl.getTaskProxyToken()))) {
            // 租户不相同，则根据租户获取虚拟token作为代理新的proxyToken
            if (!StringUtils.equals(bl.getProjectTenantId(), ptmBacklog.getTargetTenantId())) {
                activityDataBO.setProxyToken(tenantTokenService.queryVirtualToken(bl.getProjectTenantId()));
            }
            // 租户相同，理论上不需要proxyToken，但PWD等应用的客制作业需要，贸然去掉会导致这些作业客制逻辑报错，因此这里使用登录token塞给proxyToken
            else {
                activityDataBO.setProxyToken(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
            }
        }
        // 任务/项目上无代理token
        else {
            activityDataBO.setProxyToken(null);
        }

        Optional<PtmTaskCardDataDTO> taskData = taskCardDataList.stream().filter(t -> Objects.equals(t.getTaskId(), bl.getTaskId())).findFirst();

        if (taskData.isPresent()) {
            PtmTaskCardDataDTO td = taskData.get();
            if (td.getData() != null) {
                activityDataBO.setBpmData(JsonUtils.objectToString(td.getData()));
            }
        }
        return activityDataBO;
    }

    /**
     * 整合待办信息
     *
     * @param bl
     * @param ptmBacklog
     * @return
     */
    private BpmActivityWorkitem getBpmActivityWorkitem(PtmWorkItemDetailResp bl, PtmBacklog ptmBacklog) {
        return BpmActivityWorkitem.builder()
                .id(bl.getWorkItemId())
                .activityStepId(bl.getActivityId())
                .createTime(bl.getWorkItemCreateTime())
                .closedTime(bl.getClosedTime())
                .performerId(bl.getPerformerId())
                .performerName(bl.getPerformerName())
                .workitemId(bl.getTeWorkItemId())
                .comment(bl.getComment())
                .state(bl.getWorkItemState())
                .agentPerformerId(bl.getAgentPerformerId())
                .agentPerformerName(bl.getAgentPerformerName())
                .performerType(bl.getPerformerType())
                .actionId(ptmBacklog.getClosed() ? ptmBacklog.getActionId() : null)
                .subState(bl.getWorkItemSubState())
                .createType(bl.getCreateType())
                .approvalState(ptmBacklog.getApprovalState())
                .fromWorkitemId(bl.getFromWorkItemId())
                .groupId(bl.getGroupId())
                .targetTenantId(bl.getWorkItemTenantId())
                .importanceSource("self")
                .commonPerformerId(null)
                .build();
    }

    /**
     * 获取待办的显示数据
     *
     * @param backlogId
     * @param ptmWorkItemRecordBO
     * @return
     */
    public List<ActivityDataBO> getActivityDataByBacklogId(Long backlogId, Boolean isHistory, PtmWorkItemRecordBO ptmWorkItemRecordBO) {
        List<ActivityDataBO> result = new ArrayList<>();
        if (!enablePTM()) {
            return result;
        }
        List<PtmBacklogItemBO> backlogItemList = new ArrayList<>();
        PtmBacklog overdueBacklog = ptmBacklogMapper.selectOverdueById(backlogId);
        PtmBacklog dataUniformityBacklog = null;
        if (overdueBacklog != null) {
            //逾时任务卡
            backlogId = overdueBacklog.getOverdueBacklogId();
        } else {
            //数据一致性异常排除任务卡
            dataUniformityBacklog = ptmBacklogMapper.selectDataUniformityById(backlogId);
            if (dataUniformityBacklog != null) {
                backlogId = dataUniformityBacklog.getSourceBacklogId();
            }
        }

        backlogItemList = ptmService.getBacklogByBacklogId(backlogId);
        if (CollectionUtils.isEmpty(backlogItemList)) {
            //可能是workitemId,只留下该workitem的，其他不需要
            Long workitemId = backlogId;
            List<PtmBacklog> backlogList = ptmBacklogMapper.selectByWorkItemId(backlogId);
            if (CollectionUtils.isEmpty(backlogList)) {
                //可能是合并的某个workitem
                //性能优化：如果前面已经查过了，不用再查
                PtmWorkItemRecordBO workItemRecordDTO = Objects.nonNull(ptmWorkItemRecordBO)
                        ? ptmWorkItemRecordBO : ptmService.getWorkItemRecord(backlogId);
                if (workItemRecordDTO != null) {
                    backlogList = new ArrayList<>();
                    PtmBacklog ptmBacklog = new PtmBacklog();
                    ptmBacklog.setBacklogId(workItemRecordDTO.getBacklogId());
                    backlogList.add(ptmBacklog);
                }
            }
            if (CollectionUtils.isNotEmpty(backlogList)) {
                backlogId = backlogList.get(0).getBacklogId();
                backlogItemList = ptmService.getBacklogByBacklogId(backlogId);
                backlogItemList = backlogItemList.stream().filter(x -> Objects.equals(x.getWorkItemId(), workitemId)).collect(Collectors.toList());
            }
        }
        if (CollectionUtils.isNotEmpty(backlogItemList)) {
            PtmBacklog ptmBacklogSelf = commonPtmTransformService.getBacklogByIdFromDb(backlogId);
            PtmBacklogRecordDTO ptmBacklogRecord = ptmService.getBacklogRecord(backlogId);
            List<PtmTaskCardDataDTO> taskCardDataList = ptmService.getTaskDataByBacklogId(backlogId, "");
            // 构建ptmBacklog
            if (null == ptmBacklogSelf) {
                ptmBacklogSelf = buildPtmBacklog(ptmBacklogRecord);
            }
            PtmBacklog ptmBacklog = ptmBacklogSelf;

            // 判断任务卡是否已完成
            if (null == isHistory || BooleanUtils.isFalse(isHistory)) {
                isHistory = BooleanUtils.isTrue(ptmBacklogRecord.getClosed());
            }
            //获取任务合并信息
            Map<Long, PtmMergeBacklogBO> mergeBacklogDTOMap = this.dealBacklogData(backlogItemList.stream()
                    .map(PtmBacklogItemBO::getBacklogId).collect(Collectors.toList()), isHistory);

            PtmBacklog finalDataUniformityBacklog = dataUniformityBacklog;
            //modify by gonghx  at 2021-10-13 for 性能优化，科盛德任务卡为多张合并，效能问题，使用批量的方式处理数据start
            //批量获取taskRecord信息
            List<PtmTaskRecordBO> ptmTaskRecordList = ptmService.getTaskRecordByTaskIds(backlogItemList.stream().map(PtmBacklogItemBO::getTaskId).distinct().collect(Collectors.toList()));
            Map<Long, PtmTaskRecordBO> ptmTaskRecordMap = ptmTaskRecordList.stream().collect(Collectors.toMap(PtmTaskRecordBO::getId, v -> v, (p1, p2) -> p1));
            //批量获取的activityRecord信息
            List<PtmActivityRecordDTO> ptmActivityRecordList = ptmService.getActivityRecordByActivityIds(backlogItemList.stream().map(PtmBacklogItemBO::getActivityId).distinct().collect(Collectors.toList()));
            Map<Long, PtmActivityRecordDTO> ptmActivityRecordMap = ptmActivityRecordList.stream().collect(Collectors.toMap(PtmActivityRecordDTO::getId, v -> v, (p1, p2) -> p1));
            //批量获取projectRecord信息
            List<PtmProjectRecordBO> ptmProjectRecordList = ptmService.getProjectRecordByProjectIds(backlogItemList.stream().map(PtmBacklogItemBO::getProjectId).distinct().collect(Collectors.toList()));
            Map<Long, PtmProjectRecordBO> ptmProjectRecordMap = ptmProjectRecordList.stream().collect(Collectors.toMap(PtmProjectRecordBO::getId, v -> v, (p1, p2) -> p1));
            //批量获取WorkItem信息
            List<PtmWorkItemRecordBO> workItemRecordList = ptmService.getWorkItemRecordByWorkItemIds(backlogItemList.stream().map(PtmBacklogItemBO::getWorkItemId).distinct().collect(Collectors.toList()));
            Map<Long, PtmWorkItemRecordBO> ptmWorkItemRecordMap = workItemRecordList.stream().collect(Collectors.toMap(PtmWorkItemRecordBO::getId, v -> v, (p1, p2) -> p1));

            //根据异常排除任务id查询发起的签核任务，先判断是不是签核任务
            List<PtmSolveTask> ptmSolveTasks = new ArrayList<>();
            BacklogBO backlogBO = this.selectBacklogForCard(backlogId);
            if (backlogBO != null && org.springframework.util.StringUtils.pathEquals(TmTaskCategoryEnum.SOLVE.getValue(), backlogBO.getTmCategory())) {
                ptmSolveTasks = ptmService.getSolveTaskListV2(backlogId);
            }
            List<PtmSolveTask> finalPtmSolveTasks = ptmSolveTasks;

            //modify by gonghx  at 2021-10-13 for 性能优化，科盛德任务卡为多张合并，效能问题，使用批量的方式处理数据end

            backlogItemList.forEach(bl -> {
                //获取task信息
                PtmTaskRecordBO ptmTaskRecord = ptmTaskRecordMap.get(bl.getTaskId());
                //获取activity信息
                PtmActivityRecordDTO ptmActivityRecord = ptmActivityRecordMap.get(bl.getActivityId());
                //获取project信息
                PtmProjectRecordBO ptmProjectRecord = ptmProjectRecordMap.get(bl.getProjectId());
                //获取workitem信息
                PtmWorkItemRecordBO ptmWorkItemRecord = ptmWorkItemRecordMap.get(bl.getWorkItemId());

                PtmMergeBacklogBO ptmMergeBacklogBO = mergeBacklogDTOMap.get(ptmBacklogRecord.getId());

                //region 创建基本信息
                ActivityDataBO activityDataBO = ActivityDataBO.builder()
                        .bpmActivitySqlId(ptmTaskRecord.getId())
                        .bpmData(null)
                        .bpmStateData(null)
                        .tmActivityId(ptmTaskRecord.getTaskDefCode())
                        .tmPattern(ptmTaskRecord.getTaskDefPattern())
                        .tmCategory(ptmTaskRecord.getTaskDefCategory())
                        .bpmActivityType(ptmTaskRecord.getBpmnType())
                        .bpmActivityId(ptmActivityRecord.getBpmActivityId())
                        .performerIds(null) //不需要
                        .processSerialNumber(ptmProjectRecord.getProcessSerialNumber())
                        .backlogId(ptmBacklogRecord.getId())
                        .backlogName(ptmTaskRecord.getTaskName())
                        .targetTenantId(ptmBacklog.getTargetTenantId())// ptmBacklogRecord.getTenantId()
                        .performerId(ptmWorkItemRecord.getPerformerId())
                        .performerName(ptmWorkItemRecord.getPerformerName())
                        .readCount(ptmBacklog.getReadCount())// 0
                        .startTime(ptmTaskRecord.getCreateTime())
                        .planEndTime(ptmTaskRecord.getPlanEndTime())
                        .endTime(bl.getClosedTime())
                        .taskId(ptmBacklogRecord.getProjectId())
                        .tmTaskId(ptmProjectRecord.getProjectDefCode())
                        .chargeId(ptmProjectRecord.getPersonInCharge())
                        .chargeName(ptmProjectRecord.getPersonInChargeName())
                        .tenantId(ptmProjectRecord.getTenantId())
                        .businessUnit(JsonUtils.objectToString(ptmTaskRecord.getBusinessUnit()))
                        .taskStartTime(ptmProjectRecord.getStartTime())
                        .taskEndTime(ptmProjectRecord.getEndTime())
                        .taskName(ptmProjectRecord.getProjectName())
                        .tmActivityName(ptmTaskRecord.getTaskDefName())
                        .finishedActionId(ptmBacklog.getClosed() ? ptmBacklog.getActionId() : null)// ptmBacklogRecord.getClosed()
                        .tmShowFlow(false)
                        .stepId(ptmActivityRecord.getId())
                        .taskSourceIds(ptmProjectRecord.getSourceIds())
                        .stepSignReason(ptmActivityRecord.getSignReason())
                        .approvalState(ptmBacklog.getApprovalState())
                        .backlogClosed(ptmBacklog.getClosed() || bl.getClosed())
                        .merge(ptmBacklogRecord.getMerge())
                        .proxyToken(StringUtils.isNotBlank(ptmTaskRecord.getProxyToken()) ? ptmTaskRecord.getProxyToken() : ptmProjectRecord.getProxyToken())
                        .processCreateTime(ptmProjectRecord.getCreateTime())
                        .compositionId("")
                        .overdueWorkitemId(0L)
                        .workItemId(ptmWorkItemRecord.getId())
                        .taskUid(ptmTaskRecord.getBpmTaskUid())
                        .preTaskUid(null)
                        .stepState(ptmActivityRecord.getState())
                        .stepSubState(ptmActivityRecord.getSubState())
                        .traceId(ptmProjectRecord.getTraceId())
                        // 优先使用backlog.eocName，没有则使用project.eocName
                        .eocName(StringUtils.isNotBlank(bl.getEocCode()) ? bl.getEocName() : ptmProjectRecord.getEocName())
                        .dataFrom(BpmConstant.DATA_FROM_PTM)
                        .ptmBacklogId(ptmBacklogRecord.getId())
                        .ptmWorkItemId(ptmWorkItemRecord.getId())
                        .ptmActivityId(ptmWorkItemRecord.getActivityId())
                        .type(SummaryMessagesUtils.translateBacklogType(ptmBacklog.getType()))
                        .sourceWorkitemId(ptmBacklog.getSourceBacklogId())//0
                        .merge(ptmBacklog.getMerge())
                        .backlogPerformId(ptmBacklog.getPerformerId())
                        .build();

                if (ptmBacklog.getMerge() && ptmMergeBacklogBO != null) {
                    activityDataBO.setPlanEndTimeMin(ptmMergeBacklogBO.getPlanEndTimeMin());
                    activityDataBO.setPlanEndTimeMax(ptmMergeBacklogBO.getPlanEndTimeMax());
                }

                // 任务/项目上有代理token
                if ((StringUtils.isNotBlank(ptmProjectRecord.getProxyToken()) || StringUtils.isNotBlank(ptmTaskRecord.getProxyToken()))) {
                    // 租户不相同，则根据租户获取虚拟token作为代理新的proxyToken
                    if (!StringUtils.equals(ptmProjectRecord.getTenantId(), ptmBacklog.getTargetTenantId())) {
                        activityDataBO.setProxyToken(tenantTokenService.queryVirtualToken(ptmProjectRecord.getTenantId()));
                    }
                    // 租户相同，理论上不需要proxyToken，但PWD等应用的客制作业需要，贸然去掉会导致这些作业客制逻辑报错，因此这里使用登录token塞给proxyToken
                    else {
                        activityDataBO.setProxyToken(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
                    }
                }
                // 任务/项目上无代理token
                else {
                    activityDataBO.setProxyToken(null);
                }

                //endregion

                //region workitemList
                BpmActivityWorkitem bpmWorkitem = SummaryMessagesUtils.translatePtmWorkItem(ptmBacklog, ptmWorkItemRecord);
                activityDataBO.setWorkitemList(JsonUtils.objectToString(Arrays.asList(bpmWorkitem)));
                //endregion

                Optional<PtmTaskCardDataDTO> taskData = taskCardDataList.stream().filter(t -> Objects.equals(t.getTaskId(), ptmTaskRecord.getId())).findFirst();

                if (taskData.isPresent()) {
                    PtmTaskCardDataDTO td = taskData.get();
                    if (td.getData() != null) {
                        activityDataBO.setBpmData(JsonUtils.objectToString(td.getData()));
                        if (CollectionUtils.isNotEmpty(finalPtmSolveTasks)) {
                            Map stateData = this.getStateData(td.getData(), finalPtmSolveTasks, ptmActivityRecord, ptmWorkItemRecord);
                            if (MapUtils.isNotEmpty(stateData)) {
                                activityDataBO.setBpmStateData(JsonUtils.objectToString(stateData));
                            }
                        }
                    }
                }

                if (overdueBacklog != null) {
                    //region 逾时卡特殊处理
                    activityDataBO.setBacklogId(overdueBacklog.getBacklogId());
                    activityDataBO.setPtmBacklogId(overdueBacklog.getBacklogId());
                    activityDataBO.setPtmWorkItemId(overdueBacklog.getBacklogId());
                    activityDataBO.setWorkItemId(overdueBacklog.getBacklogId());

                    activityDataBO.setOverdueWorkitemId(overdueBacklog.getOverdueBacklogId());
                    activityDataBO.setBacklogClosed(overdueBacklog.getClosed());
                    activityDataBO.setReadCount(overdueBacklog.getReadCount());
                    activityDataBO.setPerformerId(overdueBacklog.getPerformerId());
                    activityDataBO.setPrePerformerName(overdueBacklog.getPerformerName());

                    bpmWorkitem.setId(overdueBacklog.getBacklogId());
                    bpmWorkitem.setOverdueWorkitemId(overdueBacklog.getOverdueBacklogId());
                    activityDataBO.setWorkitemList(JsonUtils.objectToString(Arrays.asList(bpmWorkitem)));
                    //endregion
                }

                if (finalDataUniformityBacklog != null) {
                    activityDataBO.setType(finalDataUniformityBacklog.getType());
                    activityDataBO.setSourceWorkitemId(finalDataUniformityBacklog.getSourceBacklogId());
                }

                // 项目流程变量
                if (null != ptmProjectRecord.getData()) {
                    activityDataBO.setProjectBpmData(JsonUtils.objectToString(ptmProjectRecord.getData()));
                }

                result.add(activityDataBO);
            });
        }

        return result;
    }

    public List<ActivityDataBO> getActivityDataByBacklogId(List<Map<String, Object>> list, Boolean isHistory) {
        return getActivityDataByBacklogId(list, isHistory, false);
    }


    public List<ActivityDataBO> getActivityDataByBacklogId(List<Map<String, Object>> list, Boolean isHistory, boolean isAllQuery) {
        List<Long> backlogIds = list.stream().map(e -> (Long) e.get("backlogId")).collect(Collectors.toList());
        StopWatch stopWatch = new StopWatch();
        stopWatch.start("ptm server");
        List<PtmBacklogItemBO> backlogItemList = ptmService.getBacklogByBacklogIds(backlogIds);
        List<ActivityDataBO> result = new ArrayList<>();
        List<PtmBacklogRecordDTO> backlogRecordByBacklogIds = ptmService.getBacklogRecordByBacklogIds(backlogIds);
        Map<Long, PtmBacklogRecordDTO> ptmBacklogRecordDTOMap = backlogRecordByBacklogIds.stream().collect(Collectors.toMap(PtmBacklogRecordDTO::getId, v -> v, (p1, p2) -> p1));

//        List<PtmTaskCardDataDTO> taskCardDataList = ptmService.getTaskDataByBacklogIds(backlogIds, "data", "true");
        List<PtmTaskCardDataDTO> taskCardDataList = ptmService.getTaskDataByBacklogIds(list, "data", isAllQuery ? "true" : "false");
        stopWatch.stop();
        stopWatch.start("ptm batch data");
        Map<Long, PtmTaskCardDataDTO> taskCardDataDTOMap;
        if (isAllQuery) {
            taskCardDataDTOMap = taskCardDataList.stream().collect(Collectors.toMap(p -> p.getWorkItemId(), p -> p, (p1, p2) -> p1));
        } else {
            taskCardDataDTOMap = taskCardDataList.stream().collect(Collectors.toMap(p -> p.getBacklogId(), p -> p, (p1, p2) -> p1));
        }
        List<PtmBacklog> backlogList = ptmBacklogMapper.selectBatchIds(backlogIds);
        Map<Long, PtmBacklog> ptmBacklogMap = backlogList.stream().collect(Collectors.toMap(PtmBacklog::getBacklogId, v -> v, (p1, p2) -> p1));


        //批量获取taskRecord信息
        List<PtmTaskRecordBO> ptmTaskRecordList = ptmService.getTaskRecordByTaskIds(backlogItemList.stream().map(PtmBacklogItemBO::getTaskId).distinct().collect(Collectors.toList()));
        Map<Long, PtmTaskRecordBO> ptmTaskRecordMap = ptmTaskRecordList.stream().collect(Collectors.toMap(PtmTaskRecordBO::getId, v -> v, (p1, p2) -> p1));
        //批量获取的activityRecord信息
        List<PtmActivityRecordDTO> ptmActivityRecordList = ptmService.getActivityRecordByActivityIds(backlogItemList.stream().map(PtmBacklogItemBO::getActivityId).distinct().collect(Collectors.toList()));
        Map<Long, PtmActivityRecordDTO> ptmActivityRecordMap = ptmActivityRecordList.stream().collect(Collectors.toMap(PtmActivityRecordDTO::getId, v -> v, (p1, p2) -> p1));
        //批量获取projectRecord信息
        List<PtmProjectRecordBO> ptmProjectRecordList = ptmService.getProjectRecordByProjectIds(backlogItemList.stream().map(PtmBacklogItemBO::getProjectId).distinct().collect(Collectors.toList()));
        Map<Long, PtmProjectRecordBO> ptmProjectRecordMap = ptmProjectRecordList.stream().collect(Collectors.toMap(PtmProjectRecordBO::getId, v -> v, (p1, p2) -> p1));
        //批量获取WorkItem信息
        List<PtmWorkItemRecordBO> workItemRecordList = ptmService.getWorkItemRecordByWorkItemIds(backlogItemList.stream().map(PtmBacklogItemBO::getWorkItemId).distinct().collect(Collectors.toList()));
        Map<Long, PtmWorkItemRecordBO> ptmWorkItemRecordMap = workItemRecordList.stream().collect(Collectors.toMap(PtmWorkItemRecordBO::getId, v -> v, (p1, p2) -> p1));

        backlogItemList.forEach(bl -> {
            PtmBacklogRecordDTO ptmBacklogRecord = ptmBacklogRecordDTOMap.get(bl.getBacklogId());
            PtmBacklog ptmBacklog = ptmBacklogMap.get(bl.getBacklogId());
            if (ptmBacklog == null) {
                ptmBacklog = buildPtmBacklog(ptmBacklogRecord);
            }
            //获取task信息
            PtmTaskRecordBO ptmTaskRecord = ptmTaskRecordMap.get(bl.getTaskId());
            //获取activity信息
            PtmActivityRecordDTO ptmActivityRecord = ptmActivityRecordMap.get(bl.getActivityId());
            //获取project信息
            PtmProjectRecordBO ptmProjectRecord = ptmProjectRecordMap.get(bl.getProjectId());
            //获取workitem信息
            PtmWorkItemRecordBO ptmWorkItemRecord = ptmWorkItemRecordMap.get(bl.getWorkItemId());

            ActivityDataBO activityDataBO = ActivityDataBO.builder()
                    .bpmActivitySqlId(ptmTaskRecord.getId())
                    .tmActivityId(ptmTaskRecord.getTaskDefCode())
                    .tmPattern(ptmTaskRecord.getTaskDefPattern())
                    .tmCategory(ptmTaskRecord.getTaskDefCategory())
                    .bpmActivityType(ptmTaskRecord.getBpmnType())
                    .bpmActivityId(ptmActivityRecord.getBpmActivityId())
                    .backlogName(ptmTaskRecord.getTaskName())
                    .targetTenantId(ptmBacklog.getTargetTenantId())// ptmBacklogRecord.getTenantId()
                    .performerId(ptmWorkItemRecord.getPerformerId())
                    .performerName(ptmWorkItemRecord.getPerformerName())
                    .readCount(ptmBacklog.getReadCount())// 0
                    .startTime(ptmTaskRecord.getCreateTime())
                    .planEndTime(ptmTaskRecord.getPlanEndTime())
                    .endTime(bl.getClosedTime())
                    .taskId(ptmBacklogRecord.getProjectId())
                    .tmTaskId(ptmProjectRecord.getProjectDefCode())
                    .chargeId(ptmProjectRecord.getPersonInCharge())
                    .processSerialNumber(ptmProjectRecord.getProcessSerialNumber())
                    .chargeName(ptmProjectRecord.getPersonInChargeName())
                    .tenantId(ptmProjectRecord.getTenantId())
                    .businessUnit(JsonUtils.objectToString(ptmTaskRecord.getBusinessUnit()))
                    .taskStartTime(ptmProjectRecord.getStartTime())
                    .taskEndTime(ptmProjectRecord.getEndTime())
                    .taskName(ptmProjectRecord.getProjectName())
                    .tmActivityName(ptmTaskRecord.getTaskDefName())
                    .finishedActionId(ptmBacklog.getClosed() ? ptmBacklog.getActionId() : null)// ptmBacklogRecord.getClosed()
                    .tmShowFlow(false)
                    .stepId(ptmActivityRecord.getId())
                    .taskSourceIds(ptmProjectRecord.getSourceIds())
                    .stepSignReason(ptmActivityRecord.getSignReason())
                    .approvalState(ptmBacklog.getApprovalState())
                    .backlogClosed(ptmBacklog.getClosed() || bl.getClosed())
                    .merge(ptmBacklogRecord.getMerge())
                    .proxyToken(StringUtils.isNotBlank(ptmTaskRecord.getProxyToken()) ? ptmTaskRecord.getProxyToken() : ptmProjectRecord.getProxyToken())
                    .processCreateTime(ptmProjectRecord.getCreateTime())
                    .compositionId("")
                    .overdueWorkitemId(0L)
                    .workItemId(ptmWorkItemRecord.getId())
                    .taskUid(ptmTaskRecord.getBpmTaskUid())
                    .preTaskUid(null)
                    .stepState(ptmActivityRecord.getState())
                    .stepSubState(ptmActivityRecord.getSubState())
                    .traceId(ptmProjectRecord.getTraceId())
                    .eocName(ptmProjectRecord.getEocName())
                    .dataFrom(BpmConstant.DATA_FROM_PTM)
                    .ptmBacklogId(ptmBacklogRecord.getId())
                    .ptmWorkItemId(ptmWorkItemRecord.getId())
                    .ptmActivityId(ptmWorkItemRecord.getActivityId())
                    .type(SummaryMessagesUtils.translateBacklogType(ptmBacklog.getType()))
                    .sourceWorkitemId(ptmBacklog.getSourceBacklogId())//0
                    .merge(ptmBacklog.getMerge())
                    .backlogPerformId(ptmBacklog.getPerformerId())
                    .backlogId(bl.getBacklogId())
                    .build();
            if (isAllQuery) {
                if (taskCardDataDTOMap.get(bl.getWorkItemId()) != null) {
                    activityDataBO.setBpmData(JsonUtils.objectToString(taskCardDataDTOMap.get(bl.getWorkItemId()).getData()));
                }
            } else {
                if (taskCardDataDTOMap.get(bl.getBacklogId()) != null) {
                    activityDataBO.setBpmData(JsonUtils.objectToString(taskCardDataDTOMap.get(bl.getBacklogId()).getData()));
                }
            }

            result.add(activityDataBO);
        });
        stopWatch.stop();
        log.info(" query data from ptmservice :{}", stopWatch.prettyPrint());
        return result;

    }


    /**
     * 获取 待办卡片详情的数据，如果查询不到可能是逾时任务卡
     *
     * @param backlogId 或 workitemId
     * @return
     */
    public BacklogBO selectBacklogForCard(Long backlogId) {
        BacklogBO result = null;

        if (!enablePTM()) {
            return result;
        }

        List<PtmBacklogItemBO> backlogList = ptmService.getBacklogByBacklogId(backlogId);
        PtmBacklog ptmOverdueBacklog = null;
        if (CollectionUtils.isEmpty(backlogList)) {
            //可能是逾时任务卡
            ptmOverdueBacklog = ptmBacklogMapper.selectOverdueById(backlogId);
            if (ptmOverdueBacklog != null) {
                backlogList = ptmService.getBacklogByBacklogId(ptmOverdueBacklog.getOverdueBacklogId());
            }

            if (CollectionUtils.isEmpty(backlogList)) {
                //可能是ptm.workitem.id，兼容，只留下该id
                Long workItemId = backlogId;
                PtmWorkItemRecordBO workItemRecordDTO = ptmService.getWorkItemRecord(workItemId);
                if (workItemRecordDTO != null) {
                    backlogList = ptmService.getBacklogByBacklogId(workItemRecordDTO.getBacklogId());
                    backlogList = backlogList.stream().filter(x -> Objects.equals(x.getWorkItemId(), workItemId)).collect(Collectors.toList());
                }
            }
        }
        if (CollectionUtils.isNotEmpty(backlogList)) {
            PtmBacklogItemBO ptmBacklog = backlogList.get(0);
            PtmTaskRecordBO ptmTaskRecord = ptmService.getTaskRecord(ptmBacklog.getTaskId());
            PtmProjectRecordBO ptmProjectRecord = ptmService.getProjectRecord(ptmTaskRecord.getProjectId());

            result = BacklogBO.builder()
                    .id(backlogId)
                    .tmTaskId(ptmProjectRecord.getProjectDefCode())
                    .tmActivityId(ptmTaskRecord.getTaskDefCode())
                    .tmPattern(ptmTaskRecord.getTaskDefPattern())
                    .tmCategory(ptmTaskRecord.getTaskDefCategory())
                    .summaryLayoutCacheText(null)
                    .overdueWorkitemId(0L)
                    .approvalState(null)
                    .tmActivityName(ptmTaskRecord.getTaskDefName())
                    .dataFrom(BpmConstant.DATA_FROM_PTM)
                    .build();

            if (ptmOverdueBacklog != null) {
                result.setId(ptmOverdueBacklog.getBacklogId());
                result.setOverdueWorkitemId(ptmOverdueBacklog.getOverdueBacklogId());
            } else {
                // 查询有没有记录签核状态
                PtmBacklog backlog = commonPtmTransformService.getBacklogByIdFromDb(backlogId);
                ;
                if (backlog != null) {
                    result.setApprovalState(backlog.getApprovalState());
                    if (StringUtils.isNotBlank(result.getApprovalState())) {
                        result.setApprovalState(GlobalConstant.backlogStatusMap.getOrDefault(result.getApprovalState(), ""));
                    }
                }
            }
        }

        return result;
    }

    /**
     * 根据id从PTM获取backlog信息
     *
     * @param backlogId 支持【任务卡id】 或者 【待办id】
     * @return 任务卡执行者（performer）和类型（type）
     */
    public PtmBacklog getPtmBacklogPerformerAndType(Long backlogId) {
        // 从PTM获取任务详情
        List<PtmBacklogItemBO> backlogList = ptmService.getBacklogByBacklogId(backlogId);
        if (CollectionUtils.isEmpty(backlogList)) {
            // 兼容是ptm.workItem.id的情况，并根据workItemId=backlogId过滤出backlogList
            Long workItemId = backlogId;
            PtmWorkItemRecordBO workItemRecordDTO = ptmService.getWorkItemRecord(workItemId);
            if (workItemRecordDTO != null) {
                backlogList = ptmService.getBacklogByBacklogId(workItemRecordDTO.getBacklogId());
                backlogList = backlogList.stream().filter(x -> Objects.equals(x.getWorkItemId(), workItemId)).collect(Collectors.toList());
            }
        }

        if (CollectionUtils.isEmpty(backlogList)) {
            return null;
        }

        //组装PTM相关信息，适配现有代码结构
        PtmBacklogItemBO ptmBacklog = backlogList.get(0);
        PtmBacklog result = PtmBacklog.builder()
                .backlogId(ptmBacklog.getBacklogId())
                .performerId(ptmBacklog.getPerformerId())
                .performerName(ptmBacklog.getPerformerName())
                .type(ptmBacklog.getType())
                .build();
        return result;
    }

    public PtmBacklog getBacklogForCard(Long backlogId) {
        return getBacklogForCard(backlogId, null);
    }

    /**
     * 封装查询selectByid
     *
     * @param backlogId
     * @return
     */
    public PtmBacklog getBacklogByIdFromDb(Long backlogId) {
        return ptmBacklogMapper.selectById(backlogId);
    }

    /**
     * 基于PTM获取任务卡详情
     *
     * @param backlogId
     * @return
     */
    public PtmBacklog getBacklogForCard(Long backlogId, Long workItemRecordBackLogId) {
        PtmBacklog result = null;

        if (!enablePTM()) {
            return result;
        }

        //从PTM获取任务详情
        List<PtmBacklogItemBO> backlogList = ptmService.getBacklogByBacklogId(backlogId);
        PtmWorkItemRecordBO workItemRecordDTO = null;
        if (CollectionUtils.isEmpty(backlogList)) {
            //可能是ptm.workitem.id，兼容，只留下该id
            Long workItemId = backlogId;
            //性能优化，如果前面已经有workItemRecordBackLogId，则不用再去调接口查，workItemRecordBackLogId为null才去查
            if (Objects.isNull(workItemRecordBackLogId)) {
                workItemRecordDTO = ptmService.getWorkItemRecord(workItemId);
                workItemRecordBackLogId = Objects.nonNull(workItemRecordDTO) ? workItemRecordDTO.getBacklogId() : null;
            }
            if (Objects.nonNull(workItemRecordBackLogId)) {
                backlogList = ptmService.getBacklogByBacklogId(workItemRecordBackLogId);
                backlogList = backlogList.stream().filter(x -> Objects.equals(x.getWorkItemId(), workItemId)).collect(Collectors.toList());
            }
        }
        //PTM无数据，查看本地库，防止脏数据,兼容之前的处理流程
        PtmBacklog backlog = commonPtmTransformService.getBacklogByIdFromDb(backlogId);
        if (CollectionUtils.isEmpty(backlogList)) {
            return backlog;
        }
        //非常规卡，以本地库数据为准
        if (backlog != null && backlog.getType().compareTo(0) != 0) {
            return backlog;
        }
        //组装PTM相关信息，适配现有代码结构
        PtmBacklogItemBO ptmBacklog = backlogList.get(0);

        PtmTaskRecordBO ptmTaskRecord = ptmService.getTaskRecord(ptmBacklog.getTaskId());
        PtmProjectRecordBO ptmProjectRecord = ptmService.getProjectRecord(ptmTaskRecord.getProjectId());
        result = PtmBacklog.builder()
                .backlogId(ptmBacklog.getBacklogId())
                .createTime(ptmBacklog.getCreateTime())
                .closedTime(ptmBacklog.getClosedTime())
                .closed(ptmBacklog.getClosed())
                .projectCardId(ptmProjectRecord.getProjectCardId())
                .projectId(ptmProjectRecord.getId())
                .taskId(ptmBacklog.getTaskId())
                .performerId(ptmBacklog.getPerformerId())
                .performerName(ptmBacklog.getPerformerName())
                .activityId(ptmBacklog.getActivityId())
                .workItemId(ptmBacklog.getWorkItemId())
                .planEndTime(ptmBacklog.getPlanEndTime())
                .taskDefCode(ptmBacklog.getTaskDefCode())
                .taskName(ptmBacklog.getTaskName())
                .taskWithName(ptmBacklog.getTaskWithName())
                .taskDefName(ptmBacklog.getTaskDefName())
                .taskDefWithName(ptmBacklog.getTaskDefWithName())
                .eocCode(ptmBacklog.getEocCode())
                .eocName(ptmBacklog.getEocName())
                .eocType(ptmBacklog.getEocType())
                .build();

        if (backlog != null) {
            result.setPerformerId(backlog.getPerformerId());
            result.setPerformerName(backlog.getPerformerName());
            result.setTenantId(backlog.getTenantId());
            result.setTenantName(backlog.getTenantName());
            result.setTargetTenantId(backlog.getTargetTenantId());
            result.setFavorite(backlog.getFavorite());
            result.setCreateDate(backlog.getCreateDate());
            result.setModifyDate(backlog.getModifyDate());
            result.setReadCount(backlog.getReadCount());
            result.setOverdueBacklogId(backlog.getOverdueBacklogId());
            result.setType(backlog.getType());
            result.setMerge(backlog.getMerge());
            result.setSourceBacklogId(backlog.getSourceBacklogId());
            result.setSubmitId(backlog.getSubmitId());
            result.setCreateDate(backlog.getCreateDate());
            result.setModifyDate(backlog.getModifyDate());
            result.setActionId(backlog.getActionId());
        } else {
            result.setCreateDate(LocalDateTime.now());
            result.setModifyDate(LocalDateTime.now());
        }

        //补全缺失字段
        this.buildDeletionField(result, workItemRecordDTO);

        return result;
    }

    /**
     * 补全缺失字段
     *
     * @param ptmBacklog
     * @param workItemRecordDTO
     */
    private void buildDeletionField(PtmBacklog ptmBacklog, PtmWorkItemRecordBO workItemRecordDTO) {
        if (StringUtils.isEmpty(ptmBacklog.getApprovalState())) {
            //性能优化：减少一次接口查询
            boolean hasAlreadyKnowItemRecord = Objects.nonNull(workItemRecordDTO) && Objects.nonNull(ptmBacklog) && ptmBacklog.getWorkItemId().equals(workItemRecordDTO.getId());
            PtmWorkItemRecordBO ptmWorkItemRecord = hasAlreadyKnowItemRecord ? workItemRecordDTO : ptmService.getWorkItemRecord(ptmBacklog.getWorkItemId());
            ptmBacklog.setApprovalState(commonPtmTransformService.getApprovalState(ptmWorkItemRecord.getCreateType()));
        }
    }

    /**
     * 基于PTM获取项目卡详情
     *
     * @param projectId
     * @return
     */
    public PtmProjectCard getProjectInfo(Long projectId) {
        PtmProjectCard result = null;

        if (!enablePTM()) {
            return result;
        }

        //从PTM获取任务详情
        PtmProjectCardRecordDTO ptmProjectCardRecordDTO = ptmService.getProjectCardRecord(projectId);
        PtmProjectCard ptmProjectCard = ptmProjectCardMapper.selectById(projectId);
        if (ptmProjectCardRecordDTO == null) {
            return ptmProjectCard;
        }
        if (ptmProjectCard == null) {
            ptmProjectCard = new PtmProjectCard();
        }
        //组装PTM相关信息，适配现有代码结构
        result = PtmProjectCard.builder()
                .id(ptmProjectCardRecordDTO.getId())
                .projectCode(ptmProjectCardRecordDTO.getProjectDefCode())
                .projectName(ptmProjectCardRecordDTO.getProjectName())
                .tenantId(ptmProjectCardRecordDTO.getTenantId())
                .tenantName(ptmProjectCardRecordDTO.getTargetName())
                .personInCharge(ptmProjectCardRecordDTO.getPersonInCharge())
                .personInChargeName(ptmProjectCardRecordDTO.getPersonInChargeName())
                .startTime(ptmProjectCardRecordDTO.getStartTime())
                .endTime(ptmProjectCardRecordDTO.getEndTime())
                .completedReport(ptmProjectCardRecordDTO.getCompletedReport())
                .state(ptmProjectCardRecordDTO.getState())
                .readCount(ptmProjectCard.getReadCount())
                .favorite(ptmProjectCard.getFavorite())
                .targetId(ptmProjectCard.getTargetId())
                .targetName(ptmProjectCard.getTargetName())
                .engineType(ptmProjectCard.getEngineType())
                .closedTime(ptmProjectCardRecordDTO.getClosedTime())
                .name(ptmProjectCardRecordDTO.getProjectName())
                .submitId(ptmProjectCard.getSubmitId())
                .hide(ptmProjectCard.getHide())
                .build();
        //sonar:Change this condition so that it does not always evaluate to "true"
        result.setCreateDate(ptmProjectCard.getCreateDate() != null ? ptmProjectCard.getCreateDate() : LocalDateTime.now());
        result.setModifyDate(ptmProjectCard.getModifyDate() != null ? ptmProjectCard.getModifyDate() : LocalDateTime.now());
        return result;
    }

    private Map getStateData(Map taskData, List<PtmSolveTask> ptmSolveTasks, PtmActivityRecordDTO ptmActivityRecord, PtmWorkItemRecordBO ptmWorkItemRecord) {
        if (CollectionUtils.isEmpty(ptmSolveTasks)) {
            return new HashMap();
        }
        //数据比对改成workItemId比对
        Long ptmWorkItemId = ptmWorkItemRecord.getId();
        List<String> unitKeys = ptmSolveTasks.get(0).getUniKeys();
        String dataKeys = ptmSolveTasks.get(0).getUniKeys().stream().collect(Collectors.joining(";"));
        //组装originalQueryVariableName对应的数据list
        Map<String, Object> stateData = new HashMap<>();
        stateData.put("dataKeys", dataKeys);
        stateData.put("originalQueryVariableName", "ptm_abnormal_data");
        List<Map<String, Object>> originalQueryVariables = new ArrayList<>();
        stateData.put("ptm_abnormal_data", originalQueryVariables);
        ptmSolveTasks.forEach(ptmSolveTask -> {
            if (ptmSolveTask.getData() != null) {
                ptmSolveTask.getData().forEach(dataItem -> {
                    if (ptmWorkItemId.equals(ptmSolveTask.getOriginalWorkItemId())) {
                        String serialNumber = ptmSolveTask.getSerialNumber();
                        Map<String, Object> instance = new HashMap<>();
                        instance.put("taskUid", ptmSolveTask.getTaskUid());
                        instance.put("processSerialNumber", serialNumber);
                        instance.put("tmActivityIds", new ArrayList<>());
                        instance.put("solveTaskId", ptmActivityRecord.getTaskId());
                        instance.put("solveBpmActivityId", ptmActivityRecord.getBpmActivityId());
                        instance.put("solveBpmActivityName", ptmActivityRecord.getBpmActivityName());
                        instance.put("bpmActivityId", ptmSolveTask.getTaskDefCode());
                        instance.put("bpmActivityName", ptmSolveTask.getTaskName());
                        instance.put("solveWorkItemId", ptmWorkItemRecord.getId());
                        instance.put("state", ptmSolveTask.getState());
                        Integer index = SummaryMessagesUtils.withinStateData(originalQueryVariables, dataItem, unitKeys);
                        List<Map<String, Object>> processInstances = new ArrayList<>();
                        if (index >= 0) {
                            processInstances = (List<Map<String, Object>>) originalQueryVariables.get(index).get("processInstances");
                        }
                        processInstances.add(instance);
                        dataItem.put("processInstances", processInstances);
                        dataItem.put("state", ptmSolveTask.getState());
                        originalQueryVariables.add(dataItem);
                    }
                });
            }
        });
        return stateData;
    }

    /**
     * 获取 指定项目指定任务的查询数据
     *
     * @param projectId    项目id
     * @param tmActivityId KM定义的任务id
     * @return
     */
    public List<ActivityDataBO> getViewActionsByProjectId(long projectId, String tmActivityId){
        if(!enablePTM()){
            return new ArrayList<>();
        }

        PtmProjectCardRecordDTO projectCardRecord = ptmService.getProjectCardRecord(projectId);
        List<PtmBacklogItemBO> backlogItemDTOList = null;
        if (projectCardRecord != null) {
            backlogItemDTOList = ptmService.getBacklogByProjectCardIdAndTaskCode(projectId, tmActivityId);
        } else {
            backlogItemDTOList = ptmService.getBacklogByProjectIdAndTaskCode(projectId, tmActivityId);
        }

        return getActivityData(tmActivityId, projectCardRecord, backlogItemDTOList);
    }

    /**
     * 获取 项目下指定任务卡的查询数据
     *
     * @param projectId    项目ID
     * @param tmActivityId KM定义的任务id
     * @param backlogId    任务卡id
     * @return
     */
    public List<ActivityDataBO> getViewActionsByProjectIdAndBacklogId(long projectId, String tmActivityId, long backlogId) {
        if (!enablePTM()) {
            return new ArrayList<>();
        }

        PtmProjectCardRecordDTO projectCardRecord = ptmService.getProjectCardRecord(projectId);
        List<PtmBacklogItemBO> backlogItemDTOList = ptmService.getBacklogByBacklogId(backlogId);

        return getActivityData(tmActivityId, projectCardRecord, backlogItemDTOList);
    }

    private List<ActivityDataBO> getActivityData(String tmActivityId, PtmProjectCardRecordDTO projectCardRecord, List<PtmBacklogItemBO> backlogItemDTOList) {
        List<ActivityDataBO> result = new ArrayList<>();
        if(CollectionUtils.isEmpty(backlogItemDTOList)){
            return result;
        }

        // 查询租户下所有用户信息
        List<EocTenantAllUserDTO.EocUserDTO> allUser = eocService.getAllUserByTenant();

        //region 组织数据
        List<PtmProjectRecordBO> projectRecordList = new ArrayList<>();

        //perf:通过空间换时间，相同用户信息，减少对eoc的调用
        Map<String, EocEmployeeDTO> tempEmployee = new HashMap<>();
        Map<String, ProxyUserResp> tempProxy = new HashMap<>();

        backlogItemDTOList.forEach(ptmBacklog -> {
            PtmTaskRecordBO ptmTaskRecord = ptmService.getTaskRecord(ptmBacklog.getTaskId());
            List<PtmTaskCardDataDTO> taskCardDataList = null;
            if (ptmBacklog.getActivityId() != null) {
                PtmTaskCardDataDTO taskCardDataDTO = ptmService.getTaskDataByActivityId(ptmBacklog.getActivityId(), "data");
                if (taskCardDataDTO != null) {
                    taskCardDataList = new ArrayList<>();
                    taskCardDataList.add(taskCardDataDTO);
                }
            } else if (ptmBacklog.getTaskId() != null) {
                PtmTaskCardDataDTO taskCardDataDTO = ptmService.getTaskDataByTaskId(ptmBacklog.getTaskId(), "data");
                if (taskCardDataDTO != null) {
                    taskCardDataList = new ArrayList<>();
                    taskCardDataList.add(taskCardDataDTO);
                }
            } else if (ptmBacklog.getBacklogId() != null) {
                taskCardDataList = ptmService.getTaskDataByBacklogId(ptmBacklog.getBacklogId(), "data");
            }
            PtmWorkItemRecordBO workItemRecordDTO = ptmService.getWorkItemRecord(ptmBacklog.getWorkItemId());
            Optional<PtmProjectRecordBO> optionalProject = projectRecordList.stream().filter(x-> Objects.equals(x.getId(), ptmBacklog.getProjectId())).findFirst();
            PtmProjectRecordBO projectRecord = null;
            if (optionalProject.isPresent()) {
                projectRecord = optionalProject.get();
            } else {
                projectRecord = ptmService.getProjectRecord(ptmBacklog.getProjectId());
                projectRecordList.add(projectRecord);
            }

            List<BpmActivityWorkitem> workitemList = new ArrayList<>();
            //region 组织workitem数组
            if (workItemRecordDTO != null) {
                EocEmployeeDTO eocEmployeeDTO = null;
                ProxyUserResp proxyUser = null;
                //perf:优化查看项目卡详情的时候，包含任务多，查看人员状态和代理人信息的调用耗时引起的性能问题
                //人员状态查询
                if (!tempEmployee.containsKey(workItemRecordDTO.getPerformerId())) {
                    eocEmployeeDTO = this.getEmployee(workItemRecordDTO.getPerformerId());
                    tempEmployee.put(workItemRecordDTO.getPerformerId(), eocEmployeeDTO);
                } else {
                    eocEmployeeDTO = tempEmployee.get(workItemRecordDTO.getPerformerId());
                }
                // 代理人查询
                final String tempProxyKey = workItemRecordDTO.getPerformerId() + AppAuthContextHolder.getContext().getAuthoredUser().getToken();
                if (!tempProxy.containsKey(tempProxyKey)) {
                    proxyUser = eocService.getProxyUser(workItemRecordDTO.getPerformerId(), AppAuthContextHolder.getContext().getAuthoredUser().getToken());
                    tempProxy.put(tempProxyKey, proxyUser);
                } else {
                    proxyUser = tempProxy.get(tempProxyKey);
                }

                BpmActivityWorkitem wi = BpmActivityWorkitem.builder()
                        .id(workItemRecordDTO.getId())
                        .performerId(workItemRecordDTO.getPerformerId())
                        .performerType(workItemRecordDTO.getPerformerType())
                        .performerName(workItemRecordDTO.getPerformerName())
                        .performerAgentId(null != proxyUser ? proxyUser.getUserId() : null)
                        .performerAgentName(getUserNameById(proxyUser.getUserId(), allUser))
                        .performerState(eocEmployeeDTO != null ? eocEmployeeDTO.getStatus() : null)
                        .workitemId(workItemRecordDTO.getWorkItemId())
                        .createType(workItemRecordDTO.getCreateType())
                        .state(workItemRecordDTO.getState())
                        .subState(workItemRecordDTO.getSubState())
                        .agentPerformerId(workItemRecordDTO.getAgentPerformerId())
                        .agentPerformerName(workItemRecordDTO.getAgentPerformerName())
                        .comment(workItemRecordDTO.getComment())
                        .groupId(0L)
                        .build();
                workitemList.add(wi);
            }

            ActivityDataBO data = new ActivityDataBO();
            data.setTaskId(projectRecord.getId());
            data.setTmTaskId(projectRecord.getProjectDefCode());
            data.setTmActivityId(tmActivityId);
            data.setTenantId(projectRecord.getTenantId());
            data.setBpmActivitySqlId(ptmBacklog.getTaskId());
            if (CollectionUtils.isNotEmpty(taskCardDataList) && taskCardDataList.get(0).getData() != null) {
                data.setBpmData(JsonUtils.objectToString(taskCardDataList.get(0).getData()));
            }
            data.setBpmStateData(null);     //不需要
            data.setBpmActivityId(null);    //不需要
            data.setBpmActivityType(ptmBacklog.getBpmnType());
            data.setPerformerIds(null);     //不需要
            data.setProcessSerialNumber(projectRecord.getProcessSerialNumber());
            data.setTmActivityName(null);
            data.setTmPattern(ptmBacklog.getTaskDefPattern());
            data.setTmCategory(ptmBacklog.getTaskDefCategory());
            data.setBacklogName(null);  //不需要
            data.setStartTime(ptmBacklog.getCreateTime());
            data.setPlanEndTime(ptmBacklog.getPlanEndTime());
            data.setEndTime(ptmBacklog.getClosedTime());
            data.setWorkitemList(JsonUtils.objectToString(workitemList));
            data.setChargeId(projectRecord.getPersonInCharge());
            data.setChargeName(projectRecord.getPersonInChargeName());
            data.setBusinessUnit(JsonUtils.objectToString(projectRecord.getBusinessUnit()));
            data.setEocName(projectRecord.getEocName());
            data.setTaskStartTime(projectRecord.getStartTime());
            data.setTaskEndTime(projectRecord.getEndTime());
            data.setProcessCreateTime(projectRecord.getCreateTime());
            data.setTaskName(ptmBacklog.getTaskName());
            data.setTmShowFlow(false);
            data.setStepId(ptmBacklog.getActivityId());
            data.setTaskSourceIds(projectRecord.getSourceIds());
            data.setStepSignReason(null);
            data.setApprovalState(null);
            data.setBacklogClosed(ptmBacklog.getClosed());
            data.setMerge(false);
            data.setProxyToken(projectRecord.getProxyToken());
            data.setCompositionId(null);    //不需要
            data.setOverdueWorkitemId(null);
            data.setTaskUid(ptmTaskRecord.getBpmTaskUid());
            data.setPreTaskUid(null);
            data.setPrePerformerId(null);
            data.setPrePerformerName(null);
            data.setPreAgentPerformerId(null);
            data.setPreAgentPerformerName(null);
            data.setPrePerformerType(null);
            data.setStepState(null);
            data.setStepSubState(null);
            data.setTraceId(projectRecord.getTraceId());
            data.setPtmBacklogId(ptmBacklog.getBacklogId());
            data.setPtmWorkItemId(ptmBacklog.getWorkItemId());
            data.setDataFrom(BpmConstant.DATA_FROM_PTM);

            if (CollectionUtils.isNotEmpty(workitemList)) {
                BpmActivityWorkitem workitem = workitemList.get(0);
                data.setBacklogId(workitem.getId());
                data.setPerformerId(workitem.getPerformerId());
                data.setPerformerType(workitem.getPerformerType());
                data.setPerformerName(workitem.getPerformerName());
                data.setPerformerAgentId(workitem.getPerformerAgentId());
                data.setPerformerAgentName(workitem.getPerformerAgentName());
                data.setPerformerState(workitem.getPerformerState());
                data.setWorkItemId(workitem.getId());
            }
            // 项目流程变量
            if (null != projectRecord.getData()) {
                data.setProjectBpmData(JsonUtils.objectToString(projectRecord.getData()));
            }
            // 项目卡id
            data.setProjectCardId(ptmBacklog.getProjectCardId());
            // 项目id
            if (projectCardRecord != null) {
                data.setProjectCardState(projectCardRecord.getState());
            }
            // 项目状态
            data.setProjectState(ptmBacklog.getProjectState());

            result.add(data);

        });

        //endregion

        return result;
    }

    private String getUserNameById(String userId, List<EocTenantAllUserDTO.EocUserDTO> allUser) {

        if (StringUtils.isBlank(userId)) {
            return null;
        }
        Optional<EocTenantAllUserDTO.EocUserDTO> optional = allUser.stream().filter(eocUserDTO -> userId.equals(eocUserDTO.getUserId())).findFirst();
        return optional.map(EocTenantAllUserDTO.EocUserDTO::getUserName).orElse(null);
    }

    public EocEmployeeDTO getEmployee(String userId) {
        EocEmployeeDTO eocEmployeeDTO = null;
        try {
            eocEmployeeDTO = eocService.getEmployeeByUserId(userId);
        } catch (Exception e) {
            log.warn("query eocUser(api/eoc/v2/emp/info) error:" + e.getMessage());
        }
        return eocEmployeeDTO;
    }

    /**
     * 获取PTM 项目卡指定任务的数据
     *
     * @param projectIds   项目Id
     * @param activityPass
     * @return
     */
    public List<ActivityDataBO> selectTaskEngineProjectCardActivityData(String projectIds, String activityPass) {
        if (!enablePTM()) {
            return null;
        }


        List<Long> projectIdList = new ArrayList<>();
        String[] ids = StringUtils.split(projectIds, ",");
        for (String id : ids) {
            projectIdList.add(Long.valueOf(id));
        }

        List<PtmProjectCardRouteTaskDTO> routeTaskList = ptmService.getProjectCardRouteTasks(projectIdList, activityPass);
        List<ActivityDataBO> result = new ArrayList<>();

        routeTaskList.forEach(rt -> {

                    PtmProjectRecordBO projectRecord = ptmService.getProjectRecord(rt.getProjectId());
                    ActivityDataBO data = new ActivityDataBO();
                    data.setTaskId(rt.getProjectId());
                    data.setTmTaskId(projectRecord.getProjectDefCode());
                    data.setTmActivityId(rt.getTaskDefCode());
                    data.setTenantId(rt.getTenantId());
                    data.setBpmActivitySqlId(rt.getId());
                    data.setBpmData(JsonUtils.objectToString(rt.getData()));
                    data.setBpmStateData(null);     //不需要
                    data.setBpmActivityId(null);    //不需要
                    data.setBpmActivityType(rt.getBpmnType());
                    data.setPerformerIds(null);     //不需要
                    data.setProcessSerialNumber(projectRecord.getProcessSerialNumber());
                    data.setTmActivityName(null);
                    data.setTmPattern(rt.getTaskDefPattern());
                    data.setTmCategory(rt.getTaskDefCategory());
                    data.setBacklogName(null);  //不需要
                    data.setStartTime(rt.getCreateTime());
                    data.setPlanEndTime(rt.getPlanEndTime());
                    data.setEndTime(rt.getClosedTime());
                    data.setWorkitemList(JsonUtils.objectToString(rt.getBacklogs()));
                    data.setChargeId(projectRecord.getPersonInCharge());
                    data.setChargeName(projectRecord.getPersonInChargeName());
                    data.setBusinessUnit(projectRecord.getBusinessUnit() == null ? null : JsonUtils.objectToString(projectRecord.getBusinessUnit()));
                    data.setEocName(projectRecord.getEocName());
                    data.setTaskStartTime(projectRecord.getStartTime());
                    data.setTaskEndTime(projectRecord.getEndTime());
                    data.setProcessCreateTime(projectRecord.getCreateTime());
                    data.setTaskName(rt.getTaskName());
                    data.setTmShowFlow(false);
                    data.setStepId(rt.getActivityId());
                    data.setTaskSourceIds(projectRecord.getSourceIds());
                    data.setStepSignReason(null);
                    data.setApprovalState(null);
                    data.setBacklogClosed(rt.getState() == 3);
                    data.setMerge(false);
                    data.setProxyToken(projectRecord.getProxyToken());
                    data.setCompositionId(null);    //不需要
                    data.setOverdueWorkitemId(null);
                    data.setPreTaskUid(rt.getPreTaskUid());
                    data.setPrePerformerId(null);
                    data.setPrePerformerName(null);
                    data.setPreAgentPerformerId(null);
                    data.setPreAgentPerformerName(null);
                    data.setPrePerformerType(null);
                    data.setStepState(null);
                    data.setStepSubState(null);
                    data.setTraceId(projectRecord.getTraceId());

                    if (CollectionUtils.isNotEmpty(rt.getBacklogs())) {
                        rt.getBacklogs().forEach(bl -> {
                            if (!Objects.equals(bl.getType(), 2)) {
                                //人员状态查询
                                EocEmployeeDTO eocEmployeeDTO = eocService.getEmployeeByUserId(bl.getPerformerId());
                                //暂时排除跨租户的，因为PTM有一个当前租户的任务
                                ActivityDataBO data1 = JsonUtils.jsonToObject(JsonUtils.objectToString(data), ActivityDataBO.class);
                                data1.setBacklogId(bl.getId());
                                data1.setPerformerId(bl.getPerformerId());
                                data1.setPerformerName(bl.getPerformerName());
                                data1.setPerformerState(eocEmployeeDTO.getStatus());
                                data1.setWorkItemId(bl.getWorkItemKey());
                                result.add(data1);
                            }
                        });
                    } else {
                        result.add(data);
                    }
                }
        );

        return result;
    }

    /**
     * 获取指定项目卡的 task 格式的项目卡详情
     *
     * @param projectCardId
     * @return
     */
    public Task getTask(Long projectCardId) {
        if (!enablePTM()) {
            return null;
        }
        PtmProjectCardDetailDTO projectCardDetailDTO = ptmService.getProjectCardDetail(projectCardId);
        if (projectCardDetailDTO != null) {
            Task task = Task.builder()
                    .id(projectCardDetailDTO.getId())
                    .mainTaskId(projectCardDetailDTO.getId())
                    .projectName(projectCardDetailDTO.getProjectDefName())
                    .name(projectCardDetailDTO.getProjectCardName())
                    .startTime(projectCardDetailDTO.getStartTime())
                    .endTime(projectCardDetailDTO.getEndTime())
                    .createTime(projectCardDetailDTO.getCreateTime())
                    .personInCharge(projectCardDetailDTO.getPersonInCharge())
                    .personInChargeName(projectCardDetailDTO.getPersonInChargeName())
                    .tmTaskId(projectCardDetailDTO.getProjectDefCode())
                    .tenantId(projectCardDetailDTO.getTenantId())
                    .build();
            return task;
        }
        return null;
    }

    /**
     * 获取指定任务卡的 task 格式的项目卡详情
     *
     * @param backlogId
     * @return
     */
    public Task getActivityTask(Long backlogId) {
        if (!enablePTM()) {
            return null;
        }
        PtmBacklog ptmBacklog = this.getBacklogForCard(backlogId);
        if (ptmBacklog != null) {
            PtmProjectCard ptmProjectCard = this.getProjectInfo(ptmBacklog.getProjectCardId());
            Task task = Task.builder()
                    .id(ptmBacklog.getBacklogId())
                    .name(ptmBacklog.getTaskDefName())
                    .endTime(ptmBacklog.getPlanEndTime())
                    .personInCharge(ptmBacklog.getPerformerId())
                    .personInChargeName(ptmBacklog.getPerformerName())
                    .projectName(ptmProjectCard.getProjectName())
                    .tmTaskId(ptmBacklog.getTaskDefCode())
                    .build();
            return task;
        }
        return null;
    }

    /**
     * 根据 项目卡id 获取项目列表，并转换为 bpmProcess
     *
     * @param projectCardId
     * @return 没有值返回空集合
     */
    public List<BpmProcess> getProjectByProjectCardId(Long projectCardId) {
        List<BpmProcess> result = new ArrayList<>();
        if (!enablePTM()) {
            return result;
        }

        PtmProjectCardDetailDTO projectCardDetailDTO = ptmService.getProjectCardDetail(projectCardId);
        if (projectCardDetailDTO != null && CollectionUtils.isNotEmpty(projectCardDetailDTO.getProjects())) {
            projectCardDetailDTO.getProjects().forEach(p -> {
                PtmProjectRecordBO ptmProjectRecord = ptmService.getProjectRecord(p.getProjectId());

                BpmProcess bpmProcess = BpmProcess.builder()
                        .processSerialNumber(ptmProjectRecord.getProcessSerialNumber())
                        .data(JSONObject.fromObject(ptmProjectRecord.getData()))
                        .businessUnit(JSONObject.fromObject(ptmProjectRecord.getBusinessUnit()))
                        .proxyToken(ptmProjectRecord.getProxyToken())
                        .tmTaskId(ptmProjectRecord.getProjectDefCode())
                        .build();

                result.add(bpmProcess);
            });
        }

        return result;
    }

    public BpmActivityWorkitem getReapprovalInfoById(long backlogId) {
        return getReapprovalInfoById(backlogId, null);
    }

    /**
     * 获取 退回重签、退回重办的 原workitem
     *
     * @param backlogId
     * @return
     */
    public BpmActivityWorkitem getReapprovalInfoById(long backlogId, PtmWorkItemRecordBO ptmWorkItemRecordBO) {
        BpmActivityWorkitem result = null;
        Long workItemRecordBackLogId = Objects.nonNull(ptmWorkItemRecordBO) ? ptmWorkItemRecordBO.getBacklogId() : null;
        PtmBacklog ptmBacklog = this.getBacklogForCard(backlogId, workItemRecordBackLogId);
        if (ptmBacklog == null) {
            return result;
        }

        Map<Long, TaskActivityItemVO> taskActivityItemVOMap = getReapprovalInfoMapById(backlogId, ptmBacklog, ptmWorkItemRecordBO);
        if(MapUtils.isEmpty(taskActivityItemVOMap)){
            return result;
        }

        TaskActivityItemVO  taskActivityItemVO = taskActivityItemVOMap.get(ptmBacklog.getWorkItemId());
        if (taskActivityItemVO != null) {
            result = BpmActivityWorkitem.builder()
                    .performerName(taskActivityItemVO.getPerformerName())
                    .closedTime(taskActivityItemVO.getClosedTime())
                    .comment(taskActivityItemVO.getComment())
                    .state(taskActivityItemVO.getState())
                    .subState(taskActivityItemVO.getSubState())
                    .groupId(0L)
                    .build();
        }
        return result;
    }

    /**
     * 获取 退回重签、退回重办的 原workitem
     *
     * @param backlogId
     * @return
     */
    public Map<Long, TaskActivityItemVO> getReapprovalInfoMapById(long backlogId, PtmBacklog ptmBacklog, PtmWorkItemRecordBO ptmWorkItemRecordBO) {
        Map<Long, TaskActivityItemVO> result = null;
        if (!enablePTM()) {
            return result;
        }

        if(ptmBacklog == null){
            Long workItemRecordBackLogId = Objects.nonNull(ptmWorkItemRecordBO) ? ptmWorkItemRecordBO.getBacklogId() : null;
            ptmBacklog = this.getBacklogForCard(backlogId, workItemRecordBackLogId);
        }

        if (ptmBacklog == null) {
            return result;
        }

        PtmTaskRecordBO ptmTaskRecordBO = ptmService.getTaskRecord(ptmBacklog.getTaskId());
        if (Objects.isNull(ptmTaskRecordBO)) {
            throw ErrorCodeEnum.PTM_QUERY_PTM_GET_TASK_RECORD.getBusinessExceptionWithArgs(ptmBacklog.getTaskId());
        }
        if (SummaryMessagesUtils.isApprovalTask(ptmTaskRecordBO.getTaskDefPattern(), ptmTaskRecordBO.getTaskDefCategory())) {
            //签核型
            result = ptmService.getReExecuteWorkItemByBacklogId(0, backlogId,  ptmTaskRecordBO.getTaskDefCode());
        } else {
            PtmProjectCard projectCard = commonPtmTransformService.getProjectInfo(ptmBacklog.getProjectCardId());
            if (Objects.isNull(projectCard)) {
                throw ErrorCodeEnum.PTM_QUERY_PTM_PROJECT_CARD_RETURN_EMPTY.getBusinessExceptionWithArgs(ptmBacklog.getProjectCardId());
            }
            String targetApproveActivity = themeMapService.getActivityTargetCode(projectCard.getProjectCode(), ptmTaskRecordBO.getTaskDefCode(), TmPageName.TASK_CARD_NAME.getValue());
            result = ptmService.getReExecuteWorkItemByBacklogId(1, backlogId, targetApproveActivity);
        }

        return result;
    }

    public String getExecuteComment(Long backlogId) {
        if (!enablePTM()) {
            return null;
        }

        BpmActivityWorkitem bpmActivityWorkitem = this.getReapprovalInfoById(backlogId);
        if (bpmActivityWorkitem != null && StringUtils.isNotBlank(bpmActivityWorkitem.getComment())) {
            return bpmActivityWorkitem.getComment();
        }
        return null;
    }

    /**
     * 获取逾时任务卡对应的原任务。将 ptm 的待办转换为 ATmc 的 待办<br>
     * 只提供 name、
     *
     * @param backlogId
     * @return
     */
    public Backlog translateBacklogById(Long backlogId) {
        if (!enablePTM()) {
            return null;
        }
        //从PTM获取任务信息
        List<PtmBacklogItemBO> backlogList = ptmService.getBacklogByBacklogId(backlogId);
        if (CollectionUtils.isEmpty(backlogList)) {
            //可能是ptm.workitem.id，兼容，只留下该id
            Long workItemId = backlogId;
            PtmWorkItemRecordBO workItemRecordDTO = ptmService.getWorkItemRecord(workItemId);
            if (workItemRecordDTO != null) {
                backlogList = ptmService.getBacklogByBacklogId(workItemRecordDTO.getBacklogId());
                backlogList = backlogList.stream().filter(x -> Objects.equals(x.getWorkItemId(), workItemId)).collect(Collectors.toList());
            }
        }
        Backlog backlog = null;
        // 获取本地ptm_backlog表中记录（可能为null）
        PtmBacklog ptmBacklog = commonPtmTransformService.getBacklogByIdFromDb(backlogId);
        /**
         * ptm存在，那肯定不是逾期、数据一致性类型任务卡；
         */
        if (CollectionUtils.isNotEmpty(backlogList)) {
            PtmBacklogItemBO ptmBacklogItemBO = backlogList.get(0);
            String performerId = Objects.nonNull(ptmBacklog) && StringUtil.isNotBlank(ptmBacklog.getPerformerId()) ? ptmBacklog.getPerformerId() : ptmBacklogItemBO.getPerformerId();
            /**
             * ptm存在、本地不存在（可能MQ消费异常导致），or 本地存在并判断出是辅助执行者（type=1）、外部供应商（type=2）任务卡，则需从ptm再查一遍任务卡信息：
             * 上述过程获取的backlogList中，performerId可能是不正确的（都是原始卡的performerId），
             * 因此需通过ptmService.getBacklogRecord接口重新获取任务卡的performerId，
             * 但是该接口获取到的任务卡名称、项目卡名称中未返回全部语系的，因此需要backlogList、backlogRecord两者集合起来构造方法返回数据
             */
            if (Objects.isNull(ptmBacklog) || PtmBacklogTypeEnum.isReplyAssistOrCrossTenantBacklogType(ptmBacklog.getType())) {
                PtmBacklogRecordDTO backlogRecord = ptmService.getBacklogRecord(backlogId);
                if (Objects.nonNull(backlogRecord)) {
                    performerId = backlogRecord.getPerformerId();
                    //同步backlog
                    this.syncBackLog(ptmBacklog, backlogRecord);
                }
            }

            backlog = Backlog.builder()
                    .id(ptmBacklogItemBO.getBacklogId())
                    .name(ptmBacklogItemBO.getTaskName())
                    .endTime(ptmBacklogItemBO.getPlanEndTime())
                    .startTime(ptmBacklogItemBO.getCreateTime())
                    .performerId(performerId)
                    .importance(Optional.ofNullable(ptmBacklog).map(PtmBacklog::getFavorite).orElse(0))
                    .content(ptmBacklogItemBO.getTaskDefName())
                    .contentWithName(ptmBacklogItemBO.getTaskDefWithName())
                    .withName(ptmBacklogItemBO.getTaskWithName())
                    .activityCode(ptmBacklogItemBO.getTaskDefCode())
                    .taskId(ptmBacklogItemBO.getTaskId())
                    .build();
            //任务引擎存在直接使用引擎数据
            log.info("使用任务引擎数据");
            return backlog;
        }
        //从平台库获取任务信息
        if (ptmBacklog != null) {
            backlog = Backlog.builder()
                    .id(ptmBacklog.getBacklogId())
                    .name(ptmBacklog.getTaskName())
                    .endTime(ptmBacklog.getPlanEndTime())
                    .startTime(ptmBacklog.getCreateTime())
                    .performerId(ptmBacklog.getPerformerId())
                    .importance(ptmBacklog.getFavorite())
                    .content(ptmBacklog.getTaskDefName())
                    .contentWithName(ptmBacklog.getTaskDefWithName())
                    .activityCode(ptmBacklog.getTaskDefCode())
                    .taskId(ptmBacklog.getTaskId())
                    .build();
            log.info("使用平台库任务信息");
            return backlog;
        }

        return null;
    }

    /**
     * 根据ptm信息构建ptmBacklog
     *
     * @param ptmBacklogRecord
     * @return
     */
    public PtmBacklog buildPtmBacklog(PtmBacklogRecordDTO ptmBacklogRecord) {
        PtmWorkItemRecordBO ptmWorkItemRecord = ptmService.getWorkItemRecord(ptmBacklogRecord.getWorkItemId());

        PtmBacklog ptmBacklog = new PtmBacklog();
        ptmBacklog.setBacklogId(ptmBacklogRecord.getId());
        ptmBacklog.setTargetTenantId(ptmBacklogRecord.getTenantId());
        ptmBacklog.setReadCount(0);
        ptmBacklog.setClosed(ptmBacklogRecord.getClosed());
        ptmBacklog.setApprovalState(getApprovalState(ptmWorkItemRecord.getCreateType()));
        ptmBacklog.setMerge(ptmBacklogRecord.getMerge());
        return ptmBacklog;
    }

    /**
     * 根据ptm信息构建ptmBacklog 优化后
     *
     * @param ptmWorkItemDetailResp
     * @return
     */
    public PtmBacklog buildPtmBacklogV2(PtmWorkItemDetailResp ptmWorkItemDetailResp) {
        PtmBacklog ptmBacklog = new PtmBacklog();
        ptmBacklog.setBacklogId(ptmWorkItemDetailResp.getBacklogId());
        ptmBacklog.setTargetTenantId(ptmWorkItemDetailResp.getBacklogTenantId());
        ptmBacklog.setReadCount(0);
        ptmBacklog.setClosed(ptmWorkItemDetailResp.getClosed());
        ptmBacklog.setApprovalState(getApprovalState(ptmWorkItemDetailResp.getCreateType()));
        ptmBacklog.setMerge(ptmWorkItemDetailResp.getMerge());
        return ptmBacklog;
    }


    /**
     * 转换类型
     *
     * @param createType
     * @return
     */
    public String getApprovalState(Integer createType) {
        if (Objects.equals(createType, WorkitemCreateType.REASSIGN.getValue())) {
            return BpmConstant.APPROVAL_STATE_REASSIGN;
        } else if (Objects.equals(createType, WorkitemCreateType.ADDTASK.getValue())) {
            return BpmConstant.APPROVAL_STATE_ADD_tASK;
        } else if (Objects.equals(createType, WorkitemCreateType.REEXECUTE.getValue())) {
            return BpmConstant.APPROVAL_STATE_REEXECUTE;
        } else if (Objects.equals(createType, WorkitemCreateType.REAPPROVAL.getValue())) {
            return BpmConstant.APPROVAL_STATE_REAPPROVAL;
        } else if (Objects.equals(createType, WorkitemCreateType.HANDOVER.getValue())) {
            return BpmConstant.APPROVAL_STATE_HANDOVER;
        } else if (Objects.equals(createType, WorkitemCreateType.RECALL.getValue())) {
            return BpmConstant.APPROVAL_STATE_RECALL;
        }
//        else if (Objects.equals(createType, WorkitemCreateType.ROLLBACK_APPROVE.getValue())) {
//            return BpmConstant.APPROVAL_STATE_ROLLBACK_APPROVE;
//        }else if (Objects.equals(createType, WorkitemCreateType.ROLLBACK.getValue())) {
//            return BpmConstant.APPROVAL_STATE_ROLLBACK;
//        }

        return null;
    }


    private void syncBackLog(PtmBacklog ptmBacklog, PtmBacklogRecordDTO ptmBacklogRecordDTO) {
        //主要是atmc本地和ptm的不一致，就应该同步变更本地
        if (ptmBacklog != null && !StringUtils.equals(ptmBacklog.getPerformerId(), ptmBacklogRecordDTO.getPerformerId())) {
            PtmBacklog syncBacklog = new PtmBacklog();
            syncBacklog.setBacklogId(ptmBacklog.getBacklogId());
            syncBacklog.setPerformerId(ptmBacklogRecordDTO.getPerformerId());
            syncBacklog.setPerformerName(ptmBacklogRecordDTO.getPerformerName());
            SyncBackLogEventDTO syncEvent = new SyncBackLogEventDTO(syncBacklog, null);
            localEventBus.post(syncEvent);
        }
    }
}
