package com.digiwin.athena.atmc.common.service.ptm;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.Base64Converter;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atmc.application.configuration.EnvProperties;
import com.digiwin.athena.atmc.common.constant.TaskConstant;
import com.digiwin.athena.atmc.infrastructure.mapper.biz.migration.PtmBacklogMapper;
import com.digiwin.athena.atmc.common.dao.ptm.PtmProjectCardMapper;
import com.digiwin.athena.atmc.common.domain.approval.ApprovalActivityDTO;
import com.digiwin.athena.atmc.common.domain.approval.ApprovalStepDTO;
import com.digiwin.athena.atmc.http.restful.ptm.model.ApprovalWorkItemByDataDTO;
import com.digiwin.athena.atmc.common.domain.approval.ApprovalWorkItemDTO;
import com.digiwin.athena.atmc.common.domain.backlog.CloseBacklogActionDTO;
import com.digiwin.athena.atmc.infrastructure.pojo.bo.migration.*;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmBacklog;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmProjectCard;
import com.digiwin.athena.atmc.common.enums.ActivityWorkitemSubState;
import com.digiwin.athena.atmc.common.enums.SignCategory;
import com.digiwin.athena.atmc.common.service.bpm.BpmActivityService;
import com.digiwin.athena.atmc.http.constant.BpmConstant;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.digiwin.athena.atmc.http.domain.Task;
//import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApproveActivityDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApproveRequestDTO;
import com.digiwin.athena.atmc.http.restful.bpm.model.BpmTaskApprovelDTO;
import com.digiwin.athena.atmc.http.restful.iam.UserService;
import com.digiwin.athena.atmc.http.restful.iam.model.UserDTO;
import com.digiwin.athena.atmc.http.restful.im.ImService;
import com.digiwin.athena.atmc.http.restful.im.model.ImNeteaseUserDTO;
import com.digiwin.athena.atmc.http.restful.ptm.PtmService;
import com.digiwin.athena.atmc.http.restful.ptm.model.*;
import com.digiwin.athena.atmc.http.restful.thememap.ThemeMapService;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmActivityResponseDTO;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmAttachmentConfigInfo;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmDmcAccount;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class PtmBacklogTransformService {

    @Autowired
    private EnvProperties envProperties;

    @Autowired
    private PtmService ptmService;

    @Autowired
    private PtmBacklogMapper ptmBacklogMapper;

    @Autowired
    private PtmProjectCardMapper ptmProjectCardMapper;

    @Autowired
    private UserService userService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private BpmActivityService bpmActivityService;

    @Autowired
    private ImService imService;

    @Autowired
    private ThemeMapService themeMapService;

    @Autowired
    private CommonPtmTransformService commonPtmTransformService;

    /**
     * 批量 更新任务的名称、计划开始时间、结束时间等
     * @param paramMap
     * @return
     */

    public int batchUpdateNameAndTimeById(Map paramMap){
        int result = 0;
        if(!enablePTM()){
            return result;
        }

        List<Map> list = (List<Map>)paramMap.get("data");
        if(CollectionUtils.isEmpty(list)){
            return result;
        }

        //转换成PTM格式参数
        List<Map> paraList = new ArrayList<>();
        list.forEach(m->{
            Map map = new HashMap();

            map.put("taskId", MapUtils.getLongValue(m, "backlogId"));
            if(m.containsKey("name")){
                map.put("taskName", m.get("name"));
            }
            if(m.containsKey("startTime")){
                map.put("startTime", m.get("startTime"));
            }
            if(m.containsKey("endTime")){
                map.put("endTime", m.get("endTime"));
            }
            paraList.add(map);
        });

        Map map = new HashMap();
        map.put("data", paraList);

        if(ptmService.batchUpdateBacklogInfo(map)){
            paraList.forEach(u->{
                updateNameAndTimeByTaskId(u);
            });
            result = 1;
        }

        return result;
    }

    /**
     * 根据 taskId 更新关联 ptm_backlog 的名称和时间
     * @param map 入参格式：{"taskId":1, "taskName": "aaaaa","startTime": "2022-02-12 15:14:12", "endTime": "2022-04-19 15:29:00"}
     */
    private void updateNameAndTimeByTaskId(Map map){
        Long taskId = MapUtils.getLong(map, "taskId");
        List<PtmBacklogPartialDTO> backlogList = ptmService.queryBacklogPartialByTaskIds(Arrays.asList(taskId));
        if(CollectionUtils.isNotEmpty(backlogList)){
            backlogList.forEach(bl->{
                boolean hasChanged = false;

                UpdateWrapper<PtmBacklog> updateWrapper = new UpdateWrapper<>();
                updateWrapper.eq("backlog_id", bl.getId());
                if(map.containsKey("taskName") && map.get("taskName")!=null){
                    updateWrapper.set("task_name", map.get("taskName"));
                    hasChanged=true;
                }
                if(map.containsKey("endTime") && map.get("endTime")!=null){
                    updateWrapper.set("plan_end_time", map.get("endTime"));
                    hasChanged=true;
                }
                if(hasChanged){
                    ptmBacklogMapper.update(null, updateWrapper);
                }
            });
        }
    }

    /**
     * 终止任务
     * @param paraMap
     * @return
     */

    public int terminateTask(Map paraMap){
        int result = 0;
        if(!enablePTM()){
            return result;
        }

        if(ptmService.batchTerminateTask(paraMap)){
            result = 1;
        }

        return result;
    }

    /**
     * 根据backlog获取所属项目卡的部分信息<br>
     * id、perser_in_charge、tm_task_id
     */

    public Task translateProjectCardByBacklogId(Long backlogId){
        Task  task = null;
        if(!enablePTM()){
            return task;
        }

        PtmBacklog ptmBacklog = commonPtmTransformService.getBacklogForCard(backlogId);
        //如果ptm也没有backlog信息返回，则直接返回task
        if(ptmBacklog == null)
        {
            return task;
        }

        PtmProjectCard ptmProjectCard = commonPtmTransformService.getProjectInfo(ptmBacklog.getProjectCardId());
        if(ptmProjectCard ==null){
            PtmProjectCardDetailDTO ptmProjectCardDetailDTO = ptmService.getProjectCardDetail(ptmBacklog.getProjectCardId());
            ptmProjectCard = PtmProjectCard.builder()
                    .id(ptmProjectCardDetailDTO.getId())
                    .projectCode(ptmProjectCardDetailDTO.getProjectDefCode())
                    .personInCharge(ptmProjectCardDetailDTO.getPersonInCharge())
                    .build();
        }
        task = Task.builder()
                .id(ptmProjectCard.getId())
                .tmTaskId(ptmProjectCard.getProjectCode())
                .personInCharge(ptmProjectCard.getPersonInCharge())
                .build();
        return task;
    }

    /**
     * 关闭指定的待办，并设置actionId,并发送工作项关闭事件
     * <p>更新字段包括：action_id、state、sub_state、closed_time、row_moidfy_time</p>
     * @param action
     * @return 更新的数据条数
     */

    public int closeBacklog(CloseBacklogActionDTO action){
        log.info("closeBacklog:{}", JsonUtils.objectToString(action));
        int result = 0;
        if(!enablePTM()){
            return result;
        }
        List<PtmBacklog> list = new ArrayList<>();

        PtmBacklog ptmBacklog = ptmBacklogMapper.selectById(action.getId());
        if(ptmBacklog!=null){
            list.add(ptmBacklog);
        }else{
            list.addAll(ptmBacklogMapper.selectByWorkItemId(action.getId()));
        }

        if(CollectionUtils.isNotEmpty(list)){

            AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();

            for (PtmBacklog backlog : list) {
                if(Objects.equals(backlog.getMerge(),true)){
                    //合并的，只更新actionId，真正关闭等PTM推送消息
                    UpdateWrapper<PtmBacklog> updateWrapper = new UpdateWrapper<>();
                    updateWrapper.eq("backlog_id", backlog.getBacklogId());
                    updateWrapper.set("action_id", action.getActionId())
                            .set("modify_date", LocalDateTime.now());
                    result += ptmBacklogMapper.update(null, updateWrapper);

                    log.info("close ptm backlog-{}, actionId-{}", backlog.getBacklogId(), action.getActionId());
                }else{
                    //非合并，正常更新
                    UpdateWrapper<PtmBacklog> updateWrapper = new UpdateWrapper<>();
                    updateWrapper.eq("backlog_id", backlog.getBacklogId());
                    updateWrapper.set("action_id", action.getActionId())
                            .set("closed", 1)
                            .set("closed_time", LocalDateTime.now())
                            .set("modify_date", LocalDateTime.now());
                    result += ptmBacklogMapper.update(null, updateWrapper);
                }
            }
        }

        return result;
    }

    /**
     * 根据异常排除任务的 backlogId，查找未完成的任务数据<br>
     * 方便撤回
     * @param backlogId
     * @return 没有数据，返回空数组
     */

    public List<Map> selectNoFinishedListByOriginBacklogIdForSolve(Long backlogId){
        List<Map> result = new ArrayList<>();
        if(!enablePTM()){
            return result;
        }

        List<Long> originalBacklogIds = new ArrayList<>();
        originalBacklogIds.add(backlogId);

        PtmBacklog ptmBacklog = commonPtmTransformService.getBacklogForCard(backlogId);
        if (ptmBacklog == null) {
            return result;
        }
        List<PtmBacklogItemBO> backlogItemList = ptmService.getBacklogByBacklogId(ptmBacklog.getBacklogId());
        backlogItemList.forEach(x-> originalBacklogIds.add(x.getWorkItemId()));
        List<Map> list = ptmService.getSolveTaskList(originalBacklogIds);
        log.info("getSolveTask list:{}",JsonUtils.objectToString(list));

        if(CollectionUtils.isNotEmpty(list)){
            for (Map x : list) {
                if(x.containsKey("state") && MapUtils.getInteger(x,"state",0)!=3) {
                    x.put("tm_task_id", x.get("projectDefCode"));
                    x.put("tm_activity_id", x.get("taskDefCode"));
                    x.put(BpmConstant.DATA_FROM_NAME, BpmConstant.DATA_FROM_PTM);
                    result.add(x);
                }
            }
        }
        return result;
    }

    /**
     * 获取异常排除 方案历程
     * @param approvalStateMap
     * @param user
     */

    public List<ApprovalActivityDTO> programProgress(Map<String,Object> approvalStateMap, AuthoredUser user){
        List<ApprovalActivityDTO> result = new ArrayList<>();
        if(!enablePTM()){
            return result;
        }

        if (approvalStateMap.containsKey("processInstances")) {
            List<Map<String, Object>> processInstances = (List<Map<String, Object>>) approvalStateMap.get("processInstances");
            if (CollectionUtils.isNotEmpty(processInstances)) {
                Long solveTaskId = MapUtils.getLong(processInstances.get(0), "solveTaskId", 0L);
                String solveBpmActivityId = MapUtils.getString(processInstances.get(0), "solveBpmActivityId", "");
                String solveBpmActivityName = MapUtils.getString(processInstances.get(0), "solveBpmActivityName", "");
                Long solveWorkItemId = MapUtils.getLong(processInstances.get(0), "solveWorkItemId", 0L);
                //手动拼原始的异常排除任务
                ApprovalActivityDTO solveActivity = ApprovalActivityDTO.builder()
                        .id(solveTaskId)
                        .bpmActivityId(solveBpmActivityId)
                        .bpmActivityName(solveBpmActivityName)
                        .build();
                PtmWorkItemRecordBO ptmWorkItemRecordBO = ptmService.getWorkItemRecord(solveWorkItemId);
                if (ptmWorkItemRecordBO != null) {
                    ApprovalWorkItemDTO workItem = ApprovalWorkItemDTO.builder()
                            .id(0L)
                            .createTime(ptmWorkItemRecordBO.getCreateTime())
                            .closedTime(ptmWorkItemRecordBO.getClosedTime())
                            .performerId(ptmWorkItemRecordBO.getPerformerId())
                            .performerName(ptmWorkItemRecordBO.getPerformerName())
                            .workitemId(ptmWorkItemRecordBO.getWorkItemId())
                            .comment(ptmWorkItemRecordBO.getComment())
                            .state(ptmWorkItemRecordBO.getState())
                            .subState(ptmWorkItemRecordBO.getSubState())
                            .agentPerformerId(ptmWorkItemRecordBO.getAgentPerformerId())
                            .agentPerformerName(ptmWorkItemRecordBO.getAgentPerformerId())
                            .createType(ptmWorkItemRecordBO.getCreateType())
                            .performerType(ptmWorkItemRecordBO.getPerformerType())
                            .reassign(new JSONArray())
                            .build();
                    List<ApprovalWorkItemDTO> workitemList = new ArrayList<>();
                    workitemList.add(workItem);
                    solveActivity.setWorkitemList(workitemList);
                }
                result.add(solveActivity);
                processInstances.forEach(processInstance -> {
                    ApprovalActivityDTO activity = ApprovalActivityDTO.builder()
                            .bpmActivityId(MapUtils.getString(processInstance, "bpmActivityId", ""))
                            .bpmActivityName(MapUtils.getString(processInstance, "bpmActivityName", ""))
                            .state(MapUtils.getInteger(processInstance, "state", 0))
                            .build();
                    activity.setSteps(setApprovalStep(MapUtils.getString(processInstance, "taskUid", "")));
                    result.add(activity);
                });
            }
        }
        return result;
    }

    private List<ApprovalStepDTO> setApprovalStep( String taskUid){
        List<ApprovalStepDTO> result = new ArrayList<>();
        BpmTaskApproveRequestDTO requestDTO = BpmTaskApproveRequestDTO.builder()
                .locale(LocaleContextHolder.getLocale().toLanguageTag())
                .taskUid(taskUid)
                .build();
        BpmTaskApprovelDTO bpmTaskApprovelDTO = ptmService.queryApproveByTaskUid(requestDTO);

        if(null != bpmTaskApprovelDTO && CollectionUtils.isNotEmpty(bpmTaskApprovelDTO.getSignProcess())){
            Map<String,String> nameMap = new HashMap<>();
            List<BpmTaskApproveActivityDTO> approvalList = bpmTaskApprovelDTO.getSignProcess();
            List<PtmActivitySignInformerDTO> noticeList = ptmService.getSignInformer(approvalList.stream()
                    .filter(x->StringUtils.isNotBlank(x.getBpmActivityOID()))
                    .map(x->x.getBpmActivityOID()).collect(Collectors.toList()));

            approvalList.forEach(approval->{

                //签核前知会通知
                ApprovalStepDTO noticeStep = builderSignInformer(noticeList, approval, "noticeBefore");
                if(noticeStep!=null){
                    noticeStep.setSignType(81);
                    result.add(noticeStep);
                }

                ApprovalStepDTO step = ApprovalStepDTO.builder()
                        .id(0L)
                        .bpmActivityId(approval.getActivityId())
                        .bpmActivityName(approval.getActivityName())
                        .state(approval.getState())
                        .isReassign(false)
                        .isReexecute(false)
                        .signType(SignCategory.parse(approval.getSignType()).getValue())
                        .build();
                /**
                 * 附件配置
                 */
                step.setAttachmentConfig(generateAttachmentConfig(bpmTaskApprovelDTO.getProjectDefCode(), bpmTaskApprovelDTO.getTaskDefCode()));

                if(!Objects.equals(-1, approval.getState()) && CollectionUtils.isNotEmpty(approval.getWorkitemList())) {
                    List<ApprovalWorkItemDTO> workItemList = new ArrayList<>();
                    approval.getWorkitemList().forEach(wi->{

                        ApprovalWorkItemDTO workItem = ApprovalWorkItemDTO.builder()
                                .id(0L)
                                .createTime(wi.getCreateTime())
                                .closedTime(wi.getClosedTime())
                                .performerId(wi.getPerformerId())
                                .performerName(wi.getPerformerId())
                                .workitemId(wi.getWorkitemId())
                                .comment(wi.getComment())
                                .state(wi.getState())
                                .subState(wi.getSubState())
                                .agentPerformerId(wi.getAgentPerformerId())
                                .agentPerformerName(wi.getAgentPerformerId())
                                .createType(wi.getCreateType())
                                .performerType(wi.getPerformerType())
                                .reassign(new JSONArray())
                                .attachments(wi.getAttachments())
                                .build();
                        if(!Objects.equals(wi.getCreateType(),0) && wi.getSubState()>10){
                            //PTM优先使用createType,否则createType和subState会显示两个状态
                            workItem.setSubState(ActivityWorkitemSubState.COMPLETED.getValue());
                        }
                        if(nameMap.containsKey(wi.getPerformerId())){
                            workItem.setPerformerName(nameMap.get(wi.getPerformerId()));
                        }else{
                            UserDTO userDTO = userService.query(wi.getPerformerId());
                            if(userDTO!=null){
                                workItem.setPerformerName(userDTO.getName());
                            }
                            nameMap.put(workItem.getPerformerId(), workItem.getPerformerName());
                        }
                        if(StringUtils.isNotBlank(wi.getAgentPerformerId())) {
                            if (nameMap.containsKey(wi.getAgentPerformerId())) {
                                workItem.setAgentPerformerName(nameMap.get(wi.getAgentPerformerId()));
                            } else {
                                UserDTO userDTO = userService.query(wi.getAgentPerformerId());
                                if (userDTO != null) {
                                    workItem.setAgentPerformerName(userDTO.getName());
                                }
                                nameMap.put(workItem.getAgentPerformerId(), workItem.getAgentPerformerName());
                            }
                        }
                        workItemList.add(workItem);

                        //退回
//                        step.setIsReexecute(wi.getCreateType()==2 || wi.getCreateType()==3);

                    });
                    step.setWorkitemList(workItemList);
                }
                result.add(step);

                //签核后知会通知
                noticeStep = builderSignInformer(noticeList, approval, "noticeAfter");
                if(noticeStep!=null){
                    noticeStep.setSignType(82);
                    result.add(noticeStep);
                }

            });
        }

        return result;
    }

    /**
     * 获取附件配置
     * @param projectDefCode
     * @param taskDefCode
     * @return
     */
    private Map<String, Object> generateAttachmentConfig(String projectDefCode, String taskDefCode) {
        if (StringUtils.isNotBlank(projectDefCode)&& StringUtils.isNotBlank(taskDefCode)){
            TmActivityResponseDTO activityAction = themeMapService.getActivityAction(projectDefCode, taskDefCode, "task-detail");
            TmAttachmentConfigInfo attachment = activityAction.getAttachment();
            if (null != attachment && null != attachment.getApprovalConfig()){
                Map<String, Object> attachmentConfig = new HashMap<>();
                TmDmcAccount dmcAccount = attachment.getApprovalConfig().getDmcAccount();
                if (null != dmcAccount && StringUtils.isNotBlank(dmcAccount.getPassword())){
                    dmcAccount.setPassword(Base64Converter.encode(dmcAccount.getPassword()));
                    attachmentConfig.put("dmcAccount", dmcAccount);
                }
                attachmentConfig.put("buckets", attachment.getApprovalConfig().getBuckets());
                attachmentConfig.put("readCategory", Collections.singletonList(attachment.getApprovalConfig().getUploadCategory()));
                return attachmentConfig;
            }
        }
        return null;
    }

    /**
     * 获取签核历程
     * @param id  可能为代办backlogId 也可能为ptmWorkItem.id
     * @return
     */

    public List<ApprovalStepDTO> getTaskApprovalList(Long id){
        List<ApprovalStepDTO> result = new ArrayList<>();
        if(id==null){
            return result;
        }
        if(!enablePTM()){
            return result;
        }

//        PtmBacklogRecordDTO ptmBacklogRecord = ptmService.getBacklogRecord(backlogId);
//        if(ptmBacklogRecord==null){
//            //有可能是ptmWorkItem.id
//            PtmWorkItemRecordDTO ptmWorkItemRecord = ptmService.getWorkItemRecord(backlogId);
//            if(ptmWorkItemRecord!=null){
//                //如果ptm返回的workItem没有backlogId则直接使用workItem.id去查询签核历程
//                if (null != ptmWorkItemRecord.getBacklogId()) {
//                    backlogId = ptmWorkItemRecord.getBacklogId();
//                }
//            }else{
//                return result;
//            }
//        }
//
//        List<PtmBacklogApprovalDTO> approvalList = ptmService.queryApprovalNew(backlogId);

        List<PtmBacklogApprovalDTO> approvalList;
        // 根据id先查询代办信息
        PtmBacklogRecordDTO ptmBacklogRecord = ptmService.getBacklogRecord(id);
        if (null != ptmBacklogRecord ){
            // 传入的为代办的id,根据代办的id查询签核历程
            approvalList = ptmService.queryApprovalNew(id);
        }
        else {
            // 根据workItemId查询WorkItemRecord信息
            PtmWorkItemRecordBO ptmWorkItemRecord = ptmService.getWorkItemRecord(id);
            if (null == ptmWorkItemRecord){
                return result;
            }
            // 代办id如果不为空，根据代办id查询
            if (null != ptmWorkItemRecord.getBacklogId()){
                approvalList = ptmService.queryApprovalNew(ptmWorkItemRecord.getBacklogId());
            }
            else {
                // 如果代办id为空，根据workItemId查询签核历程
                approvalList = ptmService.queryApprovalByWorkItemId(id);
            }

        }
        if(CollectionUtils.isEmpty(approvalList)){
            return result;
        }

        dealApprovalList(approvalList, result);
        return result;
    }

    /**
     * 数据对应待办的签核历程
     * @param workItemDataDTO
     * @return
     */
    public List<ApprovalStepDTO> getTaskApprovalList(ApprovalWorkItemByDataDTO workItemDataDTO) {
        List<ApprovalStepDTO> result = new ArrayList<>();
        // 调用ptm接口获取签核历程
        List<PtmBacklogApprovalDTO> approvalList = ptmService.queryApprovalByData(workItemDataDTO);
        // 整理签核历程数据
        dealApprovalList(approvalList, result);
        return result;
    }

    /**
     * 处理签核历程结果
     * @param approvalList
     * @param result
     */
    private void dealApprovalList(List<PtmBacklogApprovalDTO> approvalList, List<ApprovalStepDTO> result) {
        approvalList.forEach(approval->{
            if(CollectionUtils.isNotEmpty(approval.getSignProcess())){
                Map<String,String> nameMap = new HashMap<>();
                for (PtmBacklogApprovalDTO.SignProcess signProcess : approval.getSignProcess()) {
                    //region 发起人
                    if (signProcess.getInitiatorNode()) {
                        String userName = signProcess.getInitiatorId();
                        UserDTO userDTO = userService.query(userName);
                        if (userDTO != null) {
                            userName = userDTO.getName();
                        }

                        ApprovalStepDTO step = ApprovalStepDTO.builder()
                                .id(0L)
                                .bpmActivityId("__uibot__start__")
                                .bpmActivityName(messageUtils.getMessage("backlog.first.step.name"))
                                .state(3)
                                .isReassign(false)
                                .isReexecute(false)
                                .build();

                        ApprovalWorkItemDTO workItem = ApprovalWorkItemDTO.builder()
                                .id(0L)
                                .createTime(signProcess.getStartTime())
                                .closedTime(signProcess.getCompletedTime())
                                .performerId(signProcess.getInitiatorId())
                                //签核历程展示 发起人姓名字段新增默认值“系统”,多语系
                                .performerName(StringUtils.isEmpty(userName)?messageUtils.getMessage("system.default.initiator.name"):userName)
                                .state(3)
                                .subState(10)
                                .reassign(new JSONArray())
                                .build();
                        List<ApprovalWorkItemDTO> workItemList = new ArrayList<>();
                        workItemList.add(workItem);
                        step.setWorkitemList(workItemList);
                        result.add(step);
                        continue;
                    }
                    //endregion

                    JSONObject informer = null;
                    if (StringUtils.isNotBlank(signProcess.getNoticeInformer())) {
                        informer = JSONObject.fromObject(signProcess.getNoticeInformer());
                    }

                    //签核前知会
                    ApprovalStepDTO noticeStep = builderSignInformer(signProcess, informer, "noticeBefore");
                    if (noticeStep != null) {
                        noticeStep.setSignType(81);
                        result.add(noticeStep);
                    }

                    //region 签核信息
                    ApprovalStepDTO step = ApprovalStepDTO.builder()
                            .id(0L)
                            .bpmActivityId(signProcess.getActivityId())
                            .bpmActivityName(signProcess.getActivityName())
                            .state(signProcess.getState())
                            .isReassign(false)
                            .isReexecute(false)
                            .signType(SignCategory.parse(signProcess.getSignType()).getValue())
                            .build();

                    // 附件配置信息
                    /**
                     * 附件配置
                     */
                    step.setAttachmentConfig(generateAttachmentConfig(approval.getProjectDefCode(),approval.getTaskDefCode()));

                    if (!Objects.equals(-1, signProcess.getState()) && CollectionUtils.isNotEmpty(signProcess.getWorkitemList())) {
                        List<ApprovalWorkItemDTO> workItemList = new ArrayList<>();
                        signProcess.getWorkitemList().forEach(wi -> {

                            ApprovalWorkItemDTO workItem = ApprovalWorkItemDTO.builder()
                                    .id(0L)
                                    .createTime(wi.getCreateTime())
                                    .closedTime(wi.getClosedTime())
                                    .performerId(wi.getPerformerId())
                                    .performerName(wi.getPerformerId())
                                    .workitemId(wi.getWorkitemId())
                                    .comment(wi.getComment())
                                    .state(wi.getState())
                                    .subState(wi.getSubState())
                                    .agentPerformerId(wi.getAgentPerformerId())
                                    .agentPerformerName(wi.getAgentPerformerId())
                                    .createType(wi.getCreateType())
                                    .performerType(wi.getPerformerType())
                                    .reassign(new JSONArray())
                                    .attachments(wi.getAttachments())
                                    .build();
                            if (!Objects.equals(wi.getCreateType(), 0) && wi.getSubState() > 10) {
                                //PTM优先使用createType,否则createType和subState会显示两个状态
                                workItem.setSubState(ActivityWorkitemSubState.COMPLETED.getValue());
                            }

                            if (nameMap.containsKey(wi.getPerformerId())) {
                                workItem.setPerformerName(nameMap.get(wi.getPerformerId()));
                            } else {
                                UserDTO userDTO = userService.query(wi.getPerformerId());
                                if (userDTO != null) {
                                    workItem.setPerformerName(userDTO.getName());
                                }
                                nameMap.put(workItem.getPerformerId(), workItem.getPerformerName());
                            }
                            if (StringUtils.isNotBlank(wi.getAgentPerformerId())) {
                                if (nameMap.containsKey(wi.getAgentPerformerId())) {
                                    workItem.setAgentPerformerName(nameMap.get(wi.getAgentPerformerId()));
                                } else {
                                    UserDTO userDTO = userService.query(wi.getAgentPerformerId());
                                    if (userDTO != null) {
                                        workItem.setAgentPerformerName(userDTO.getName());
                                    }
                                    nameMap.put(workItem.getAgentPerformerId(), workItem.getAgentPerformerName());
                                }
                            }
                            workItemList.add(workItem);

                            //退回
//                            step.setIsReexecute(wi.getCreateType()==2 || wi.getCreateType()==3);

                        });
                        step.setWorkitemList(workItemList);
                    }
                    result.add(step);

                    //endregion

                    //签核后知会
                    noticeStep = builderSignInformer(signProcess, informer, "noticeAfter");
                    if (noticeStep != null) {
                        noticeStep.setSignType(81);
                        result.add(noticeStep);
                    }
                }
            }
        });
    }


    /**
     * 查询签核历程相关的签核任务数据
     * @param paramMap
     * @return
     */

    public List<Map> queryApprovalTask(Map<String,Object> paramMap){
        List<Map> result = new ArrayList<>();

        if(MapUtils.isNotEmpty(paramMap)){
            List<PtmProjectReportItemDTO> list = ptmService.queryProjectReport(paramMap);
            if(CollectionUtils.isNotEmpty(list)){
                list.forEach(x->{
                    Map m = new HashMap();
                    m.put("bpmActivityId", x.getTaskId());
                    m.put("bpmActivityName", x.getTaskName());
                    m.put("workitemId", x.getId());
                    m.put("processSerialNumber", x.getProcessSerialNumber());
                    m.put("tmActivityId", x.getTaskDefCode());
                    result.add(m);
                });
            }
        }

        return result;
    }

    /**
     * 获取待办的关闭状态
     * @param workItemIdList，有可能是workitem.id，也可能是backlog.id
     * @return
     */

    public List<BacklogStateBO> isBacklogClosed(List<Long> workItemIdList){
        List<BacklogStateBO> result = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(workItemIdList)) {

            List<Long> backlogIdList = new ArrayList<>();
            List<PtmWorkItemRecordBO> workItemRecordList = ptmService.getWorkItemRecordByWorkItemIds(workItemIdList);
            if (CollectionUtils.isNotEmpty(workItemRecordList)) {
                //加入找到的workitem的backlogId
                backlogIdList.addAll(workItemRecordList.stream().map(x -> x.getBacklogId()).distinct().collect(Collectors.toList()));
                for (Long workItemId : workItemIdList) {
                    if(workItemRecordList.stream().filter(x->Objects.equals(x.getId(), workItemId)).count()==0){
                        //加入原来就是backlogid的
                        backlogIdList.addAll(workItemIdList);
                    }
                }
            } else {
                backlogIdList.addAll(workItemIdList);
            }

            QueryWrapper<PtmBacklog> queryWrapper = new QueryWrapper<>();
            queryWrapper.in("backlog_id", backlogIdList)
                .select("backlog_id","closed");
            List<PtmBacklog> backlogList = ptmBacklogMapper.selectList(queryWrapper);
            result = backlogList.stream().map(bl->{
                BacklogStateBO backlogState = new BacklogStateBO();
                backlogState.setId(bl.getBacklogId());
                backlogState.setClosed(bl.getClosed());
                return backlogState;
            }).collect(Collectors.toList());
        }
        return result;
    }

    /**
     * 获取任务卡的状态兼容backlogId是任务卡id、待办id（根据待办id查询任务卡id）的两种情况
     *
     * @param backlogId 任务卡id/待办id
     * @return 返回任务卡的状态信息，包含id、closed两个字段
     */
    public boolean isBacklogClosed(Long backlogId) {
        if (Objects.isNull(backlogId)) {
            return false;
        }

        // 根据backlogId调用ptm接口查询任务卡信息
        PtmBacklogRecordDTO backlogRecord = ptmService.getBacklogRecord(backlogId);
        if (null != backlogRecord && BooleanUtils.isTrue(backlogRecord.getClosed())) {
            return true;
        }

        /**
         * 根据backlogId没查询出任务卡信息或者状态都是未关闭，则根据backlogId本地查询ptm_backlog表再次确认下，
         * 因为ptm  dispatch接口内部是异步执行的，而更新本地ptm_backlog是同步的，因此有可能出现本地patm_backlog表中已关闭，但是ptm返回还未关闭的情况，
         * 这种以本地backlog状态优先。
         */
        QueryWrapper<PtmBacklog> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("backlog_id", backlogId)
                .select("backlog_id", "closed");
        PtmBacklog backlog = ptmBacklogMapper.selectOne(queryWrapper);
        // 如果本地ptm_backlog表中能查询到记录，则以本地ptm_backlog表为准
        if (null != backlog) {
            return BooleanUtils.isTrue(backlog.getClosed());
        }
        // 本地ptm_backlog表中未查询到记录，但从ptm根据backlogId查询到记录，则以ptm返回的backlogRecord为准
        if (null != backlogRecord) {
            return BooleanUtils.isTrue(backlogRecord.getClosed());
        }

        // 兼容workItemId的情况
        List<BacklogStateBO> backlogStateList = this.isBacklogClosed(Lists.newArrayList(backlogId));
        if (CollectionUtils.isEmpty(backlogStateList)) {
            return false;
        }
        return backlogStateList.stream()
                .filter(backlogState -> BooleanUtils.isTrue(backlogState.getClosed()))
                .findFirst()
                .isPresent();
    }

    private ApprovalStepDTO builderSignInformer(List<PtmActivitySignInformerDTO> noticeList, BpmTaskApproveActivityDTO approval, String noticeType){
        if(CollectionUtils.isEmpty(noticeList)){
            return null;
        }

        Optional<PtmActivitySignInformerDTO> noticeOptional = noticeList.stream().filter(x -> Objects.equals(x.getBpmActivityUid(), approval.getBpmActivityOID())).findFirst();
        if(noticeOptional.isPresent()){
            PtmActivitySignInformerDTO notice = noticeOptional.get();
            JSONObject informer = JSONObject.fromObject(StringUtils.isBlank(notice.getInformer())?new JSONObject():notice.getInformer());
            if(informer!=null && informer.containsKey(noticeType)){
                JSONArray noticeArray = JSONArray.fromObject(informer.get(noticeType));
                if(noticeArray==null || noticeArray.isEmpty()){
                    return null;
                }
                ApprovalStepDTO step = ApprovalStepDTO.builder()
                        .id(0L)
                        .bpmActivityId(approval.getActivityId())
                        .bpmActivityName(approval.getActivityName())
                        .state(TaskConstant.ACTIVITY_COMPLETED)
                        .isReassign(false)
                        .isReexecute(false)
                        .build();
                List<ApprovalWorkItemDTO> workItemList = new ArrayList<>();
                for (Object sn : noticeArray) {
                    JSONObject noticeObject= JSONObject.fromObject(sn);
                    if (!noticeObject.containsKey("userId")  || StringUtils.isBlank(noticeObject.getString("userId"))) {
                        continue;
                    }

                    ApprovalWorkItemDTO workItem = ApprovalWorkItemDTO.builder()
                            .performerId(noticeObject.getString("userId"))
                            .build();
                    if (noticeObject.containsKey("userName")  && StringUtils.isNotBlank(noticeObject.getString("userName"))) {
                        workItem.setPerformerName(noticeObject.getString("userName"));
                    }
                    if (noticeObject.containsKey("sendTime")  && StringUtils.isNotBlank(noticeObject.getString("sendTime"))) {
                        workItem.setClosedTime(null);
                    }
                    workItemList.add(workItem);
                }
                step.setWorkitemList(workItemList);
                return step;
            }
        }


        return null;
    }

    private ApprovalStepDTO builderSignInformer(PtmBacklogApprovalDTO.SignProcess signProcess, JSONObject informer, String noticeType){
        if(signProcess==null || informer==null){
            return null;
        }

        if(informer.containsKey(noticeType)){
            JSONArray noticeArray = JSONArray.fromObject(informer.get(noticeType));
            if(noticeArray==null || noticeArray.isEmpty()){
                return null;
            }
            ApprovalStepDTO step = ApprovalStepDTO.builder()
                    .id(0L)
                    .bpmActivityId(signProcess.getActivityId())
                    .bpmActivityName(signProcess.getActivityName())
                    .state(TaskConstant.ACTIVITY_COMPLETED)
                    .isReassign(false)
                    .isReexecute(false)
                    .build();
            List<ApprovalWorkItemDTO> workItemList = new ArrayList<>();
            for (Object sn : noticeArray) {
                JSONObject noticeObject= JSONObject.fromObject(sn);
                if (!noticeObject.containsKey("userId")  || StringUtils.isBlank(noticeObject.getString("userId"))) {
                    continue;
                }

                ApprovalWorkItemDTO workItem = ApprovalWorkItemDTO.builder()
                        .performerId(noticeObject.getString("userId"))
                        .build();
                if (noticeObject.containsKey("userName")  && StringUtils.isNotBlank(noticeObject.getString("userName"))) {
                    workItem.setPerformerName(noticeObject.getString("userName"));
                }
                if (noticeObject.containsKey("sendTime")  && StringUtils.isNotBlank(noticeObject.getString("sendTime"))) {
                    workItem.setClosedTime(null);
                }
                workItemList.add(workItem);
            }
            step.setWorkitemList(workItemList);
            return step;
        }


        return null;
    }

    /**
     * 数据终止，查询符合条件的待办列表
     *
     * @param tenantId
     * @param mainProjectId
     * @param workItemIdList
     * @return 如果不存在，返回空列表
     */

    public List<BacklogBO> queryWorkItemListByIdAndProcessSerialNumber(String tenantId, Long mainProjectId, List<Long> workItemIdList) {
        List<BacklogBO> result = new ArrayList<>();
        if (CollectionUtils.isEmpty(workItemIdList)) {
            return result;
        }

        List<PtmWorkItemRecordBO> workItemRecordList = ptmService.getWorkItemRecordByWorkItemIds(workItemIdList);
        if (CollectionUtils.isEmpty(workItemRecordList)) {
            return result;
        }

        List<PtmProjectRecordBO> projectList = ptmService.getProjectRecordByProjectIdsV2(workItemRecordList.stream().map(x -> x.getProjectId()).collect(Collectors.toList()), false);
        List<PtmTaskRecordBO> taskRecordList = ptmService.getTaskRecordByTaskIds(workItemRecordList.stream().map(x -> x.getTaskId()).collect(Collectors.toList()));
        workItemRecordList.forEach(wi -> {
            Optional<PtmProjectRecordBO> project = projectList.stream().filter(x -> Objects.equals(x.getId(), wi.getProjectId()) && Objects.equals(x.getProjectCardId(), mainProjectId)).findFirst();
            if (!project.isPresent()) {
                return;
            }
            Optional<PtmTaskRecordBO> task = taskRecordList.stream().filter(x -> Objects.equals(x.getId(), wi.getTaskId())).findFirst();
            if (!task.isPresent()) {
                return;
            }

            BacklogBO backlogBO = BacklogBO.builder()
                    .id(wi.getBacklogId())
                    .workItemId(wi.getId())
                    .tmTaskId(project.get().getProjectDefCode())
                    .tmActivityId(task.get().getTaskDefCode())
                    .tmActivityName((task.get().getTaskDefName()))
                    .build();

            result.add(backlogBO);
        });
        return result;
    }

    /**
     * 获取回复型待办的 原始待办
     * @param backlogId
     * @return
     */

    public PtmBacklog getReplyOriginalBacklog(Long backlogId){
        PtmBacklog originaalBacklog = null;

        PtmBacklog backlog = ptmBacklogMapper.selectById(backlogId);
        if(backlog!=null){
            List<PtmBacklog> list = ptmBacklogMapper.selectByWorkItemId(backlog.getWorkItemId());
            if(CollectionUtils.isNotEmpty(list)){
                for (PtmBacklog bl : list) {
                    if(Objects.equals(bl.getType(), 0) && !Objects.equals(bl.getBacklogId(), backlogId)){
                        originaalBacklog = bl;
                        break;
                    }
                }
            }
        }
        return originaalBacklog;
    }

    /**
     * 根据backlogId从PTM获取Backlog的performer、type信息
     *
     * @param backlogId
     * @return
     */
    public PtmBacklog getPtmBacklogPerformerAndType(Long backlogId) {
        PtmBacklog ptmBacklog = ptmBacklogMapper.selectById(backlogId);
        if (null != ptmBacklog) {
            return PtmBacklog.builder()
                    .backlogId(ptmBacklog.getBacklogId())
                    .performerId(ptmBacklog.getPerformerId())
                    .performerName(ptmBacklog.getPerformerName())
                    .type(ptmBacklog.getType())
                    .build();
        }
        return commonPtmTransformService.getPtmBacklogPerformerAndType(backlogId);
    }

    /**
     * 增加 多元交互任务
     * @param user
     * @param backlogId
     * @param email
     */

    public void addEmailBacklog(AuthoredUser user, Long backlogId, String email){
        PtmBacklog backlog = commonPtmTransformService.getBacklogForCard(backlogId);

        String tenantId = null;
        String userId = null;
        String userName = null;
        Map<String, String> tenantAuthUserMap = bpmActivityService.getTargetTenantAndUser(user, email,StringUtils.EMPTY);
        if (tenantAuthUserMap != null && tenantAuthUserMap.size() > 0) {
            tenantId = tenantAuthUserMap.get("workItemToTenantId");
            userId = tenantAuthUserMap.get("commonTenantUserId");
            userName = userId;
            UserDTO userDTO = userService.query(userId);
            if(userDTO!=null){
                userName = userDTO.getName();
            }
        }else{
            throw ErrorCodeEnum.CREATE_PERSON_TENANT_FAIL.getBusinessException();
        }

        PtmEmailBacklogRequestDTO emailRequest = PtmEmailBacklogRequestDTO.builder()
                .workItemId(backlog.getWorkItemId())
                .performerId(userId)
                .performerName(userName)
                .tenantId(tenantId)
                .build();
        ptmService.addCrossTenantBacklog(emailRequest);

    }

    /**
     * 根据待办获取traceId
     * @param backlogId
     * @return
     */

    public String getTraceIdByBacklogId(Long backlogId){
        Long ptmProjectId = null;
        PtmBacklog ptmBacklog = ptmBacklogMapper.selectById(backlogId);
        if(ptmBacklog==null){
            //可能是 workitemId
            PtmWorkItemRecordBO ptmWorkItemRecord = ptmService.getWorkItemRecord(backlogId);
            if(ptmWorkItemRecord!=null){
                ptmProjectId = ptmWorkItemRecord.getProjectId();
            }
        }else{
            ptmProjectId = ptmBacklog.getProjectId();
        }

        if(ptmProjectId!=null){
            PtmProjectRecordBO ptmProjectRecord = ptmService.getProjectRecord(ptmProjectId);
            if(ptmProjectRecord!=null){
                return ptmProjectRecord.getTraceId();
            }
        }
        return null;
    }

    /**
     * 获取待办所属项目卡的当责者信息<br>
     * 仅支持任务引擎
     * @param user
     * @param backlogId
     * @return
     */

    public List<Map> getPersonInChargesInBacklog(AuthoredUser user, Long backlogId){
        List<Map> result = new ArrayList<>();

        List<PtmBacklogItemBO> backlogItemList = ptmService.getBacklogByBacklogId(backlogId);
        if(CollectionUtils.isNotEmpty(backlogItemList)){
            List<PtmProjectCardRecordDTO> projectList = ptmService.getProjectCardRecordByProjectCardIds(backlogItemList.stream().map(x->x.getProjectCardId()).distinct().collect(Collectors.toList()));
            if(CollectionUtils.isNotEmpty(projectList)){
                List<ImNeteaseUserDTO> imUserList = imService.queryNeteaseUsers(projectList.stream().map(x->x.getPersonInCharge()).distinct().collect(Collectors.toList()));
                projectList.forEach(p->{
                    Map m = new HashMap();
                    m.put("projectCardId", p.getId());
                    m.put("projectCardName", p.getProjectName());
                    m.put("personInCharge", p.getPersonInCharge());
                    m.put("personInChargeName", p.getPersonInChargeName());

                    Optional<ImNeteaseUserDTO> imUserOptional = imUserList.stream().filter(x->Objects.equals(x.getUserId(), p.getPersonInCharge())).findFirst();
                    if(imUserOptional.isPresent()){
                        m.put("neteaseImAccId", imUserOptional.get().getAccid());
                    }

                    result.add(m);
                });
            }
        }

        return result;
    }

    /**
     * 是否启用了PTM
     * @return
     */
    private boolean enablePTM(){
        return StringUtils.isNotBlank(envProperties.getPtmUri());
    }


    /**
     * 生成type=0 ；type=1的backlogid的映射关系
     */
    public Map<Long,Long> backlogIdMap(List<Long> ids){
        Map<Long, Long> map = new HashMap<>();

        //分段查询1000个一组
        List<PtmBacklog> ptmBacklogs = new ArrayList<>();
        int batchSize = 1000;
        int totalElements = ids.size();

        for (int i = 0; i < totalElements; i += batchSize) {
            int fromIndex = i;
            int toIndex = Math.min(i + batchSize, totalElements);
            List<Long> batch = ids.subList(fromIndex, toIndex);
            List<PtmBacklog> subList = ptmBacklogMapper.selectBacklogByTypeAndBacklogId(batch);
            ptmBacklogs.addAll(subList);
        }

        Map<Long, List<PtmBacklog>> mapAssist = ptmBacklogs.stream().collect(Collectors.groupingBy(PtmBacklog::getWorkItemId));
        mapAssist.forEach((key,value)->{
            Long type0 = null;
            Long type1 = null;
            Optional<PtmBacklog> first = value.stream().filter(e -> e.getType() == 0).findFirst();
            if (first.isPresent()) {
                type0 = first.get().getBacklogId();
            }
            Optional<PtmBacklog> second = value.stream().filter(e -> e.getType() == 1).findFirst();
            if (second.isPresent()) {
                type1 = second.get().getBacklogId();
            }
            if (type1 != null && type0 != null) {
                map.put(type1, type0);
            }
          });

        return map;
    }

}
