package com.digiwin.athena.atmc.common.util.engine;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.BpmActivityWorkitem;
import com.digiwin.athena.atmc.common.domain.BpmProcess;
import com.digiwin.athena.atmc.common.enums.FlowEngineTableEnum;
import com.digiwin.athena.atmc.http.domain.Task;
import org.apache.commons.collections.CollectionUtils;

import java.util.List;
import java.util.Set;

/**
 * 解析引擎类型
 */
public class AnalysisEngineTypeUtils {
    /**
     * 判断是否为PTM任务
     *
     * @param id
     * @param tableName
     * @return
     */
    public static Boolean isPtm(Long id, FlowEngineTableEnum tableName) {
        // 不是流程引擎即为PTM
        return !isFlowEngine(id, tableName);
    }

    /**
     * 判断是否为PTM任务（id需兼容多个表）
     *
     * @param id
     * @param tableNames
     * @return
     */
    public static Boolean isPtm(Long id, List<FlowEngineTableEnum> tableNames) {
        return tableNames.stream().noneMatch(t -> isFlowEngine(id, t));
    }

    /**
     * 判断是否为PTM任务（根据流程序号判断）
     * @param processSerialNumbers
     * @param tableName
     * @return
     */
    public static Boolean isPtm(List<String> processSerialNumbers, FlowEngineTableEnum tableName) {
        return !checkBpmProcessByProcessSerialNumber(tableName, processSerialNumbers);
    }

    /**
     * 判断是否为PTM任务（多id，并且集合内只有一种引擎的id）
     *
     * @param ids
     * @param tableName
     * @return
     */
    public static Boolean isPtmActivity(List<Long> ids, FlowEngineTableEnum tableName) {
        //如果id为空则默认为走流程引擎的代码逻辑
        if (CollectionUtil.isEmpty(ids)) {
            return false;
        }
        return isPtm(ids.get(0), tableName);
    }


    /**
     * 判断是否为PTM任务(常量)
     */
    public static Boolean isPtmActivity(String param, String value) {
        return param.equals(value);
    }


    /**
     * 判断是否为流程引擎任务
     *
     * @param id
     * @param tableName
     * @return
     */
    private static Boolean isFlowEngine(Long id, FlowEngineTableEnum tableName) {
        // id为空直接return false,走Bpm***Service(全的代码)
        if (null == id) {
            return true;
        }
        // 从redis中取缓存，用于判断是否为流程引擎
        if (FlowEngineDataRedisUtils.getFlowEngineIdFromRedis(id)) {
            return true;
        }
        // 是否是流程引擎
        Boolean isFlowEngine = checkBpmActivityWorkItem(tableName, id) || checkBacklog(tableName, id) || checkBpmActivityStep(tableName, id)
                || checkBpmActivity(tableName, id) || checkActivity(tableName, id)
                || checkTask(tableName, id) || checkBpmProcess(tableName, id);
        // 若是流程引擎的任务则将id存入redis中
        if (isFlowEngine) {
            FlowEngineDataRedisUtils.insertFlowEngineIdToRedis(tableName.getValue(), id);
        }
        return isFlowEngine;
    }


    /**
     * 判断是否是BpmActivityWorkitem
     * 查询数据只返回id (待修改)
     *
     * @param tableName
     * @return
     */
    private static Boolean checkBpmActivityWorkItem(FlowEngineTableEnum tableName, Long id) {
        if (tableName == null) {
            return FlowEngineBeanUtils.getBpmActivityWorkitemMapper().queryById(id) != null;
        }
        return FlowEngineTableEnum.bpmActivityWorkitem.equals(tableName) && FlowEngineBeanUtils.getBpmActivityWorkitemMapper().queryById(id) != null;
    }

    /**
     * 判断是否是backlog
     *
     * @param tableName
     * @param id
     * @return
     */
    private static Boolean checkBacklog(FlowEngineTableEnum tableName, Long id) {
        return FlowEngineTableEnum.backlog.equals(tableName) && FlowEngineBeanUtils.getBacklogMapper().queryById(id) != null;
    }

    /**
     * 判断是否是bpmActivityStep
     *
     * @param tableName
     * @param id
     * @return
     */
    private static Boolean checkBpmActivityStep(FlowEngineTableEnum tableName, Long id) {
        return FlowEngineTableEnum.bpmActivityStep.equals(tableName) && FlowEngineBeanUtils.getBpmActivityStepMapper().queryById(id) != null;
    }

    /**
     * 判断是否是bpmActivity
     *
     * @param tableName
     * @param id
     * @return
     */
    private static Boolean checkBpmActivity(FlowEngineTableEnum tableName, Long id) {
        return FlowEngineTableEnum.bpmActivity.equals(tableName) && FlowEngineBeanUtils.getBpmActivityMapper().queryById(id) != null;
    }

    /**
     * 判断是否是activity
     *
     * @param tableName
     * @param id
     * @return
     */
    private static Boolean checkActivity(FlowEngineTableEnum tableName, Long id) {
        return FlowEngineTableEnum.activity.equals(tableName) && FlowEngineBeanUtils.getActivityMapper().queryById(id) != null;
    }

    /**
     * 判断是否为task
     *
     * @param tableName
     * @param id
     * @return
     */
    private static Boolean checkTask(FlowEngineTableEnum tableName, Long id) {
        return FlowEngineTableEnum.task.equals(tableName) && FlowEngineBeanUtils.getTaskMapper().queryById(id) != null;
    }

    /**
     * 判断是否为bpm_process
     *
     * @param tableName
     * @param id
     * @return
     */
    private static Boolean checkBpmProcess(FlowEngineTableEnum tableName, Long id) {
        return FlowEngineTableEnum.bpmProcess.equals(tableName) && FlowEngineBeanUtils.getBpmProcessMapper().queryById(id) != null;
    }

    /**
     * 根据流程序号判断是否为task
     * @return
     */
    private static Boolean checkBpmProcessByProcessSerialNumber(FlowEngineTableEnum tableEnum, List<String> processSerialNumbers) {
        // 流程序号为空默认为流程引擎
        if (CollectionUtils.isEmpty(processSerialNumbers)) {
            return true;
        }
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.in("process_serial_number", processSerialNumbers);
        List<BpmProcess> list = FlowEngineBeanUtils.getBpmProcessMapper().selectList(queryWrapper);
        return CollectionUtils.isNotEmpty(list);
    }

    /**
     * 根据activity_step_id查询
     *
     * @param stepIdSet
     * @return
     */
    public static List<BpmActivityWorkitem> queryActivityByStepId(Set<String> stepIdSet) {
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.in("activity_step_id", stepIdSet);
        queryWrapper.ne("type", 89);
        queryWrapper.ne("type", 88);
        return FlowEngineBeanUtils.getBpmActivityWorkitemMapper().selectList(queryWrapper);
    }

    /**
     * 根据main_task_id查询
     *
     * @param mainTaskId
     * @return
     */
    public static List<Task> queryTaskByMainTaskId(Long mainTaskId) {
        LambdaQueryWrapper<Task> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(Task::getMainTaskId, mainTaskId)
                .select(Task::getId);
        return FlowEngineBeanUtils.getTaskMapper().selectList(wrapper);
    }
}
