package com.digiwin.athena.atmc.common.bk.parser.instance;


import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.common.bk.parser.core.Constant.*;
import com.digiwin.athena.atmc.common.bk.parser.dto.BkNode;
import com.digiwin.athena.atmc.common.bk.parser.dto.BkNodeDto;
import com.digiwin.athena.atmc.common.bk.parser.dto.metadata.MetadataUnit;
import com.digiwin.athena.atmc.http.constant.ErrorCodeEnum;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import net.sf.json.JSONObject;
import org.apache.commons.collections.map.MultiValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author zhangzhi@digiwin.com
 * @date 2021/11/23
 */
public abstract class AbstractBkInstanceParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractBkInstanceParser.class);

    /**
     * BK实例数据节点解析
     *
     * @param tempMetadata
     * @param originalNode
     * @return
     */
    protected List<JSONObject> parse(MetadataUnit tempMetadata, JsonNode originalNode, String rootName) throws Exception {
        /** 如果是根节点下带BK数据, dataNode节点必须是processVar 或者 parameter 对应的对象节点 */
//        BkNodeDto bkNode = getBkNode(tempMetadata, originalNode, rootName);

//        JsonNode dataNode = bkNode.getDataNode();
//        String dataNodePath = bkNode.getDataNodePath();

        JsonNode dataNode = originalNode;
        String dataNodePath = rootName;

        if (dataNode.isObject()) {
            return parseWhenBkNodeIsObject(tempMetadata, dataNode, dataNodePath);
        }

        if (dataNode.isArray()) {
            return parseWhenBkNodeIsArray(tempMetadata, dataNode, dataNodePath);
        }

        LOGGER.error("数据节点类型为{}，非Object/Array类型!", dataNode.getNodeType());
        throw ErrorCodeEnum.TASK_BK_DATANODE_GET_NODE_TYPE.getBusinessExceptionWithArgs(dataNode.getNodeType());
    }

    /**
     * 根据条件获取符合条件的BkNode进行遍历
     *
     * @param tempMetadata
     * @param originalNode
     * @param rootName
     * @return
     */
    private BkNodeDto getBkNode(MetadataUnit tempMetadata, JsonNode originalNode, String rootName){
        boolean isSinglePath = tempMetadata.getShortestPath().indexOf(".") == -1;
        String dataNodePath = tempMetadata.getShortestPath();
        if(!isSinglePath){
            dataNodePath = tempMetadata.getShortestPath().substring(0, tempMetadata.getShortestPath().lastIndexOf("."));
        }
        /** 如果是根节点下带BK数据, dataNode节点必须是processVar 或者 parameter 对应的对象节点 */
        if (tempMetadata.isRoot()) {
            return new BkNodeDto(originalNode, dataNodePath);
        }
        /** 如果获取到的dataNode节点是Object节点， 需要根据是否是首层节点，决定是否需要对当前path路径进行处理 */
        JsonNode dataNode = originalNode.at("/" + rootName + "/" + dataNodePath.replace(".", "/"));
        if(dataNode.isObject()){
            if(dataNodePath.indexOf(".") == -1){
                return new BkNodeDto(originalNode.at("/" + rootName), dataNodePath);
            } else{
                String tempDataNodePath = dataNodePath.substring(0, dataNodePath.lastIndexOf("."));
                dataNode = originalNode.at("/" + rootName + "/" + tempDataNodePath.replace(".", "/"));
                return new BkNodeDto(dataNode, dataNodePath);
            }
        }

        return new BkNodeDto(dataNode, dataNodePath);
    }

    /**
     * BK实例数据节点为Array时
     *
     * @param tempMetadata
     * @param dataNode
     * @param dataNodePath
     * @return
     */
    private List<JSONObject> parseWhenBkNodeIsArray(MetadataUnit tempMetadata, JsonNode dataNode, String dataNodePath) {
        List<JSONObject> eventData = new ArrayList<>();
        dataNode.elements().forEachRemaining(item -> {
            /** 上线文变量,用于存储Event解析出来的实际BK信息 */
            MultiValueMap eventContext = new MultiValueMap();
            parseBkInstanceData(item, dataNodePath, dataNodePath + ".0", eventContext, tempMetadata, 0);

            if (CollectionUtils.isEmpty(eventContext)) {
                LOGGER.warn("节点数据[{}]中解析的对应Bk数据为空,数据不一致处理结束!", JsonUtils.objectToString(item));
                return;
            }

            /** 获得event的BK原始数据*/
            if (tempMetadata.isExistLongest()) {
                eventData.addAll(processBkWhenExistLongest(tempMetadata, eventContext));
            } else {
                eventData.addAll(processBkWhenNotExistLongest(tempMetadata, eventContext));
            }
        });

        return eventData;
    }

    /**
     * BK实例数据节点为Object时
     *
     * @param tempMetadata
     * @param dataNode
     * @param dataNodePath
     * @return
     */
    private List<JSONObject> parseWhenBkNodeIsObject(MetadataUnit tempMetadata, JsonNode dataNode, String dataNodePath) throws Exception {
        List<JSONObject> eventData = new ArrayList<>();
        MultiValueMap eventContext = new MultiValueMap();
        dataNode.elements().forEachRemaining(item -> {
            /** 上线文变量,用于存储Event解析出来的实际BK信息 */
            parseBkInstanceData(item, dataNodePath, dataNodePath + ".0", eventContext, tempMetadata, 0);
        });

        if(CollectionUtils.isEmpty(eventContext)){
            LOGGER.warn("节点数据[{}]中解析的对应Bk数据为空,数据不一致处理结束!", JsonUtils.objectToString(dataNode));
            throw ErrorCodeEnum.TASK_BK_DATA_EMPTY.getBusinessExceptionWithArgs(JsonUtils.objectToString(dataNode));
        }

        /** 获得event的BK原始数据*/
        if (tempMetadata.isExistLongest()) {
            eventData.addAll(processBkWhenExistLongest(tempMetadata, eventContext));
        } else {
            eventData.addAll(processBkWhenNotExistLongest(tempMetadata, eventContext));
        }
        return eventData;
    }

    /**
     * 解析之后的上下文数据配合当前的元数据定义,针对存在不同长度路径情况下的处理方式
     *
     * @param metadataUnit
     * @param valueMap
     * @return
     */
    private List<JSONObject> processBkWhenExistLongest(MetadataUnit metadataUnit, MultiValueMap valueMap) {
        List<JSONObject> tempResult = new ArrayList<>();

        List<JSONObject> longestObjects = (List<JSONObject>) valueMap.getCollection(metadataUnit.getPathInstance());
        for (JSONObject obj : longestObjects) {
            JSONObject singleObject = new JSONObject();
            Iterator<Map.Entry<String, BkNode>> iterator = metadataUnit.getBkFields().entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, BkNode> entry = iterator.next();
                if (entry.getValue().getNodePath().equals(metadataUnit.getPathInstance())) {
                    singleObject.put(entry.getKey(), obj.getString(Common.VALUE_KEY));
                    continue;
                }
                List<JSONObject> tempObjects = (List<JSONObject>) valueMap.getCollection(entry.getValue().getNodePath());
                String tempValue = getBkValue(tempObjects, obj.getString(Common.INDEX_VALUE_KEY), entry.getValue());
                if (Common.EMPTY_STRING.equals(tempValue)) {
                    LOGGER.warn("通过实例路径[{}]获取目标jsonPath[{}]对应属性值失败, 请查看！", obj.getString(Common.INDEX_VALUE_KEY), entry.getValue());
                    singleObject.clear();
                    continue;
                }
                singleObject.put(entry.getKey(), tempValue);
            }
            if (CollectionUtils.isNotEmpty(singleObject)) {
                tempResult.add(singleObject);
            }
        }

        return tempResult;
    }

    /**
     * 根据路径匹配符合当前节点的父节点或者兄弟节点值
     * NOTE 当节点路径中存在两个相同的节点名称信息时候，此处需要调整，判断对应的终止节点是前一个还是后一个
     *
     * @param bkInfo
     * @param insInxJsonPath
     * @param bkNode
     * @return
     */
    private String getBkValue(List<JSONObject> bkInfo, String insInxJsonPath, BkNode bkNode) {
        String targetJsonPath = bkNode.getNodePath();
        String relativePath = targetJsonPath.substring(0, targetJsonPath.lastIndexOf("."));
        String propName = targetJsonPath.substring(targetJsonPath.lastIndexOf(".") + 1);
        String parentPropName = relativePath.substring(relativePath.lastIndexOf(".") + 1);

        if (insInxJsonPath.indexOf(parentPropName) < 0) {
            return Common.EMPTY_STRING;
        }

        /** 拼接当前jsonPath的带索引全路径值，去对应的List结果中进行匹配 */
        String relativeInxJsonPath;
        int insParentProIndex = insInxJsonPath.indexOf(parentPropName) + parentPropName.length() + 1;
        if (bkNode.isParentIsArray()) {
            //relativeInxJsonPath = insInxJsonPath.substring(0, insInxJsonPath.indexOf(parentPropName) + parentPropName.length() + 3) + propName;
            int insPropIndex = insInxJsonPath.indexOf(".", insParentProIndex);
            relativeInxJsonPath = insInxJsonPath.substring(0, insPropIndex + 1) + propName;
        } else {
            relativeInxJsonPath = insInxJsonPath.substring(0, insParentProIndex) + propName;
        }
        for (JSONObject object : bkInfo) {
            if (object.getString(Common.INDEX_VALUE_KEY).equals(relativeInxJsonPath)) {
                return object.getString(Common.VALUE_KEY);
            }
        }

        return Common.EMPTY_STRING;
    }

    /**
     * 解析之后的上下文数据配合当前的元数据定义，针对都是相同路径情况下的处理方式
     *
     * @param metadataUnit
     * @param valueMap
     * @return
     */
    private List<JSONObject> processBkWhenNotExistLongest(MetadataUnit metadataUnit, MultiValueMap valueMap) {
        int insMaxSize = ((List<JSONObject>) valueMap.get(metadataUnit.getPathInstance())).size();
        List<JSONObject> eventBk = new ArrayList<>();
        for (int i = 0; i < insMaxSize; i++) {
            JSONObject tempObj = new JSONObject();
            Iterator<Map.Entry<String, BkNode>> iterator = metadataUnit.getBkFields().entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<String, BkNode> entry = iterator.next();
                List<JSONObject> tempVal = (List<JSONObject>) valueMap.get(entry.getValue().getNodePath());
                tempObj.put(entry.getKey(), tempVal.get(i).getString("value"));
            }
            eventBk.add(tempObj);
        }
        return eventBk;
    }

    /**
     * 遍历DataNode节点，解析出Event相关的bk信息
     *
     * @param dataNode
     * @param currentJsonPath
     * @param currentJsonIndexPath
     * @param eventContext
     * @param metadataUnit
     * @param index
     */
    private void parseBkInstanceData(JsonNode dataNode, String currentJsonPath, String currentJsonIndexPath, MultiValueMap eventContext, MetadataUnit metadataUnit, int index) {
        if (dataNode.isObject()) {
            ObjectNode object = (ObjectNode) dataNode;
            Iterator<Map.Entry<String, JsonNode>> iterator = object.fields();
            while (iterator.hasNext()) {
                Map.Entry<String, JsonNode> entry = iterator.next();
                String tempCurrentPath = currentJsonPath + "." + entry.getKey();
                String tempCurrentIndexPath = currentJsonIndexPath + "." + entry.getKey();
                parseBkInstanceData(entry.getValue(), tempCurrentPath, tempCurrentIndexPath, eventContext, metadataUnit, index);
            }
        } else if (dataNode.isArray()) {
            ArrayNode tempNode = (ArrayNode) dataNode;
            Iterator<JsonNode> iterator = tempNode.elements();
            int i = 0;
            while (iterator.hasNext()) {
                JsonNode temp = iterator.next();
                parseBkInstanceData(temp, currentJsonPath, currentJsonIndexPath + "." + i, eventContext, metadataUnit, i);
                i++;
            }
        } else {
            currentJsonPath = currentJsonPath.substring(currentJsonIndexPath.indexOf(".") + 1);
            if (metadataUnit.getAllJsonPath().contains(currentJsonPath)) {
                JSONObject temp = new JSONObject();
                /** 路径对应的值，以及对应的带索引的全路径 */
                temp.put(Common.VALUE_KEY, dataNode.asText());
                temp.put(Common.INDEX_VALUE_KEY, currentJsonIndexPath);
                eventContext.put(currentJsonPath, temp);
            }
        }
    }

    protected JsonNode getNode(JSONObject data) throws Exception {
        ObjectMapper mapper = new ObjectMapper();
        /** 此处需要根据isRoot决定, 是否是取当前传过来的数据节点 API是 parameter 还是AI传递过来的processVar，后续遍历的时候需要使用 */
        try {
            return mapper.readTree(JsonUtils.objectToString(data));
        } catch (Exception e) {
            LOGGER.error(ErrorCodeEnum.TASK_BK_DATA_NODE_EXCEPTION.getErrMsg(), e);
            throw ErrorCodeEnum.TASK_BK_DATA_NODE_EXCEPTION.getBusinessExceptionWithArgs();
        }
    }

}
