package com.digiwin.athena.atmc.common.bk.service;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.common.constant.BkConstant;
import com.digiwin.athena.atmc.common.bk.domain.DifferentDataDTO;
import com.digiwin.athena.atmc.common.bk.parser.bo.ParserBo;
import com.digiwin.athena.atmc.common.dao.BpmActivityWorkitemMapper;
import com.digiwin.athena.atmc.infrastructure.mapper.biz.migration.PtmBacklogMapper;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.BpmActivityWorkitem;
import com.digiwin.athena.atmc.infrastructure.pojo.po.migration.PtmBacklog;
import com.digiwin.athena.atmc.common.mongodb.MongodbService;
import com.digiwin.athena.atmc.common.mongodb.data.IndexData;
import com.digiwin.athena.atmc.common.util.BKUtils;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.*;

/**
 * @Author wuzq
 * @Date 2023/6/13 14:45
 * @Description: 数据一致性，生成BK
 * @Version 1.0
 */
@Slf4j
@Service
public class BusinessKeyService {
    @Autowired
    private MongodbService mongodbService;

    @Autowired
    private BpmActivityWorkitemMapper bpmActivityWorkitemMapper;

    @Autowired
    private PtmBacklogMapper ptmBacklogMapper;

    /**
     * 记录businessKey和项目卡/任务卡关联关系
     *
     * @Author：SYQ
     * @Date：2021/12/28 15:47
     */
    public void insertBkTaskActivity(List<ParserBo> bkList, Long cardId, String tenantId, String type) {
        //集合名称
        IndexData indexData = new IndexData();
        indexData.append(BkConstant.CARD_ID,BkConstant.ASC);
        indexData.append(BkConstant.SEARCH_INDEX_PREFIX,BkConstant.ASC);
        mongodbService.createCollectionLock(tenantId, indexData.getIndexList());
        //新增数据
        List<Map> dataList = new ArrayList<>();
        for (ParserBo parserBo : bkList) {
            JSONObject bk = parserBo.getBkInfo();
            String bkStr = JsonUtils.objectToString(bk);
            String bkContent = bkStr.replaceAll("\\{|}|\"", "").replace(",", "&");
            Map<String, Object> dataMap = new HashMap<>();
            TreeMap treeMap = JsonUtils.jsonToObject(JsonUtils.objectToString(bk), TreeMap.class);
            dataMap.put("bk", treeMap);
            dataMap.put("bkContent", bkContent);

            TreeMap bkSortMap = BKUtils.recursionToTreeMap(bk);
            String searchJsonStr = JsonUtils.objectToString(bkSortMap);
            String searchContent = searchJsonStr.replaceAll("\\{|}|\"", "").replace(",", "&");
            dataMap.put(BkConstant.SEARCH_CONTENT,searchContent);
            dataMap.put(BkConstant.CARD_ID, cardId);
            dataMap.put("type", type);
            dataMap.put("tenantId", tenantId);
            dataMap.put(BkConstant.ENTITY_NAME, parserBo.getEntityName());
            dataMap.put("createDate", LocalDateTime.now());
            dataMap.put("modifyDate", LocalDateTime.now());

            //生成bkSearch字段信息，作为bk查询的索引字段
            List<Map<String, Object>> bkSearchList = BKUtils.convertBkIndex(bk);
            dataMap.put(BkConstant.SEARCH_PREFIX,bkSearchList);
            dataMap.put(BkConstant.BK_INDEX_STATE, BkConstant.INDEX_STATE_CREATE);
            dataList.add(dataMap);
        }
        mongodbService.insert(dataList, tenantId);
    }

    /**
     * 更新数据
     *
     * @Author：SYQ
     * @Date：2022/5/13 11:11
     */
    public void updateData(JSONObject bkJson, Set<Long> cardIdSet, String type, String collectionName, DifferentDataDTO differentDataDTO) {
        TreeMap treeMap = JsonUtils.jsonToObject(JsonUtils.objectToString(bkJson), TreeMap.class);
        //查询条件
        Query query = Query.query(Criteria.where("bk").is(treeMap).and(BkConstant.CARD_ID).in(cardIdSet)
                .and("type").is(type).and(BkConstant.ENTITY_NAME).is(differentDataDTO.getEntityName()));
        List<Map> list = mongodbService.query(collectionName, query);
        log.info("【数据一致性数据查询bk】：{}", treeMap);
        log.info("【数据一致性数据查询cardId】：{}", cardIdSet);
        log.info("【数据一致性数据查询type】：{}", differentDataDTO.getType());
        log.info("【数据一致性数据查询结果】{}", list);
        //需要更新的字段及值
        Update update = new Update();
        update.set("optType", differentDataDTO.getDiffType());
        update.set("lastModifyTime", differentDataDTO.getLastModifyTime());
        update.set("differentData", differentDataDTO.getDiffData());
        update.set("modifyDate", LocalDateTime.now());
        update.set("deleteOrUpdate", true);
        //更新数据
        mongodbService.updateMulti(collectionName, query, update);
        log.info("【数据一致性异动数据保存】：{}", differentDataDTO);
    }


    /**
     * 删除索引
     *
     * @Author：SYQ
     * @Date：2022/1/26 13:51
     */
    public void deleteIndexFromCollections(String deleteIndexName) {
        mongodbService.deleteIdexFromCollections(deleteIndexName);
    }

    /**
     * 添加索引
     *
     * @Author：SYQ
     * @Date：2022/1/26 13:51
     */
    public void insertIndexFromCollections(Map<String, Object> indexMap) {
        mongodbService.insertIndex(indexMap);
    }

    /**
     * 根据任务id查询异动数据
     *
     * @Author：SYQ
     * @Date：2022/5/12 16:13
     */
    public List<Map> queryDifferentDataByWorkitemId(String collection, List<Long> workitemIdList) {
        if (CollectionUtils.isEmpty(workitemIdList)) {
            return new ArrayList<>();
        }
        //查询条件，cardId在workitemIdList里，optType不等于空
        List<String> dealTypeList = new ArrayList<>();
        dealTypeList.add("ignore");
        dealTypeList.add("end");
        Query query = Query.query(Criteria.where("type").is("workitem")
                .and("cardId").in(workitemIdList).and("optType").exists(true)
                .and("dealType").nin(dealTypeList).and("completed").exists(false));
        return mongodbService.query(collection, query);
    }

    /**
     * 根据主任务id查询当前未关闭的异动卡
     *
     * @Author：xusijia
     * @Date：2022/7/29 09:47
     */
    public List<BpmActivityWorkitem> queryDataUniformifyTasks(Long backlogId) {
        List<BpmActivityWorkitem> result = new ArrayList<>();
        //查询原任务卡是否已存在未结束的异常排除任务卡
        QueryWrapper queryWrapper = new QueryWrapper();
        queryWrapper.eq("source_workitem_id", backlogId);
//        queryWrapper.lt("state", 3);
        queryWrapper.eq("type", 89);
        List<BpmActivityWorkitem> exceptionWorkitemList = bpmActivityWorkitemMapper.selectList(queryWrapper);
        if (CollectionUtils.isNotEmpty(exceptionWorkitemList)) {
            return exceptionWorkitemList;
        }
        //查询原任务卡是否已存在未结束的异常排除任务卡
        QueryWrapper queryWrapper1 = new QueryWrapper();
        queryWrapper1.eq("source_backlog_id", backlogId);
//            queryWrapper.eq("closed", 0);
        queryWrapper1.eq("type", 89);
        List<PtmBacklog> exceptionBacklogList = ptmBacklogMapper.selectList(queryWrapper1);
        if (CollectionUtils.isNotEmpty(exceptionBacklogList)) {
            exceptionBacklogList.forEach(exceptionBacklog -> {
                BpmActivityWorkitem bpmActivityWorkitem = BpmActivityWorkitem.builder()
                        .id(exceptionBacklog.getBacklogId())
                        .performerId(exceptionBacklog.getPerformerId())
                        .performerName(exceptionBacklog.getPerformerName())
                        .workitemId(exceptionBacklog.getWorkItemId().toString())
                        .sourceWorkitemId(exceptionBacklog.getSourceBacklogId())
                        .closedTime(exceptionBacklog.getClosedTime())
                        .build();
                result.add(bpmActivityWorkitem);
            });
        }
        return result;
    }
}
